/**
File:		Core/InputOutput/NeMatlabIO.h

Author:		
Email:		
Site:       

Copyright (c) 2016 . All rights reserved.
*/

#pragma once

#include <string>
#include <vector>
#include <cstdio>
#include <fstream>
#include <iostream>
#include <zlib.h>
#include <opencv2/core/core.hpp>
#include <Core/NeMatlabIOContainer.h>
#include <Core/NeEFStream.h>
#include <Core/NeTypetraits.h>

namespace NeuralEngine
{
	enum 
	{
		VERSION_5 = 5,
		VERSION_73 = 73
	};

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// <summary>	Matlab Mat file parser for C++ OpenCV. </summary>
	///
	/// <remarks>	
	/// 	This class provides the capacity to read and write Mat files
	/// 	produced by Matlab. The OpenCV cv::Mat class is used for the		
	/// 	internal storage of matrices. The class also provides methods		
	/// 	to inspect the contents of a Mat-file, read all variables, or		
	/// 	read a single variable by name.		
	/// 				
	///		HmetalT, 02/08/2019. 
	///	</remarks>
	////////////////////////////////////////////////////////////////////////////////////////////////////
	class NE_IMPEXP MatlabIO 
	{
	public:
		// constructors

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Default constructor. </summary>
		///
		/// <remarks>	Hmetal T, 03/08/2019. </remarks>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		MatlabIO() {}

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Destructor. </summary>
		///
		/// <remarks>	Hmetal T, 03/08/2019. </remarks>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		virtual ~MatlabIO() { Close(); }

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Gets the filename. </summary>
		///
		/// <remarks>	Hmetal T, 03/08/2019. </remarks>
		///
		/// <returns>	A std::string. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		std::string Filename(void) { return std::string(filename_); }

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Open a filestream for reading or writing. </summary>
		///
		/// <remarks>	Hmetal T, 03/08/2019. </remarks>
		///
		/// <param name="filename">	Filename the full name and filepath of the file. </param>
		/// <param name="mode">	   	Mode either "r" for reading or "w" for writing. </param>
		///
		/// <returns>	true if the file open succeeded, false otherwise. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		bool Open(std::string filename, std::string mode);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Close the filestream and release all resources. </summary>
		///
		/// <remarks>	Hmetal T, 03/08/2019. </remarks>
		///
		/// <returns>	true if the filestream was successfully closed,
		///				false otherwise.Even in the case of failure, the filestream
		///				will no longer point to a valid object. 
		///	</returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		bool Close(void);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Read all variables from a file. </summary>
		///
		/// <remarks>	
		/// 	Reads every variable encountered when parsing a valid Matlab .Mat file.
		/// 	If any of the variables is a function pointer, or other Matlab specific
		/// 	object, it will be passed. Most integral types will be parsed successfully.
		/// 	Matlab matrices will be converted to OpenCV matrices of the same type.
		/// 	Note: Matlab stores images in RGB format whereas OpenCV stores images in
		/// 	BGR format, so if displaying a parsed image using cv::imshow(), the
		/// 	colours will be inverted.
		/// 	
		/// 	Hmetal T, 03/08/2019. 
		/// </remarks>
		///
		/// <returns>	A std::vector&lt;MatlabIOContainer&gt; </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		std::vector<MatlabIOContainer> Read(void);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Print a formatted list of the contents of a file. </summary>
		///
		/// <remarks>	
		/// 	Similar to the 'whos' function in matlab, this function prints to stdout
		/// 	a list of variables and their C++ datatypes stored in the associated .Mat file		
		///
		/// 	Hmetal T, 03/08/2019. 
		/// </remarks>
		///
		/// <param name="variables">	The variables read from the .Mat file using the read() function. </param>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		void Whos(std::vector<MatlabIOContainer> variables) const;

		// templated functions (must be declared and defined in the header file)
		template<class T>
		T Find(std::vector<MatlabIOContainer>& variables, std::string name) const 
		{
			for (unsigned int n = 0; n < variables.size(); ++n) {
				if (variables[n].Name().compare(name) == 0) {
					if (isPrimitiveType<T>()) {
						return variables[n].Data<cv::Mat>().at<T>(0);
					}
					else {
						return variables[n].Data<T>();
					}
				}
			}
			throw new std::exception();
		}

		MatlabIOContainer Find(std::vector<MatlabIOContainer>& variables, std::string name) const 
		{
			for (unsigned int n = 0; n < variables.size(); ++n) {
				if (variables[n].Name().compare(name) == 0) return variables[n];
			}
			throw new std::exception();
		}

		template<class T>
		bool TypeEquals(std::vector<MatlabIOContainer>& variables, std::string name) const 
		{
			for (unsigned int n = 0; n < variables.size(); ++n) {
				if (variables[n].Name().compare(name) == 0) return variables[n].TypeEquals<T>();
			}
			return false;
		}

		template<typename T>
		bool IsPrimitiveType(void) const 
		{
			if (typeid(T) == typeid(uint8_t) || typeid(T) == typeid(int8_t) ||
				typeid(T) == typeid(uint16_t) || typeid(T) == typeid(int16_t) ||
				typeid(T) == typeid(uint32_t) || typeid(T) == typeid(int32_t) ||
				typeid(T) == typeid(float) || typeid(T) == typeid(double) ||
				typeid(T) == typeid(uchar) || typeid(T) == typeid(char) ||
				typeid(T) == typeid(bool)) {
				return true;
			}
			else {
				return false;
			}
		}

	private:

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Gets the .Mat file header information. </summary>
		///
		/// <remarks>	
		/// 	The fields read are:
		/// 	header_ the matlab header as a human readable string,		
		/// 	subsys_ subsystem specific information,		
		/// 	version_ the .Mat file version (5 or 73),		
		/// 	endian_ the bye ordering of the .Mat file. If the byte ordering		
		/// 	needs reversal, this is automatically handled by esfstream.		
		/// 		
		///		Hmetal T, 03/08/2019. 
		///	</remarks>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		void GetHeader(void);
		//void SetHeader(void);
		bool HasVariable(void) { return fid_.peek() != EOF; }

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Constructs a string from an extracted set of fields. </summary>
		///
		/// <remarks>	
		/// 	If the data is of type char, the data is stored as a string rather than a matrix.
		/// 	The dimensionality is ignored (the data is linearized)		
		/// 					
		/// 	Hmetal T, 03/08/2019. </remarks>
		///
		/// <param name="name">	[in,out] The variable name. </param>
		/// <param name="dims">	[in,out] The variable dimensionality (ignored). </param>
		/// <param name="real">	[in,out] The string data. </param>
		///
		/// <returns>	The wrapped string. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		MatlabIOContainer ConstructString(std::vector<char>& name, std::vector<uint32_t>& dims, std::vector<char>& real);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Construct  a sparse matrix. </summary>
		///
		/// <remarks>	Hmetal T, 03/08/2019. </remarks>
		///
		/// <param name="name">	[in,out] The name. </param>
		/// <param name="dims">	[in,out] The dims. </param>
		/// <param name="real">	[in,out] The real. </param>
		/// <param name="imag">	[in,out] The imag. </param>
		///
		/// <returns>	A MatlabIOContainer. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		MatlabIOContainer ConstructSparse(std::vector<char>& name, std::vector<uint32_t>& dims, std::vector<char>& real, std::vector<char>& imag);
		
		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Constructs a cell array. </summary>
		///
		/// <remarks>	
		/// 	If the variable is of type MAT_CELL, construct a cell array. This is done by
		/// 	iteratively calling collateMatrixFields() on each element of the cell, and		
		/// 	storing the result in a vector<MatlabIOContainer>.		
		/// 	Cell fields may not have a name, but are still required to have a name tag. In		
		/// 	this case, placeholder names are substituted. The dimensionality of the cell		
		/// 	array is ignored, and the size is linearized in column major format.		
		/// 			
		/// 	Hmetal T, 03/08/2019. </remarks>
		///
		/// <param name="name">	[in,out] The variable name. </param>
		/// <param name="dims">	[in,out] The dimesionality of the cell array (ignored). </param>
		/// <param name="real">	[in,out] The real part. </param>
		///
		/// <returns>	The wrapped cell array. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		MatlabIOContainer ConstructCell(std::vector<char>& name, std::vector<uint32_t>& dims, std::vector<char>& real);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Constructs a structure. </summary>
		///
		/// <remarks>	Hmetal T, 03/08/2019. </remarks>
		///
		/// <param name="name">	[in,out] The name. </param>
		/// <param name="dims">	[in,out] The dims. </param>
		/// <param name="real">	[in,out] The real. </param>
		///
		/// <returns>	A MatlabIOContainer. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		MatlabIOContainer ConstructStruct(std::vector<char>& name, std::vector<uint32_t>& dims, std::vector<char>& real);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Interpret the variable header information. </summary>
		///
		/// <remarks>	
		/// 	Given a binary data blob, determine the data type and number of bytes
		/// 	that constitute the data blob. This internally handles whether the		
		/// 	header is in long or short format.		
		/// 					
		/// 	Hmetal T, 03/08/2019. </remarks>
		///
		/// <param name="data_type">	[in,out] The returned data type. </param>
		/// <param name="dbytes">   	[in,out] The returned number of bytes that constitute the data blob. </param>
		/// <param name="wbytes">   	[in,out] The whole number of bytes that include the header size,
		///								the size of the data and any padding to 64 - bit boundaries.This is equivalent
		///								to the entire number of bytes effectively used by a variable. </param>
		/// <param name="data">			The input binary blob. </param>
		///
		/// <returns>	A pointer to the beginning of the data segment of the binary blob. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		const char* ReadVariableTag(uint32_t &data_type, uint32_t &dbytes, uint32_t &wbytes, const char *data);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Interpret all fields of a matrix. </summary>
		///
		/// <remarks>	
		/// 	collateMatrixFields takes a binary blob of data and strips out the matrix fields.
		/// 	These fields necessarily include: the variable dimensionality, the variable name		
		/// 	and the real part of the variable data. It optionally includes the imaginary part		
		/// 	of the variable data if that exists too. The extracted fields are used to either		
		/// 	construct a matrix, cell array or struct, or a scalar in the case where the variable		
		/// 	dimensionality is (1,1)		
		/// 			
		/// 	Hmetal T, 03/08/2019. 
		/// </remarks>
		///
		/// <param name="data_type">	The type of the data stored in the binary blob. </param>
		/// <param name="nbytes">   	The number of bytes that constitute the binary blob. </param>
		/// <param name="data">			The binary blob. </param>
		///
		/// <returns>	The variable (matrix, struct, cell, scalar) wrapped in a container. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		MatlabIOContainer CollateMatrixFields(uint32_t data_type, uint32_t nbytes, std::vector<char> data);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Uncompress a variable. </summary>
		///
		/// <remarks>	
		/// 	If the data type of a variable is MAT_COMPRESSED, then the binary data blob
		/// 	has been compressed using zlib compression. This function uncompresses the blob,		
		/// 	then calls readVariable() to interpret the actual data		
		/// 			
		/// 	Hmetal T, 03/08/2019. </remarks>
		///
		/// <param name="data_type">	[in,out] The type of the data stored in the binary blob. </param>
		/// <param name="dbytes">   	[in,out] The number of bytes that constitue the binary blob. </param>
		/// <param name="wbytes">   	[in,out] The whole number of bytes that consistute the header,
		/// 							the binary blob, and any padding to 64 - bit boundaries. </param>
		/// <param name="data">			The binary blob. </param>
		///
		/// <returns>	The binary blob, uncompressed </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		std::vector<char> UncompressVariable(uint32_t& data_type, uint32_t& dbytes, uint32_t& wbytes, const std::vector<char> &data);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Interpret a variable from a binary block of data. </summary>
		///
		/// <remarks>	
		/// 	This function may be called recursively when either uncompressing data or interpreting
		/// 	fields of a struct or cell array		
		/// 			
		/// 	Hmetal T, 03/08/2019. </remarks>
		///
		/// <param name="data_type">	The type of the data stored in the binary blob. </param>
		/// <param name="nbytes">   	The number of bytes that constitute the binary blob. </param>
		/// <param name="data">			The binary blob. </param>
		///
		/// <returns>	An interpreted variable. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		MatlabIOContainer ReadVariable(uint32_t data_type, uint32_t nbytes, const std::vector<char> &data);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Reads a block of data from the file being parsed. </summary>
		///
		/// <remarks>	
		/// 	This function attempts to read an entire variable from the file being parsed.
		/// 	The data block is then encapsulated in a vector and passed onto readVariable()		
		/// 	for interpretation. This design means that the file is touched a minimal number		
		/// 	of times, and later manipulation of the data can make use of automatic memory		
		/// 	management, reference counting, etc.		
		/// 			
		/// 	Hmetal T, 03/08/2019. 
		/// </remarks>
		///
		/// <returns>	The block of data interpreted as a variable and stored in a generic container. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		MatlabIOContainer ReadBlock(void);
		//MatlabIOContainer UncompressFromBin(std::vector<char> data, uint32_t nbytes);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Transpose a multi-channel matrix. </summary>
		///
		/// <remarks>	
		/// 	The OpenCV builtin transpose method cannot tranpose multi-dimensional
		/// 	matrices. This function provides that capability by splitting the matrix		
		/// 	into a vector of its channels, tranposing each channel, then merging		
		/// 	the result back into a single multi-channel matrix		
		/// 				
		/// 	Hmetal T, 03/08/2019. </remarks>
		///
		/// <param name="src">	The input matrix. </param>
		/// <param name="dst">	[in,out] The output matrix, where dst(i,j,k) == src(j,i,k). </param>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		void TransposeMat(const cv::Mat& src, cv::Mat& dst);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Convert the type of a variable. </summary>
		///
		/// <remarks>	
		/// 	Given a vector of type char, interpret the data as an
		/// 	vector of type T1, and convert it to a vector of type		
		/// 	T2.		
		/// 		
		/// 	HmetalT, 03/08/2019. </remarks>
		///
		/// <param name="in">	The input char vector. </param>
		///
		/// <returns>	The same data, reinterpreted as type T2 through storage type T1. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		template<class T1, class T2> std::vector<T2> ConvertPrimitiveType(const std::vector<char>& in);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Product of the elements of a vector. </summary>
		///
		/// <remarks>	
		/// 	The function is useful for calculating the total number
		/// 	of elements in an array given a vector of dims.		
		/// 			
		///		Hmetal T, 03/08/2019. </remarks>
		///
		/// <param name="vec">	The input vector. </param>
		///
		/// <returns>	The product of elements in the input. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		template<typename T> T Product(const std::vector<T>& vec);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Construct a matrix from an extracted set of fields. </summary>
		///
		/// <remarks>	
		/// 	Given the variable size, name, data and data type, construct a matrix.
		/// 	Note that Matlab may store variables in a different data type to the
		/// 	actual variable data type (T) to save space. For example matrix a = [1 2 3 4 5];
		/// 	in Matlab will intrinsically be of type double (everything is unless otherwise
		/// 	explicitly stated) but could be stored as a uint8_t to save space.
		/// 	The type of the variable returned should necessarily be double, since
		/// 	it's impossible to know at compile time which data types Matlab has decided
		/// 	to store a set of variables in.
		/// 	
		/// 	Hmetal T, 03/08/2019. 
		/// </remarks>
		///
		/// <param name="name">			[in,out] The variable name. </param>
		/// <param name="dims">			[in,out] The variable dimensionality (i, j, k, ...). </param>
		/// <param name="real">			[in,out] The real part. </param>
		/// <param name="imag">			[in,out] The imaginary part (imag.size() == 0 if the data is real). </param>
		/// <param name="stor_type">	The storage type of the value. </param>
		///
		/// <returns>	The wrapped matrix. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		template<class T> MatlabIOContainer ConstructMatrix(std::vector<char>& name, std::vector<uint32_t>& dims, std::vector<char>& real, std::vector<char>& imag, uint32_t stor_type);

		// member variables
		static const int HEADER_LENGTH = 116;
		static const int SUBSYS_LENGTH = 8;
		static const int ENDIAN_LENGTH = 2;
		char header_[HEADER_LENGTH + 1];
		char subsys_[SUBSYS_LENGTH + 1];
		char endian_[ENDIAN_LENGTH + 1];
		int16_t version_;
		bool byte_swap_;
		int bytes_read_;
		std::string filename_;
		EFStream fid_;
	};

	template<class T1, class T2>
	inline std::vector<T2> MatlabIO::ConvertPrimitiveType(const std::vector<char>& in)
	{
		std::vector<T2> out;

		// firstly reinterpret the input as type T1
		const unsigned int T1_size = in.size() / sizeof(T1);

		if (T1_size != 0)
		{
			const T1* in_ptr = reinterpret_cast<const T1*>(&(in[0]));

			// construct the new vector
			std::vector<T2> tmp(in_ptr, in_ptr + T1_size);
			out = tmp;
		}

		return out;

	}

	template<typename T>
	inline T MatlabIO::Product(const std::vector<T>& vec)
	{
		T acc = 1;
		for (unsigned int n = 0; n < vec.size(); ++n) acc *= vec[n];
		return acc;
	}
	
	template<class T>
	inline MatlabIOContainer MatlabIO::ConstructMatrix(std::vector<char>& name, std::vector<uint32_t>& dims, std::vector<char>& real, std::vector<char>& imag, uint32_t stor_type)
	{
		std::vector<T> vec_real;
		std::vector<T> vec_imag;
		std::vector<cv::Mat> vec_mat;
		cv::Mat flat;
		cv::Mat mat;
		switch (stor_type) {
		case MAT_INT8:
			vec_real = ConvertPrimitiveType<int8_t, T>(real);
			vec_imag = ConvertPrimitiveType<int8_t, T>(imag);
			break;
		case MAT_UINT8:
			vec_real = ConvertPrimitiveType<uint8_t, T>(real);
			vec_imag = ConvertPrimitiveType<uint8_t, T>(imag);
			break;
		case MAT_INT16:
			vec_real = ConvertPrimitiveType<int16_t, T>(real);
			vec_imag = ConvertPrimitiveType<int16_t, T>(imag);
			break;
		case MAT_UINT16:
			vec_real = ConvertPrimitiveType<uint16_t, T>(real);
			vec_imag = ConvertPrimitiveType<uint16_t, T>(imag);
			break;
		case MAT_INT32:
			vec_real = ConvertPrimitiveType<int32_t, T>(real);
			vec_imag = ConvertPrimitiveType<int32_t, T>(imag);
			break;
		case MAT_UINT32:
			vec_real = ConvertPrimitiveType<uint32_t, T>(real);
			vec_imag = ConvertPrimitiveType<uint32_t, T>(imag);
			break;
		case MAT_INT64:
			vec_real = ConvertPrimitiveType<int64_t, T>(real);
			vec_imag = ConvertPrimitiveType<int64_t, T>(imag);
			break;
		case MAT_UINT64:
			vec_real = ConvertPrimitiveType<uint64_t, T>(real);
			vec_imag = ConvertPrimitiveType<uint64_t, T>(imag);
			break;
		case MAT_FLOAT:
			vec_real = ConvertPrimitiveType<float, T>(real);
			vec_imag = ConvertPrimitiveType<float, T>(imag);
			break;
		case MAT_DOUBLE:
			vec_real = ConvertPrimitiveType<double, T>(real);
			vec_imag = ConvertPrimitiveType<double, T>(imag);
			break;
		case MAT_UTF8:
			vec_real = ConvertPrimitiveType<char, T>(real);
			vec_imag = ConvertPrimitiveType<char, T>(imag);
			break;
		default:
			return MatlabIOContainer();
		}

		// assert that the conversion has not modified the number of elements
		uint32_t numel = 1;
		for (unsigned int n = 0; n < dims.size(); ++n) numel *= dims[n];
		assert(vec_real.size() == numel);

		// if the data is a scalar, don't write it to a matrix
		//if (vec_real.size() == 1 && vec_imag.size() == 0) return MatlabIOContainer(string(&(name[0])), vec_real[0]);

		// get the number of channels
		const unsigned int channels = dims.size() == 3 ? dims[2] : 1;
		bool complx = vec_imag.size() != 0;

		// put each plane of the image into a vector
		std::vector<cv::Mat> sflat;
		flat = cv::Mat(vec_real, true);
		for (unsigned int n = 0; n < channels; ++n)
			sflat.push_back(flat(cv::Range(dims[0] * dims[1] * n, dims[0] * dims[1] * (n + 1)), cv::Range::all()));
		flat = cv::Mat(vec_imag, true);
		for (unsigned int n = 0; n < channels*complx; ++n)
			sflat.push_back(flat(cv::Range(dims[0] * dims[1] * n, dims[0] * dims[1] * (n + 1)), cv::Range::all()));

		// merge the planes into a matrix
		merge(sflat, flat);

		// reshape to the image dimensions
		mat = flat.reshape(flat.channels(), dims[1]);

		// transpose the matrix since matlab stores them in column major ordering
		TransposeMat(mat, mat);

		return MatlabIOContainer(std::string(&(name[0])), mat);
	}
}