/**
File:		NECore/InputOutput/NeMatlabIOContainer.h

Author:		
Email:		
Site:       

Copyright (c) 2016 . All rights reserved.
*/

#pragma once

#include <NeCoreLib.h>
#include <string>
#include <typeinfo>
#include <boost/any.hpp>
#include <Core/NeTypetraits.h>

namespace NeuralEngine
{
	typedef std::vector<MatlabIOContainer> vectorMatlabIOContainer;
	typedef std::vector<std::vector<MatlabIOContainer> > vector2DMatlabIOContainer;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// <summary>	 A container class for storing type agnostic variables. </summary>
	///
	/// <remarks>	
	/// 	MatlabIOContainer stores variables of any type using the boost::any type.
	/// 	This allows multiple MatlabIOContainers to be stored in a single vector		
	/// 	when reading multiple variables from a file or constructing a Matlab struct.						
	/// 			
	/// 	HmetalT, 02/08/2019. 
	/// </remarks>
	////////////////////////////////////////////////////////////////////////////////////////////////////
	class NE_IMPEXP MatlabIOContainer 
	{
	public:
		// constructors

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Default constructor. </summary>
		///
		/// <remarks>	Hmetal T, 02/08/2019. </remarks>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		MatlabIOContainer();

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Constructor to initalize the container with data and an associated name. </summary>
		///
		/// <remarks>	Hmetal T, 02/08/2019. </remarks>
		///
		/// <param name="name">	The string name of the variable. </param>
		/// <param name="data">	The associated data, of any type. </param>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		MatlabIOContainer(const std::string name, const boost::any data);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Destructor. </summary>
		///
		/// <remarks>	Hmetal T, 02/08/2019. </remarks>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		virtual ~MatlabIOContainer();

		// set methods

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Sets a name. </summary>
		///
		/// <remarks>	Hmetal T, 02/08/2019. </remarks>
		///
		/// <param name="name">	The name. </param>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		void SetName(const std::string name);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Sets any kind of data. </summary>
		///
		/// <remarks>	Hmetal T, 02/08/2019. </remarks>
		///
		/// <param name="data">	The data. </param>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		void SetData(const boost::any data);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Check if the stored type is equal to the templated type. </summary>
		///
		/// <remarks>	
		/// 	This method uses typeid(T) internally, so child classes may return true
		/// 	if compared with their parent.		
		/// 			
		/// 	Hmetal T, 02/08/2019. 
		/// </remarks>
		///
		/// <typeparam name="T">	Generic type parameter. </typeparam>
		///
		/// <returns>	true if the types are equal (premised on the above conditions), false otherwise. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		template<class T> bool TypeEquals(void) const
		{
			return data_.type() == typeid(T);
		}

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	The type of the variable. </summary>
		///
		/// <remarks>	
		/// 	Returns a string containing the type of the variable. This may or may not be
		/// 	human readable (Microsoft compilers will produce human readable outputs, GCC		
		/// 	compilers will not) but are guaranteed to be unique for unique types		
		/// 	@return the variable type as a string		
		/// 			
		///		HmetalT, 02/08/2019. 
		///	</remarks>
		///
		/// <returns>	A std::string. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		std::string Type(void) const;

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Gets the name. </summary>
		///
		/// <remarks>	Hmetal T, 02/08/2019. </remarks>
		///
		/// <returns>	A std::string. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		std::string Name(void) const;

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Gets the stored data. </summary>
		///
		/// <remarks>	
		/// 	Returns the stored data, cast to the templated type
		/// 	@throw boost::bad_any_cast if the requested type is not the stored data type
		/// 	
		/// 	Hmetal T, 02/08/2019. </remarks>
		///
		/// <typeparam name="T">	Generic type parameter. </typeparam>
		///
		/// <returns>	The data. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		template<class T> T Data(void) const
		{
			return boost::any_cast<T>(data_);
		}

	private:
		std::string name_;
		boost::any data_;
	};
}