/**
File:		NECore/DataTypes/NeMinHeap.inl

Author:		
Email:		
Site:       

Copyright (c) 2016 . All rights reserved.
*/

template <typename KeyType, typename ValueType>
MinHeap<KeyType, ValueType>::MinHeap(int maxElements)
{
	Reset(maxElements);
}

template <typename KeyType, typename ValueType>
MinHeap<KeyType, ValueType>::MinHeap(MinHeap const& minHeap)
{
	*this = minHeap;
}

template <typename KeyType, typename ValueType>
MinHeap<KeyType, ValueType>& MinHeap<KeyType, ValueType>::operator=(MinHeap const& minHeap)
{
	mNumElements = minHeap.mNumElements;
	mRecords = minHeap.mRecords;
	mPointers.resize(minHeap.mPointers.size());
	for (auto& record : mRecords)
	{
		mPointers[record.index] = &record;
	}
	return *this;
}

template <typename KeyType, typename ValueType>
void MinHeap<KeyType, ValueType>::Reset(int maxElements)
{
	mNumElements = 0;
	if (maxElements > 0)
	{
		mRecords.resize(maxElements);
		mPointers.resize(maxElements);
		for (int i = 0; i < maxElements; ++i)
		{
			mPointers[i] = &mRecords[i];
			mPointers[i]->index = i;
		}
	}
	else
	{
		mRecords.clear();
		mPointers.clear();
	}
}

template <typename KeyType, typename ValueType>
inline int MinHeap<KeyType, ValueType>::GetNumElements() const
{
	return mNumElements;
}

template <typename KeyType, typename ValueType>
inline bool MinHeap<KeyType, ValueType>::GetMinimum(KeyType& key, ValueType& value) const
{
	if (mNumElements > 0)
	{
		key = mPointers[0]->key;
		value = mPointers[0]->value;
		return true;
	}
	else
	{
		return false;
	}
}

template <typename KeyType, typename ValueType>
typename MinHeap<KeyType, ValueType>::Record*
MinHeap<KeyType, ValueType>::Insert(KeyType const& key, ValueType const& value)
{
	// Return immediately when the heap is full.
	if (mNumElements == static_cast<int>(mRecords.size()))
	{
		return nullptr;
	}

	// Store the input information in the last heap record, which is the last
	// leaf in the tree.
	int child = mNumElements++;
	Record * record = mPointers[child];
	record->key = key;
	record->value = value;

	// Propagate the information toward the root of the tree until it reaches
	// its correct position, thus restoring the tree to a valid heap.
	while (child > 0)
	{
		int parent = (child - 1) / 2;
		if (mPointers[parent]->value <= value)
		{
			// The parent has a value smaller than or equal to the child's
			// value, so we now have a valid heap.
			break;
		}

		// The parent has a larger value than the child's value.  Swap the
		// parent and child:

		// Move the parent into the child's slot.
		mPointers[child] = mPointers[parent];
		mPointers[child]->index = child;

		// Move the child into the parent's slot.
		mPointers[parent] = record;
		mPointers[parent]->index = parent;

		child = parent;
	}

	return mPointers[child];
}

template <typename KeyType, typename ValueType>
bool MinHeap<KeyType, ValueType>::Remove(KeyType& key, ValueType& value)
{
	// Return immediately when the heap is empty.
	if (mNumElements == 0)
	{
		return false;
	}

	// Get the information from the root of the heap.
	Record* root = mPointers[0];
	key = root->key;
	value = root->value;

	// Restore the tree to a heap.  Abstractly, record is the new root of
	// the heap.  It is moved down the tree via parent-child swaps until it
	// is in a location that restores the tree to a heap.
	int last = --mNumElements;
	Record* record = mPointers[last];
	int parent = 0, child = 1;
	while (child <= last)
	{
		if (child < last)
		{
			// Select the child with smallest value to be the one that is
			// swapped with the parent, if necessary.
			int childP1 = child + 1;
			if (mPointers[childP1]->value < mPointers[child]->value)
			{
				child = childP1;
			}
		}

		if (record->value <= mPointers[child]->value)
		{
			// The tree is now a heap.
			break;
		}

		// Move the child into the parent's slot.
		mPointers[parent] = mPointers[child];
		mPointers[parent]->index = parent;

		parent = child;
		child = 2 * child + 1;
	}

	// The previous 'last' record was moved to the root and propagated down
	// the tree to its final resting place, restoring the tree to a heap.
	// The slot mPointers[parent] is that resting place.
	mPointers[parent] = record;
	mPointers[parent]->index = parent;

	// The old root record must not be lost.  Attach it to the slot that
	// contained the old last record.
	mPointers[last] = root;
	mPointers[last]->index = last;
	return true;
}

template <typename KeyType, typename ValueType>
void MinHeap<KeyType, ValueType>::Update(Record* record, ValueType const& value)
{
	// Return immediately on invalid record.
	if (!record)
	{
		return;
	}

	int parent, child, childP1, maxChild;

	if (record->value < value)
	{
		record->value = value;

		// The new value is larger than the old value.  Propagate it toward
		// the leaves.
		parent = record->index;
		child = 2 * parent + 1;
		while (child < mNumElements)
		{
			// At least one child exists.  Locate the one of maximum value.
			childP1 = child + 1;
			if (childP1 < mNumElements)
			{
				// Two children exist.
				if (mPointers[child]->value <= mPointers[childP1]->value)
				{
					maxChild = child;
				}
				else
				{
					maxChild = childP1;
				}
			}
			else
			{
				// One child exists.
				maxChild = child;
			}

			if (value <= mPointers[maxChild]->value)
			{
				// The new value is in the correct place to restore the tree
				// to a heap.
				break;
			}

			// The child has a larger value than the parent's value.  Swap
			// the parent and child:

			// Move the child into the parent's slot.
			mPointers[parent] = mPointers[maxChild];
			mPointers[parent]->index = parent;

			// Move the parent into the child's slot.
			mPointers[maxChild] = record;
			mPointers[maxChild]->index = maxChild;

			parent = maxChild;
			child = 2 * parent + 1;
		}
	}
	else if (value < record->value)
	{
		record->value = value;

		// The new weight is smaller than the old weight.  Propagate it
		// toward the root.
		child = record->index;
		while (child > 0)
		{
			// A parent exists.
			parent = (child - 1) / 2;

			if (mPointers[parent]->value <= value)
			{
				// The new value is in the correct place to restore the tree
				// to a heap.
				break;
			}

			// The parent has a smaller value than the child's value.  Swap
			// the child and parent:

			// Move the parent into the child's slot.
			mPointers[child] = mPointers[parent];
			mPointers[child]->index = child;

			// Move the child into the parent's slot.
			mPointers[parent] = record;
			mPointers[parent]->index = parent;

			child = parent;
		}
	}
}

template <typename KeyType, typename ValueType>
bool MinHeap<KeyType, ValueType>::IsValid() const
{
	for (int child = 0; child < mNumElements; ++child)
	{
		int parent = (child - 1) / 2;
		if (parent > 0)
		{
			if (mPointers[child]->value < mPointers[parent]->value)
			{
				return false;
			}

			if (mPointers[parent]->index != parent)
			{
				return false;
			}
		}
	}

	return true;
}