/**
File:		MachineLearning/Optimization/Base/IGradientOptimizationMethod.h

Author:		
Email:		
Site:       

Copyright (c) 2017 . All rights reserved.
*/

#pragma once

#include <NeMachineLearningLib.h>
#include <MachineLearning/IOptimizationMethod.h>
#include <MachineLearning/NonlinearObjectiveFunction.h>
#include <functional>
#include <exception>

namespace NeuralEngine
{
	namespace MachineLearning
	{

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Base class for optimization methods. </summary>
		///
		/// <remarks>	 Admin, 3/21/2017. </remarks>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		template<typename Scalar>
		class NE_IMPEXP BaseOptimizationMethod : public IOptimizationMethod<Scalar>
		{
		public:

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			/// 	Gets the number of variables (free parameters)
			/// 	in the optimization problem.
			/// </summary>
			///
			/// <value>	The number of parameters. </value>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual int GetNumberOfVariables();

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			/// 	Gets the current solution found, the values of the parameters which optimizes the
			/// 	function.
			/// </summary>
			///
			/// <remarks>	Hmetal T, 17.03.2017. </remarks>
			///
			/// <returns>	The solution. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual af::array GetSolution();

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			/// 	Sets the current solution found, the values of the parameters which optimizes the
			/// 	function.
			/// </summary>
			///
			/// <remarks>	Hmetal T, 17.03.2017. </remarks>
			///
			/// <returns>	The solution. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual void SetSolution(af::array& x);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Gets the output of the function at the current <see cref="Solution"/>. </summary>
			///
			/// <remarks>	Hmetal T, 17.03.2017. </remarks>
			///
			/// <returns>	The value. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual Scalar GetValue();

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			/// 	Finds the maximum value of a function. The solution vector will be made available at the
			/// 	<see cref="Solution"/> property.
			/// </summary>
			///
			/// <remarks>	Hmetal T, 17.03.2017. </remarks>
			///
			/// <param name="values">	The initial solution vector to start the search. </param>
			///
			/// <returns>
			/// 	Returns <c>true</c> if the method converged to a <see cref="Solution"/>. In this case,
			/// 	the found value will also be available at the <see cref="Value"/>
			/// 	property.
			/// </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual bool Maximize(af::array& values, int* cycle = nullptr);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			/// 	Finds the minimum value of a function. The solution vector will be made available at the
			/// 	<see cref="Solution"/> property.
			/// </summary>
			///
			/// <remarks>	Hmetal T, 17.03.2017. </remarks>
			///
			/// <param name="values">	The initial solution vector to start the search. </param>
			///
			/// <returns>
			/// 	Returns <c>true</c> if the method converged to a <see cref="Solution"/>. In this case,
			/// 	the found value will also be available at the <see cref="Value"/>
			/// 	property.
			/// </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual bool Minimize(af::array& values, int* cycle = nullptr);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			/// 	Finds the maximum value of a function. The solution vector will be made available at the
			/// 	<see cref="Solution"/> property.
			/// </summary>
			///
			/// <remarks>	Hmetal T, 17.03.2017. </remarks>
			///
			/// <returns>
			/// 	Returns <c>true</c> if the method converged to a <see cref="Solution"/>. In this case,
			/// 	the found value will also be available at the <see cref="Value"/>
			/// 	property.
			/// </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual bool Maximize(int* cycle = nullptr);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			/// 	Finds the minimum value of a function. The solution vector will be made available at the
			/// 	<see cref="Solution"/> property.
			/// </summary>
			///
			/// <remarks>	Hmetal T, 17.03.2017. </remarks>
			///
			/// <returns>
			/// 	Returns <c>true</c> if the method converged to a <see cref="Solution"/>. In this case,
			/// 	the found value will also be available at the <see cref="Value"/>
			/// 	property.
			/// </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual bool Minimize(int* cycle = nullptr);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Set to display optimization information. </summary>
			///
			/// <remarks>	Hmetal T, 03.05.2019. </remarks>
			///
			/// <param name="display">	true to display. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void Display(bool display);

			~BaseOptimizationMethod();

		protected:

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Sets the output of the function at the current <see cref="Solution"/>. </summary>
			///
			/// <remarks>	Hmetal T, 17.03.2017. </remarks>
			///
			/// <returns>	The value. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void SetValue(Scalar v);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			/// 	Sets the number of variables (free parameters)
			/// 	in the optimization problem.
			/// </summary>
			///
			/// <value>	The number of parameters. </value>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void SetNumberOfVariables(int n);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			/// 	Initializes a new instance of the <see cref="BaseOptimizationMethod"/> class.
			/// </summary>
			///
			/// <remarks>	Hmetal T, 17.03.2017. </remarks>
			///
			/// <param name="numberOfVariables">
			/// 	The number of free parameters in the optimization problem.
			/// </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			BaseOptimizationMethod(int numberOfVariables);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			/// 	Initializes a new instance of the <see cref="BaseOptimizationMethod"/> class.
			/// </summary>
			///
			/// <remarks>	Hmetal T, 17.03.2017. </remarks>
			///
			/// <param name="numberOfVariables">
			/// 	The number of free parameters in the optimization problem.
			/// </param>
			/// <param name="function">
			/// 	The objective function whose optimum values should be found.
			/// </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			BaseOptimizationMethod(int numberOfVariables, std::function<Scalar(const af::array&, af::array&)> function);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			/// 	Initializes a new instance of the <see cref="BaseOptimizationMethod"/> class.
			/// </summary>
			///
			/// <remarks>	Hmetal T, 17.03.2017. </remarks>
			///
			/// <param name="function">	The objective function whose optimum values should be found. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			BaseOptimizationMethod(NonlinearObjectiveFunction<Scalar>* function);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			/// 	Implements the actual optimization algorithm. This method should try to minimize the
			/// 	objective function.
			/// </summary>
			///
			/// <remarks>	Hmetal T, 18.03.2017. </remarks>
			///
			/// <returns>	true if it succeeds, false if it fails. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual bool Optimize(int* cycle = nullptr) = 0;

			//std::function<Scalar(af::array&)> _function;

			NonlinearObjectiveFunction<Scalar>* _function;

			af::array _x;

			bool _display;

			af::dtype m_dtype;

		private:

			void init(int numberOfVariables);

			Scalar _value;
			int _numVariables;
		};
	}
}
