/**
File:		MachineLearning/FactorGraph/Node/FgGPNode.h

Author:		
Email:		
Site:       

Copyright (c) 2020 . All rights reserved.
*/

#pragma once

#include <NeMachineLearningLib.h>
#include <MachineLearning/FgArrayFireSerialization.h>
#include <boost/serialization/shared_ptr.hpp>
#include <vector>
#include <memory>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		namespace GPModels
		{
			namespace AEP
			{
				template<typename Scalar>
				class SDGPLVM;

				template<typename Scalar>
				class SGPLVM;
			}
		}

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	This class represents grouping nodes in a hiearchy. </summary>
		///
		/// <remarks>	, 19.07.2018. </remarks>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		template<typename Scalar>
		class NE_IMPEXP GPNode
		{
		public:

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Default constructor. </summary>
			///
			/// <remarks>	Hmetal T, 03/04/2020. </remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			GPNode();

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Destructor. </summary>
			///
			/// <remarks>	Hmetal T, 04/04/2020. </remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual ~GPNode();

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Gets the number of children of this item. </summary>
			///
			/// <remarks>	
			/// 	This is the current number of elements in the child array.  These
			/// 	elements are not all guaranteed to be non-null.  Thus, when you		
			/// 	iterate over the array and access children with GetChild(...), you
			///		should verify the child pointer is not null before dereferencing it.
			///		
			///		, 19.07.2018. 
			///	</remarks>
			/// 
			/// <returns>	The number children. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			int GetNumChildren() const;

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Attaches a child. </summary>
			///
			/// <remarks>	
			/// 	Attach a child to this node.  If the function succeeds, the return
			///		value is the index i of the array where the child was stored, in which
			///		case 0 <= i < GetNumChildren().  The first available empty slot of the
			///		child array is used for storage.  If all slots are filled, the child
			///		is appended to the array (potentially causing a reallocation of the
			///		array).
			///
			///		The function fails when 'child' is null or when 'child' already has a
			///		parent, in which case the return value is -1.  The nodes form a tree,
			///		not a more general directed acyclic graph.  A consequence is that a
			///		node cannot have more than one parent.  For example,
			///			Node* node0 = &lt;some node>;
			///			Spatial* child = &lt;some child&gt;
			///			int index = node0->AttachChild(child);
			///			Node* node1 = &lt;some node&gt;
			///
			///			// This asserts because 'child' already has a parent (node0).
			///			node1->AttachChild(child);
			///
			///			// The following is the correct way to give 'child' a new parent.
			///			node0->DetachChild(child);  // or node0->DetachChildAt(index);
			///			node1->AttachChild(child);
			///
			///			// In the last example before the DetachChild call, if 'child' is
			///			// referenced only by node0, the detach will cause 'child' to be
			///			// deleted (Node internally reference counts its children).  If
			///			// you want to keep 'child' around for later use, do the following.
			///			Spatial::SP saveChild = SPCreate(node0->GetChild(0));
			///			node0->DetachChild(saveChild);
			///			node1->AttachChild(saveChild);
			///			
			///		, 19.07.2018. 
			///	</remarks>
			///
			/// <param name="child">	The child. </param>
			///
			/// <returns>	An int. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			int AttachChild(std::shared_ptr<GPNode<Scalar>> const& child);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Detaches a child. </summary>
			///
			/// <remarks>	
			/// 	Detach a child from this node.  If the 'child' is non-null and in the
			///		array, the return value is the index in the array that had stored the
			///		child.  Otherwise, the function returns -1.
			/// 
			/// 	, 19.07.2018. 
			/// </remarks>
			///
			/// <param name="child">	The child. </param>
			///
			/// <returns>	An int. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			int DetachChild(std::shared_ptr<GPNode<Scalar>> const& child);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Detaches a child at index. </summary>
			///
			/// <remarks>
			/// 	Detach a child from this node.  If 0 <= i < GetNumChildren(), the
			///		return value is the child at index i; otherwise, the function returns
			///		null.
			/// 			
			/// 	, 19.07.2018. 
			/// </remarks>
			///
			/// <param name="i">	Zero-based index of the child. </param>
			///
			/// <returns>	A std::shared_ptr&lt;Spatial&gt; </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			std::shared_ptr<GPNode<Scalar>> DetachChildAt(int i);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Detach all children from this node. </summary>
			///
			/// <remarks>	, 19.07.2018. </remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void DetachAllChildren();

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Sets a child. </summary>
			///
			/// <remarks>	
			/// 	The same comments for AttachChild apply here regarding the inability
			///		to have multiple parents.  If 0 <= i < GetNumChildren(), the function
			///		succeeds and returns i.  If i is out of range, the function *still*
			///		succeeds, appending the child to the end of the array.  The return
			///		value is the previous child stored at index i.
			/// 			
			/// 	, 19.07.2018. 
			/// </remarks>
			///
			/// <param name="i">		Zero-based index of the. </param>
			/// <param name="child">	The child. </param>
			///
			/// <returns>	A std::shared_ptr&lt;Spatial&gt; </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			std::shared_ptr<GPNode<Scalar>> SetChild(int i,
				std::shared_ptr<GPNode<Scalar>> const& child);


			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Gets a child at index. </summary>
			///
			/// <remarks>	
			/// 	Get the child at the specified index.  If 0 <= i < GetNumChildren(),
			///		the function succeeds and returns the child at that index.  Keep in
			///		mind that child[i] could very well be null.  If i is out of range, the
			///		function returns null.
			/// 			
			/// 	, 19.07.2018. 
			/// </remarks>
			///
			/// <param name="i">	Zero-based index of the. </param>
			///
			/// <returns>	The child. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			std::shared_ptr<GPNode<Scalar>> GetChild(int i);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			///		Access to the parent object, which is null for the root of the
			/// 	hierarchy.
			/// </summary>
			///
			/// <remarks>	Hmetal T, 10.08.2016. </remarks>
			///
			/// <returns>	null if it fails, else the parent. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			GPNode<Scalar>* GetParent();

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	
			/// 	Access to the parent object.  Node calls this during attach/detach of children. 
			/// </summary>
			///
			/// <remarks>	Hmetal T, 04/04/2020. </remarks>
			///
			/// <param name="parent">	[in,out] If non-null, the parent. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void SetParent(GPNode<Scalar>* parent);

		protected:
			// Child pointers.
			std::vector<std::shared_ptr<GPNode<Scalar>>> mChild;
			GPNode<Scalar>* mParent;
		private:
			friend class boost::serialization::access;
			friend class GPModels::AEP::SDGPLVM<Scalar>;
			friend class GPModels::AEP::SGPLVM<Scalar>;

			template<class Archive>
			void serialize(Archive& ar, unsigned int version)
			{
				//ar& boost::serialization::base_object<GPLVMBaseModel<Scalar>>(*this);

				ar.register_type<GPModels::AEP::SDGPLVM<Scalar>>();
				ar.register_type<GPModels::AEP::SGPLVM<Scalar>>();

				ar& BOOST_SERIALIZATION_NVP(mParent);
				ar& BOOST_SERIALIZATION_NVP(mChild);
			}

		};
	}
}