﻿/**
File:		MachineLearning/GPModels/Models/Layers/EmissionLayers/FgGaussEmissionLayer.h

Author:		
Email:		
Site:       

Copyright (c) 2022 . All rights reserved.
*/

#pragma once

#include <MachineLearning/FgILayer.h>


namespace NeuralEngine
{
	namespace MachineLearning
	{
		namespace GPModels
		{
			
			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			///		Likelihood emission layer for State Space Models based on Gaussian distribution.
			/// </summary>
			///
			/// <remarks>
			///		<para>
			///			We assume that the emission likelihood takes a Gaussian form,
			///			
			///			\f[ p(\mathbf{y}_t|\mathbf{x}_t) = \mathcal{N}(\mathbf{y}_t;\mathbf{Ux}_t,\mathbf{R_y}),\f]
			///			
			///			and that the approximate factor \f$\mathbf{γ}_t(\mathbf{x}_t)\f$ is also a Gaussian over
			///			\f$\mathbf{x}_t\f$. As such, it is straightforward to show that in the Gaussian case,
			///			the parameters for \f$\mathbf{\gamma}_t(\mathbf{x}_t)\f$ are identical to that of
			///			\f$p(\mathbf{y}_t|\mathbf{x}_t)\f$, when \f$p(\mathbf{y}_t|\mathbf{x}_t)\f$ is viewed
			///			as a Gaussian over \f$\mathbf{x}_t\f$,
			///
			///			\f[ \mathbf{\gamma}_t(\mathbf{x}_t)\propto \exp(−\frac{1}{2}\mathbf{x}_t\mathbf{U}^T\mathbf{R_y}\mathbf{Ux}^T_t+\mathbf{x}_t\mathbf{U}^T\mathbf{R_y}\mathbf{y}).\f]
			///
			///			Note that this optimal form does not depend on the power parameter of Power EP, \f$\alpha\f$.
			///			However,the log-normaliser of the tilted distribution and hence the contribution of the
			///			emission likelihood towards the approximate marginal likelihood does depend on \f$\alpha\f$,
			///			as follows,
			///
			///			\f[ \tilde{Z}=\int d\mathbf{x}_t q^{\backslash t}(\mathbf{x}_t) p^{\alpha}(\mathbf{y}_t|\mathbf{x}_t)
			///			=\frac{(2\pi)^{D_y/2}|\alpha^{−1}\mathbf{R_y}|}{(2\pi)^{\alpha D_y/2}|\mathbf{R_y}|^{\alpha}}
			///			\mathcal{N}(\mathbf{y}_t;\mathbf{Um}^{\backslash t}_{\mathbf{x}_t};\mathbf{US}^{\backslash t}_{\mathbf{x}_t}\mathbf{U}^T+\alpha^{−1}\mathbf{R_y}).\f] 
			///		</para>
			///		<para>
			///			References:
			///			<list type="bullet">
			///			<item>
			///			  	  <description><a href="http://mlg.eng.cam.ac.uk/thang/docs/papers/thesis-thang.pdf" target="_blank">
			///					 Bui, T. D. (2018). Efficient Deterministic Approximate Bayesian Inference for Gaussian 
			///					 Process models (Doctoral thesis). https://doi.org/10.17863/CAM.20913  </a>
			///			     </description>
			///			  </item>
			///		</para>
			///		
			/// 	HmetalT, 04/05/2022. 
			/// </remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			template<typename Scalar>
			class NE_IMPEXP GaussEmission : public ILayer<Scalar>
			{
			public:

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Constructor. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				///
				/// <param name="y">			data array. </param>
				/// <param name="outputDim">	The output dim. </param>
				/// <param name="inputDim"> 	The input dim. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////

				GaussEmission(const af::array& y, int outputDim, int inputDim);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Destructor. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual ~GaussEmission();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Probabilistic output. </summary>
				///
				/// <remarks>	Hmetal T, 22/04/2022. </remarks>
				///
				/// <param name="mf">   	The noiseless mean. </param>
				/// <param name="vf">   	The noiseless variance. </param>
				/// <param name="myOut">	[in,out] my out. </param>
				/// <param name="vyOut">	[in,out] The vy out. </param>
				/// <param name="alpha">	(Optional) The alpha. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void ProbabilisticOutput(const af::array& mf, const af::array& vf, af::array& myOut, af::array& vyOut, Scalar alpha = 1.0f);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Calculates tilted distribution and its gradients. </summary>
				///
				/// <remarks>	Hmetal T, 06/04/2022. </remarks>
				///
				/// <param name="mx">			Cavity mean. </param>
				/// <param name="vx">			Cavity variance. </param>
				/// <param name="y">			Data array. </param>
				/// <param name="outGradInput">	[in,out] The gradient of the inputs. </param>
				/// <param name="outHyperGrad">	[in,out] The gradient of the hyper parameters. </param>
				/// <param name="alpha">		(Optional) The alpha ratio. </param>
				///
				/// <returns>	The calculated log z coordinate emission. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual Scalar ComputeLogZEmission(const af::array& mx, const af::array& vx, const af::array& y, 
					const Scalar scale, std::map<std::string, af::array>* outGradInput, af::array& outHyperGrad, Scalar alpha = 1.0);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets number of parameters to be optimized. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				///
				/// <returns>	The number parameters. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual int GetNumParameters();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets the parameters for each optimization iteration. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				///
				/// <param name="param">	The parameter. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void SetParameters(const af::array& param);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets the parameters for each optimization iteration. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				///
				/// <returns>	The parameters. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual af::array GetParameters();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Updates the parameters. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void UpdateParameters();

				virtual void InitParameters();

				GaussEmission();

			protected:

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Default constructor. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////

				af::array afY;
				af::array afU;
				af::array afR;

				int iq;


			private:
				friend class boost::serialization::access;

				template<class Archive>
				void serialize(Archive& ar, unsigned int version)
				{
					ar& boost::serialization::base_object<ILayer<Scalar>>(*this);
					
					ar& BOOST_SERIALIZATION_NVP(afU);
					ar& BOOST_SERIALIZATION_NVP(afR);
					ar& BOOST_SERIALIZATION_NVP(iq);
				}
			};
		}
	}
}
