/**
File:		MachineLearning/Util/FgGaussHermiteQuadrature.h

Author:		
Email:		
Site:       

Copyright (c) 2017 . All rights reserved.
*/

#pragma once

#include <NeMachineLearningLib.h>
#include <MachineLearning/CommonUtil.h>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		template<typename Scalar>

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Gauss-Hermite Quadrature. </summary>
		///
		/// <remarks>	
		///		<para>
		///			GaussHermite quadrature is a form of Gaussian quadrature for approximating the 
		///			value of integrals of the following kind:
		///			
		///			$$\int_{-\inf}^{\inf} \exp(-x^2)f(x)dx.$$
		///			
		///			In this case
		///			
		///			$$\int_{-\inf}^{\inf} \exp(-x^2)f(x)dx\approx\sum_{i=1}^{n}w_i f(x_i),$$
		///			
		///			where \f$n\f$ is the number of sample points used. The \f$x_i\f$ are the roots
		///			of the physicists' version of the Hermite polynomial \f$H_n(x) (i = 1,2,...,n)\f$,
		///			and the associated weights \f$w_i\f$ are given by
		///			
		///			$$w_i=\frac{2^{n-1}n!\sqrt{\pi}}{n^2 [H_{n-1}(x_i)]^2}.$$
		///		</para>
		///					
		///											
		/// 	, 25.09.2019. 
		/// </remarks>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		class NE_IMPEXP GaussHermiteQuadrature
		{
		public:

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Data direct. </summary>
			///
			/// <remarks>	
			/// 	Computes the sample points and weights for Gauss-Hermite quadrature.
			/// 			
			/// 	, 25.09.2019. </remarks>
			///
			/// <param name="n">	Approximation order. </param>
			/// <param name="x">	[in,out] The sample points. </param>
			/// <param name="w">	[in,out] The weights. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static void Compute(int n, af::array& x, af::array& w);

		private:

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Return the scaled companion matrix of c. </summary>
			///
			/// <remarks>	, 25.09.2019. </remarks>
			///
			/// <param name="c">	The af::array to process. </param>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array HermCompanion(const af::array& c);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Evaluate a normalized Hermite polynomial.. </summary>
			///
			/// <remarks>	
			/// 	Compute the value of the normalized Hermite polynomial of degree $$n$$
			///		at the points $$\mathbf{x}$$.
			///			
			/// 	HmetalT, 12/11/2019. 
			/// </remarks>
			///
			/// <param name="x">	Points at which to evaluate the function. </param>
			/// <param name="n">	Degree of the normalized Hermite function to be evaluated. </param>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array NormedHermite(const af::array& x, int n);
		};
	}
}
