/**
File:		MachineLearning/Graph/Node/FgINode.h

Author:		
Email:		
Site:       

Copyright (c) 2017 . All rights reserved.
*/

#pragma once

#include <NeMachineLearningLib.h>
#include <MachineLearning/FgIMessage.h>
#include <set>
#include <string>
#include <algorithm>
#include <MachineLearning/FgArrayFireSerialization.h>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Base Class INode. </summary>
		///
		/// <remarks>
		/// 	Basic functionalities of a Node in a Factor Graph.
		/// 		
		/// 	 Admin, 7/24/2017. 
		/// </remarks>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		class NE_IMPEXP INode
		{
		public:

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Default constructor. </summary>
			///
			/// <remarks>	 Admin, 7/24/2017. </remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			INode(std::string name);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Destructor. </summary>
			///
			/// <remarks>	 Admin, 7/24/2017. </remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			~INode();

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Gets the name. </summary>
			///
			/// <remarks>	Hmetal T, 11.08.2017. </remarks>
			///
			/// <returns>	A std::string. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			std::string Name();

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Sends a message. </summary>
			///
			/// <remarks>	 Admin, 7/24/2017. </remarks>
			///
			/// <param name="toNodeID">	Identifier to which node should be the message send. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual void SendMsg(std::string toNodeName);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Receives a message. </summary>
			///
			/// <remarks>	 Admin, 7/24/2017. </remarks>
			///
			/// <param name="fromNodeID">	Identifier for from which node the message comes from. </param>
			/// <param name="message">   	[in,out] The message. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual void ReceiveMsg(std::string fromNodeName, IMessage &message);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Adds ingoing connection to the node. </summary>
			///
			/// <remarks>	 Admin, 7/25/2017. </remarks>
			///
			/// <param name="node">	[in,out] If non-null, the node. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual void AddIncoming(INode *node);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Adds outgoing connection to the node. </summary>
			///
			/// <remarks>	 Admin, 7/25/2017. </remarks>
			///
			/// <param name="node">	[in,out] If non-null, the node. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual void AddOutgoing(INode *node);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Adds a custom connection with the given tag. </summary>
			///
			/// <remarks>	 Admin, 7/25/2017. </remarks>
			///
			/// <param name="node">	[in,out] If non-null, the node. </param>
			/// <param name="tag"> 	The tag. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual void AddConnection(INode *node, const std::string &tag);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Query if message type is supported. </summary>
			///
			/// <remarks>	 Admin, 7/25/2017. </remarks>
			///
			/// <param name="type">	The type. </param>
			///
			/// <returns>	true if supported, false if not. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual bool IsSupported(MsgType type) = 0;

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Gets all the messages of this node. </summary>
			///
			/// <remarks>	 Admin, 7/25/2017. </remarks>
			///
			/// <returns>	A reference to a const MsgBox. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			const MsgBox& GetAllMessages() const;

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Clears all incomming messages for next iteration. </summary>
			///
			/// <remarks>	 Admin, 7/26/2017. </remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void ClearMessages();

			//////////////////////////////////////////////////////////////////////////////////////////////////////
			///// <summary>	Smaller-than operator (only compares labels) </summary>
			/////
			///// <remarks>	Hmetal T, 10.08.2017. </remarks>
			/////
			///// <param name="n">	The INode to process. </param>
			/////
			///// <returns>	true if the first parameter is less than the second. </returns>
			//////////////////////////////////////////////////////////////////////////////////////////////////////
			//bool operator<(INode &n);

			//////////////////////////////////////////////////////////////////////////////////////////////////////
			///// <summary>	Larger-than operator (only compares labels) </summary>
			/////
			///// <remarks>	Hmetal T, 10.08.2017. </remarks>
			/////
			///// <param name="n">	The INode to process. </param>
			/////
			///// <returns>	true if the first parameter is greater than to the second. </returns>
			//////////////////////////////////////////////////////////////////////////////////////////////////////
			//bool operator>(INode &n);

			//////////////////////////////////////////////////////////////////////////////////////////////////////
			///// <summary>	Smaller-than-or-equal-to operator (only compares labels) </summary>
			/////
			///// <remarks>	Hmetal T, 10.08.2017. </remarks>
			/////
			///// <param name="n">	The INode to process. </param>
			/////
			///// <returns>	true if the first parameter is less than or equal to the second. </returns>
			//////////////////////////////////////////////////////////////////////////////////////////////////////
			//bool operator<=(INode &n);

			//////////////////////////////////////////////////////////////////////////////////////////////////////
			///// <summary>	Larger-than-or-equal-to operator (only compares labels) </summary>
			/////
			///// <remarks>	Hmetal T, 10.08.2017. </remarks>
			/////
			///// <param name="n">	The INode to process. </param>
			/////
			///// <returns>	true if the first parameter is greater than or equal to the second. </returns>
			//////////////////////////////////////////////////////////////////////////////////////////////////////
			//bool operator>=(INode &n);

			//////////////////////////////////////////////////////////////////////////////////////////////////////
			///// <summary>	Not-equal-to operator (only compares labels) </summary>
			/////
			///// <remarks>	Hmetal T, 10.08.2017. </remarks>
			/////
			///// <param name="n">	The INode to process. </param>
			/////
			///// <returns>	true if the parameters are not considered equivalent. </returns>
			//////////////////////////////////////////////////////////////////////////////////////////////////////
			//bool operator!=(INode &n);

			//////////////////////////////////////////////////////////////////////////////////////////////////////
			///// <summary>	Equal-to operator (only compares labels) </summary>
			/////
			///// <remarks>	Hmetal T, 10.08.2017. </remarks>
			/////
			///// <param name="n">	The INode to process. </param>
			/////
			///// <returns>	true if the parameters are considered equivalent. </returns>
			//////////////////////////////////////////////////////////////////////////////////////////////////////
			//bool operator==(INode &n);

		protected:

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Sets a incomming message from specific node. </summary>
			///
			/// <remarks>	 Admin, 7/25/2017. </remarks>
			///
			/// <param name="fromNodeID">	Identifier for from node. </param>
			/// <param name="msg">		 	The message. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void SetMessage(std::string fromNodeName, IMessage &msg);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Virtual function to calculate a message to connectd node. </summary>
			///
			/// <remarks>	
			/// 	Different computations for all kinds of derived classes of node types.
			/// 	
			/// 	 Admin, 7/26/2017. 
			/// </remarks>
			///
			/// <param name="toNodeName">		 	Identifier for receiver node. </param>
			/// <param name="neededMessages">	All incomming messages, exept from receiver node. </param>
			///
			/// <returns>	The calculated message. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual IMessage ComputeMessage(std::string fromNodeName, MsgBox &neededMessages) = 0;

			std::map<std::string, INode*> mNodes;	// dictionary of connected nodes
			MsgBox mMessageBox;						// all incomming messages
			std::set<std::string> sIncomming;		// incomming connections
			std::set<std::string> sOutgoing;		// outgoing connections

			std::map<std::string, std::string> mConnections;	// custom connections

		private:
			friend class boost::serialization::access;

			template<class Archive>
			void serialize(Archive& ar, unsigned int version)
			{
				ar& BOOST_SERIALIZATION_NVP(mNodes);
				ar& BOOST_SERIALIZATION_NVP(sIncomming);
				ar& BOOST_SERIALIZATION_NVP(sOutgoing);
				ar& BOOST_SERIALIZATION_NVP(mMessageBox);
				ar& BOOST_SERIALIZATION_NVP(sName);
			}

			
			std::string sName;
			//int iId;
		};
	}
}
