﻿/**
File:		MachineLearning/GPModels/Backconstraints/FgKernelBasedRegression.h

Author:		
Email:		
Site:       

Copyright (c) 2020 . All rights reserved.
*/

#pragma once

#include <MachineLearning/FgIBackconstraint.h>
#include <MachineLearning/FgKernels.h>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		namespace GPModels
		{
			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	
			/// 	Back-constraints via kernel based regression.			
			/// </summary>
			///
			/// <remarks>	
			///		<para>
			///			Rather than maximising the likelihood with respect to \f$\mathbf{X}\f$ directly, 
			///			we replace each element of \f$\mathbf{X}\f$ with a mapping of the form:
			///			
			///			\f[ x_{nj} = g_j(\mathbf{y}_n;\mathbf{w}). \f]
			///			
			///			Two points in latent space will then be constrained to always be close if their data 
			///			space counterparts are close. Instead of direct likelihood maximization, we now maximize 
			///			a constrained likelihood, the constraints preserving nearby ‘localities’. How close is 
			///			'nearby' is determined by the smoothness of the mapping. For example we use for the 
			///			mapping an RBF kernel,
			///			
			///			\f[ g_j(\mathbf{y}_n) = \sum_{m=1}^N \alpha_{jm} k(\mathbf{y}_n, \mathbf{y}_m), \f] 
			///					
			///			where \f$\mathbf{A} = \{\{\alpha_{jn}\}_{n=1}^N\}_{j=1}^q \f$ are the parameters, and
			///			the kernel matrix is, 
			///			
			///			\f[ k(\mathbf{y}_n, \mathbf{y}_m) = \exp\left(-\frac{\gamma}{2}(\mathbf{y}_n 
			///				− \mathbf{y}_m)^T(\mathbf{y}_n − \mathbf{y}_m)\right), \f]
			///				
			///			closeness is determined by the setting of the inverse width parameter \f$\gamma\f$.
			///		</para>
			///		
			///		<para>
			///		  Reference:
			///		  <list type="bullet">
			///			<item>
			///		    	  <description><a href="https://www.microsoft.com/en-us/research/wp-content/uploads/2016/02/lawrence06bcgplvm.pdf" target="_blank">
			///					Lawrence, Neil D. and Qui\~{n}onero-Candela, Joaquin. (2006). "Local Distance 
			///					Preservation in the GP-LVM through Back Constraints".
			///					Association for Computing Machinery, New York, NY, USA </a>
			///		       </description>
			///		    </item>
			///		</para>
			///								
			/// 	HmetalT, 26.10.2017. 
			/// </remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			template<typename Scalar>
			class NE_IMPEXP KBR : public IBackconstraint<Scalar>
			{
			public:

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Default constructor. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				KBR();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Destructor. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual ~KBR();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Initializes this object. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <param name="Y">	An af::array to process. </param>
				/// <param name="X">	An af::array to process. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void Init(const af::array& Y, const af::array& X, const af::array& segments);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets number of to be optimized parameters. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <returns>	The number parameters. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual int GetNumParameters();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets the parameters. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <param name="param">	The parameter. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void SetParameters(const af::array& param);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets the parameters. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <returns>	The parameters. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual af::array GetParameters();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets constraint x coordinates. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <returns>	The constraint x coordinate. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual af::array GetConstraintX();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Back-constraint gradient. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <param name="gX">	The gradient of \f$\mathbf{X}\f$. </param>
				///
				/// <returns>	The gradient of the back-constraint. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual af::array BackconstraintGradient(const af::array& gX);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets a kernel function. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <param name="inKernel">	[in,out] If non-null, the in kernel. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void SetKernel(IKernel<Scalar>* inKernel);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets the kernel function. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <returns>	Null if it fails, else the kernel. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual IKernel<Scalar>* GetKernel();

			protected:

				af::array afA;				//!< parameters
				af::array afK;				//!< kernel matrix
				IKernel<Scalar>* kernel;	//!< kernel function

			private:
				friend class boost::serialization::access;

				template<class Archive>
				void serialize(Archive& ar, unsigned int version)
				{
					ar& boost::serialization::base_object<IBackconstraint<Scalar>>(*this);

					ar& BOOST_SERIALIZATION_NVP(afA);
					ar& BOOST_SERIALIZATION_NVP(afK);
					ar& BOOST_SERIALIZATION_NVP(kernel);
				}
			};
		}
	}
}
