/**
File:		MachineLearning/Kernel/FgRBFKernel.h

Author:		Nick Taubert
Email:		nick.taubert@uni-tuebingen.de
Site:       http://www.compsens.uni-tuebingen.de/

Copyright (c) 2017 CompSens. All rights reserved.
*/

#pragma once

#include <NeEngineLib.h>
#include <MachineLearning/FgIKernel.h>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Radial basis kernel function. </summary>
		///
		/// <remarks>
		/// 	Kernel function constructed as follows:
		/// 	
		/// 	\f[k(\mathbf{x}, \mathbf{x}') = \gamma_1 \exp\left(-\frac{1}{\gamma_2} 
		/// 	|\mathbf{x} - \mathbf{x}'|^2 \right).\f]
		/// 			
		/// 	Nick Admin, 5/24/2017. 
		/// </remarks>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		template<typename Scalar>
		class NE_IMPEXP RBFKernel : public IKernel<Scalar>
		{
		public:
			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Default constructor. </summary>
			///
			/// <remarks>	Nick Admin, 5/24/2017. </remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			RBFKernel();

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Destructor. </summary>
			///
			/// <remarks>	Nick Admin, 5/24/2017. </remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			~RBFKernel();

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Computes the kernel matrix of the kernel. </summary>
			///
			/// <remarks>	Nick Admin, 5/23/2017. </remarks>
			///
			/// <param name="inX1">	 	[in,out] First n times q matrix of latent points. </param>
			/// <param name="inX2">	 	[in,out] Second m times q matrix of latent points (X'). </param>
			/// <param name="outMatrix">	[in,out] Resulting kernel matrix.< / param> </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void ComputeKernelMatrix(const af::array& inX1, const af::array& inX2, af::array& outMatrix);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Calculates only diagonal elements of K. </summary>
			///
			/// <remarks>	Nick Admin, 5/24/2017. </remarks>
			///
			/// <param name="inX">		  	[in,out] Nxq matrix X. </param>
			/// <param name="outDiagonal">	[in,out] The out diagonal. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void ComputeDiagonal(const af::array& inX, af::array& outDiagonal);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Computes dL/dX for full fit GP. </summary>
			///
			/// <remarks>	Nick Admin, 5/23/2017. </remarks>
			///
			/// <returns>
			/// 	The method returns the gradients of the latent points of type ILArray&lt;Scalar&gt;.
			/// </returns>
			///
			/// <param name="inX">	   	[in,out] Nxq Matrix of latent points. </param>
			/// <param name="indL_dK">
			/// 	[in,out] Derivative of the loglikelihood w.r.t kernel matrix K.
			/// </param>
			/// <param name="outdL_dX">	[in,out] Derivative of the loglikelihood w.r.t latent points. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void LogLikGradientX(const af::array& inX, const af::array& indL_dK, af::array& outdL_dX);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Computes dL / dX and dL / dXu for sparse approximation GP. </summary>
			///
			/// <remarks>	Nick Admin, 5/23/2017. </remarks>
			///
			/// <param name="inXu">			[in,out] kxq Matrix of latent subset points. </param>
			/// <param name="indL_dKuu">	[in,out] Derivative of the loglikelihood w.r.t subset kernel matrix Kuu. </param>
			/// <param name="inX">			[in,out] Nxq Matrix of latent points. </param>
			/// <param name="indL_dKuf">	[in,out] Derivative of the loglikelihood w.r.t the kernel matrix K. </param>
			/// <param name="outdL_dXu">	[in,out] Derivative of the loglikelihood w.r.t latent subset points Xu. </param>
			/// <param name="outdL_dX"> 	[in,out] Derivative of the loglikelihood w.r.t latent points X. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void LogLikGradientX(const af::array& inXu, const af::array& indL_dKuu, const af::array& inX, const af::array& indL_dKuf, af::array& outdL_dXu, af::array& outdL_dX);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Computes the gradient of the kernel parameters. </summary>
			///
			/// <remarks>	Nick Admin, 5/23/2017. </remarks>
			///
			/// <param name="indL_dK">
			/// 	[in,out] Derivative of the loglikelihood w.r.t the kernel matrix K.
			/// </param>
			/// <param name="outdL_dParam">	[in,out] Gradient of kernel parameters. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void LogLikGradientParam(const af::array& inX1, const af::array& inX2, const af::array& indL_dK, af::array& outdL_dParam);

			virtual void LogLikGradientCompundKfu(const af::array& indL_dKfu, const af::array& inX, const af::array& inXu,
				af::array* outdL_dParam, af::array* outdL_dXu, const af::array* dlogZ_dv = nullptr, af::array* outdL_dX = nullptr) override;

			virtual void LogGradientCompoundKuu(const af::array& inXu, const af::array& inCovDiag,
				af::array* outdL_dParam, af::array* outdL_dXu) override;

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Computes dK/dX. </summary>
			///
			/// <remarks>	Nick Admin, 5/23/2017. </remarks>
			///
			/// <param name="inX1">	   	[in,out] First n times q matrix of latent points. </param>
			/// <param name="inX2">	   	[in,out] Second n times q matrix of latent points (X'). </param>
			/// <param name="q">	   	The latent dimension to process. </param>
			/// <param name="outdK_dX">	[in,out] dK/dX. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void GradX(const af::array& inX1, const af::array& inX2, int q, af::array& outdK_dX);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Derivative of diagonal elemts of K w.r.t X. </summary>
			///
			/// <remarks>	Nick Admin, 5/23/2017. </remarks>
			///
			/// <param name="inX">		   	[in,out] The in x coordinate. </param>
			/// <param name="outDiagdK_dX">	[in,out] Derivative of diagonal elemts of K w.r.t X. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void DiagGradX(const af::array& inX, af::array& outDiagdK_dX);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Derivative of diagonal elemts of K w.r.t kernel parameters. </summary>
			///
			/// <remarks>	Nick Admin, 5/23/2017. </remarks>
			///
			/// <param name="inX">			   	[in,out] Nxq matrix of Latent points X. </param>
			/// <param name="inCovDiag">	   	[in,out] Diagonal of kernel matrix K. </param>
			/// <param name="outDiagdK_dParam">	[in,out] Derivative of diagonal elemts of K w.r.t kernel parameters. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void DiagGradParam(const af::array& inX, const af::array& inCovDiag, af::array& outDiagdK_dParam);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Sets the parameters. </summary>
			///
			/// <remarks>	Nick, 26.06.2018. </remarks>
			///
			/// <param name="param">	The parameter. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void SetParameters(const af::array& param);

			virtual void SetLogParameters(const af::array& param) override;

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Gets the parameters. </summary>
			///
			/// <remarks>	Nick, 26.06.2018. </remarks>
			///
			/// <returns>	The parameters. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			af::array GetParameters();

			virtual af::array GetLogParameters() override;

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// PSI statistics
			////////////////////////////////////////////////////////////////////////////////////////////////////

			void Psi1Derivative(const af::array& inPsi1, const af::array& indL_dpsi1, const af::array& inZ, const af::array& inMu,
				const af::array& inSu, af::array& outdL_dParam, af::array& outdL_dXu, af::array* outdL_dX = nullptr);

		private:
			Scalar dVariance, dInvScale;

			friend class boost::serialization::access;

			template<class Archive>
			void serialize(Archive& ar, unsigned int version)
			{
				ar & boost::serialization::base_object<IKernel<Scalar>>(*this);
				ar& BOOST_SERIALIZATION_NVP(dVariance);
				ar& BOOST_SERIALIZATION_NVP(dInvScale);
			}
		};
	}
}
