/**
File:		MachineLearning/Models/GPModels/FgSparseGPLVMBaseModel.h

Author:		
Email:		
Site:       

Copyright (c) 2017 . All rights reserved.
*/

#pragma once

#include <MachineLearning/FgGPLVMBaseModel.h>
#include <MachineLearning/FgSparseGPBaseLayer.h>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		namespace GPModels
		{
			namespace PowerEP
			{
				template<typename Scalar>
				class SGPLayer;
			}

			namespace AEP
			{
				template<typename Scalar>
				class SGPLayer;
			}

			//namespace VFE
			//{
			//	//template<Scalar>
			//	class SGPLayer;
			//}

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary> Base class for all sparse GPLVM models. </summary>
			///
			/// <remarks>
			/// 	<para>	
			/// 		GPLVM are the nonlinear dual version of probabilistic PCA, where a low dimensional 
			/// 		latent variable \f$\mathbf{X}=[\mathbf{x}_1,...,\mathbf{x}_N]^T\f$ is mapped onto a
			/// 		high dimensional data variable \f$\mathbf{Y}=[\mathbf{y}_1,...,\mathbf{y}_N]^T\f$ via
			/// 		prior mapping function \f$f(\mathbf{x})\f$. The difference to normal GPs is the uncertainty
			/// 		of \f$\mathbf{X}\f$, which will be initialized via PCA and optimized during learning.
			/// 		To avoid memory issues for larger data sets, the algorithm takes use of sparse approximation
			/// 		techniques.
			///		</para>
			///		<para>
			/// 		Sparse approximations are used for larger
			/// 		data sets to reduce memory size and computational complexity.  This is
			/// 		done by introducing a subset of inducing points or pseudo inputs to approximate
			/// 		the full set. The inversion of the kernel matrix depends only on those points
			/// 		and reduces the computational complexity from \f$O(N^3)\f$ to \f$O(k^2N)\f$, where
			/// 		\f$k\f$ is the number of inducing points and \f$N\f$ the length of the data set.
			///		</para>
			/// 	<para>
			///			References:
			///			<list type="bullet">
			///			<item>
			///			  	  <description><a href="http://proceedings.mlr.press/v2/lawrence07a/lawrence07a.pdf" target="_blank">
			///					Lawrence, Neil (2007). "Learning for Larger Datasets with the Gaussian Process 
			///					Latent Variable Model". Journal of Machine Learning Research - Proceedings Track,
			///					pp 243-250, </a>
			///			     </description>
			///			  </item>
			///		</para>
			/// 	
			/// 	
			/// 	, 24.11.2019. 
			/// </remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			template<typename Scalar>
			class NE_IMPEXP SparseGPLVMBaseModel : public GPLVMBaseModel<Scalar>
			{
			public:

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Constructor. </summary>
				///
				/// <remarks>	, 21.03.2018. </remarks>
				///
				/// <param name="Y">			  	The training data. </param>
				/// <param name="latentDimension">	The latent dimension. </param>
				/// <param name="priorMean">	  	(Optional) The prior mean. </param>
				/// <param name="priorVariance">  	(Optional) The prior variance. </param>
				/// <param name="numInducing">	  	(Optional) Number of inducing points. </param>
				/// <param name="lType">		  	(Optional) The likelihood or objective type. </param>
				/// <param name="emethod">		  	(Optional) The initialisation method of the latent variable. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				SparseGPLVMBaseModel(const af::array& Y, int latentDimension, Scalar priorMean = 0.0, Scalar priorVariance = 1.0, 
					int numInducing = 200, LogLikType lType = LogLikType::Gaussian, XInit emethod = XInit::pca);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Default constructor. </summary>
				///
				/// <remarks>	, 26.03.2018. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				SparseGPLVMBaseModel();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Destructor. </summary>
				///
				/// <remarks>	, 15.05.2018. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual ~SparseGPLVMBaseModel();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Predict noise free functions values \f$\mathbf{F}_*\f$. </summary>
				///
				/// <remarks>	Hmetal T, 05/05/2020. </remarks>
				///
				/// <param name="testInputs">	The test inputs. </param>
				/// <param name="mf">		 	[in,out] mean of function values. </param>
				/// <param name="vf">		 	[in,out] The variance of function values. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void PredictF(const af::array& testInputs, af::array& mf, af::array& vf) override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Generate function samples from posterior. </summary>
				///
				/// <remarks>	Hmetal T, 18/06/2019. </remarks>
				///
				/// <param name="outFunctions">	[in,out] The out functions. </param>
				/// <param name="inputs">	   	The inputs. </param>
				/// <param name="numSamples">  	Number of samples. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void SampleY(const af::array inputs, int numSamples, af::array& outFunctions) override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Initializes the model. </summary>
				///
				/// <remarks>	Hmetal T, 29.11.2017. </remarks>
				///
				/// <returns>	true if it succeeds, false if it fails. </returns>
					////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual bool Init() override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets number of parameters. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				///
				/// <returns>	The number parameters. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual int GetNumParameters() override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets the parameters for each optimization iteration. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				///
				/// <param name="param">	The parameter. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void SetParameters(const af::array& param) override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets the parameters for each optimization iteration. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				///
				/// <param name="param">	The parameter. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual af::array GetParameters() override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Updates the parameters. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void UpdateParameters() override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets fixation for hyperparameters. </summary>
				///
				/// <remarks>	Hmetal T, 16/12/2019. </remarks>
				///
				/// <param name="isfixed">	True if isfixed. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void FixKernelParameters(bool isfixed) override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Set fixation for inducing inputs. </summary>
				///
				/// <remarks>	Hmetal T, 16/12/2019. </remarks>
				///
				/// <param name="isfixed">	True if isfixed. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void FixInducing(bool isfixed) override;

			protected:

				int ik;			//!< number of inducing inputs
				SparseGPBaseLayer<Scalar>* gpLayer; //!< sparse Gaussian Process layer

			private:
				friend class AEP::SGPLayer<Scalar>;
				friend class PowerEP::SGPLayer<Scalar>;
				//friend class VFE::SGPLayer/*<Scalar>*/;

				friend class boost::serialization::access;

				template<class Archive>
				void serialize(Archive& ar, unsigned int version)
				{
					ar& boost::serialization::base_object<GPLVMBaseModel<Scalar>>(*this);

					//ar& boost::serialization::make_nvp("GPLVMBaseModel", boost::serialization::base_object<GPLVMBaseModel<Scalar>>(*this));

					ar.register_type<AEP::SGPLayer<Scalar>>();
					ar.register_type<PowerEP::SGPLayer<Scalar>>();
					//ar.register_type<VFE::SGPLayer>();

					ar& BOOST_SERIALIZATION_NVP(ik);
					ar& BOOST_SERIALIZATION_NVP(gpLayer);
				}
			};
}
	}
}
