﻿/**
File:		MachineLearning/Models/GPModels/FgSparseGPSSMBaseModel.h

Author:		
Email:		
Site:       

Copyright (c) 2020 . All rights reserved.
*/

#pragma once

#include <MachineLearning/FgGPStateSpaceBaseModel.h>
#include <MachineLearning/FgSparseGPBaseLayer.h>
#include <MachineLearning/FgGaussEmissionLayer.h>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		namespace GPModels
		{
			template<typename Scalar>
			class GaussEmission;

			namespace PowerEP
			{
				template<typename Scalar>
				class SGPLayer;
			}

			namespace AEP
			{
				template<typename Scalar>
				class SGPLayer;
			}

			//namespace VFE
			//{
			//	//template<Scalar>
			//	class SGPLayer;
			//}

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary> Base class for all sparse GPSSM models. </summary>
			///
			/// <remarks>
			/// 	<para>	
			/// 		GP state-space model in which latent variables are continuous and the transition 
			/// 		between these variables is non-linear. Due to the non - linearity in the dynamics, 
			/// 		inferenceand learning, which involve finding the state transition as well as the 
			/// 		latent states themselves, are difficult.The presence of transition and measurement 
			/// 		noise, and a typically small number of measurements in practice make the inference task 
			/// 		even more challenging. This model subclass can be compactly represented as follows,
			/// 		\f[ p(\mathbf{x}_0) = \mathcal{N}(\mathbf{x}_0; \mu_0, \Sigma_0), \f]
			///			\f[ p(\mathbf{x}_t | f, \mathbf{x}_{t−1}) = \mathcal{N}(\mathbf{x}_t; 
			///			f(\mathbf{x}_{t−1}), \mathbf{Q}_x), t = 1 : T, \f]
			///			\f[ p(\mathbf{y}_t | \mathbf{x}_t) = \mathcal{N}(\mathbf{y}_t; \mathbf{U}\mathbf{x}_t, 
			///			\mathbf{R}_y), t = 1 : T, \f]
			///			where the dynamical noise is assumed Gaussian and the measurement or emission model is 
			///			assumed to be linear and Gaussian, \f$\mathbf{x}\f$ and \f$\mathbf{y}\f$ are the 
			///			latent variables and the measurements respectively, \f$f\f$ is a continuous non-
			///			linear transition function, and \f$T\f$ is the number of measurement steps.
			///		</para>
			/// 	<para>
			///			We assume the non-linear transition function f is a draw from a Gaussian process, 
			///			or in other words, we place a flexible nonparametric Gaussian process prior over 
			///			this function, \f$p(f) = GP(m(.), k(., .))\f$. We refer to the resulting model as 
			///			the Gaussian process state - space model (GPSSM).This model belongs to a plethora 
			///			of closely related dynamical systems which describe how the state and emission 
			///			variables evolve using Gaussian processes. The joint probablity of all variablesand 
			///			observations involved can be written as follows,
			///				\f[ p(\mathbf{y}_{1:T}, \mathbf{x}_{0:T}, f | \theta) = p(\mathbf{x}_0)p(f | \theta)
			///				\prod_{t=1}^T p(\mathbf{x}_t | f, \mathbf{x}_{t−1}, \theta)
			///				p(\mathbf{y}_t | \mathbf{x}_t, \theta), \f]
			///			where \f$\theta\f$ includes the kernel hyperparameters, the transition noise, and 
			///			the emission parameters. These hyperparameters can be found by performing model 
			///			selection using the log marginal likelihood:
			///			\f[L(\theta) = log \int p(\mathbf{y}_{1:T} | \theta) dfd\mathbf{x}_{0:T}
			///			p(\mathbf{y}_{1:T}, \mathbf{x}_{0:T}, f | \theta).\f]
			///		</para>
			/// 
			///		<para>
			///			References:
			///			<list type="bullet">
			///			<item>
			///			  	  <description><a href="https://arxiv.org/pdf/1107.4985.pdf" target="_blank">
			///					Damianou, AC, Titsias, MK, Lawrence, ND (2012). "Variational gaussian process 
			///					dynamical systems".
			///					Advances in Neural Information Processing Systems</a>
			///			     </description>
			///			  </item>
			///		</para>
			/// 	
			/// 	
			/// 	, 24.11.2019. 
			/// </remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			template<typename Scalar>
			class NE_IMPEXP SparseGPSSMBaseModel : public GPSSBaseModel<Scalar>
			{
			public:
				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Constructor. </summary>
				///
				/// <remarks>	Hmetal T, 04/05/2020. </remarks>
				///
				/// <param name="Y">			  	Observation data. </param>
				/// <param name="latentDimension">	The latent dimension. </param>
				/// <param name="numInducing">	  	(Optional) Number of inducings inputs. </param>
				/// <param name="priorMean">	  	(Optional) The prior mean. </param>
				/// <param name="priorVariance">  	(Optional) The prior variance. </param>
				/// <param name="xControl">		  	[in,out] (Optional) The control. </param>
				/// <param name="lType">		  	(Optional) The likelihood type. </param>
				/// <param name="GPemission">	  	(Optional) True to non-linear emission function. </param>
				/// <param name="controlToEmiss"> 	(Optional) True to control to emiss. </param>
				/// <param name="emethod">		  	(Optional) The embed method. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				SparseGPSSMBaseModel(const af::array& Y, int latentDimension, int numInducing = 200, Scalar priorMean = 0.0,
					Scalar priorVariance = 1.0, af::array& xControl = af::array(), PropagationMode probMode = PropagationMode::MomentMatching, LogLikType lType = LogLikType::Gaussian,
					bool GPemission = false, bool controlToEmiss = true, XInit emethod = XInit::pca);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Destructor. </summary>
				///
				/// <remarks>	Hmetal T, 05/05/2020. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// 
				virtual ~SparseGPSSMBaseModel();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Initializes the model. </summary>
				///
				/// <remarks>	Hmetal T, 29.11.2017. </remarks>
				///
				/// <returns>	true if it succeeds, false if it fails. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual bool Init() override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Predict noise free functions values \f$\mathbf{F}_*\f$. </summary>
				///
				/// <remarks>	Hmetal T, 05/05/2020. </remarks>
				///
				/// <param name="testInputs">	The test inputs. </param>
				/// <param name="mf">		 	[in,out] mean of function values. </param>
				/// <param name="vf">		 	[in,out] The variance of function values. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void PredictF(const af::array& testInputs, af::array& mf, af::array& vf) override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Prediction of test outputs \f$\mathbf{Y}_*\f$. </summary>
				///
				/// <remarks>	, 12.06.2018. </remarks>
				///
				/// <param name="my"> 	[in,out] The posterior mean function. </param>
				/// <param name="vy"> 	[in,out] The posterior covariance function. </param>
				/// <param name="testX">	[in,out] The test inputs. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void PredictY(const af::array& testInputs, af::array& my, af::array& vy) override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Noise-free Forward prediction. </summary>
				///
				/// <remarks>	Hmetal T, 07/05/2020. </remarks>
				///
				/// <param name="numTimeSamples">	Number of time samples. </param>
				/// <param name="mf">			 	[in,out] The mean uf \f$\matthbf{Y}_*\f$. </param>
				/// <param name="vf">			 	[in,out] The variance of \f$\matthbf{Y}_*\f$. </param>
				/// <param name="mx">			 	[in,out] (Optional) If non-null, the mx. </param>
				/// <param name="vx">			 	[in,out] (Optional) If non-null, the vx. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void PredictForward(int numTimeSamples, af::array& my, af::array& vy, 
					af::array* mx = nullptr, af::array* vx = nullptr);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Get posterior distribution of latent variables /f$\mathbf{X}/f$. </summary>
				///
				/// <remarks>	Hmetal T, 09/12/2019. </remarks>
				///
				/// <param name="my">	[in,out] The posterior mean of the data. </param>
				/// <param name="vy">	[in,out] The posterior variance of the data. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void PosteriorData(af::array& my, af::array& vy);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets number of parameters. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				///
				/// <returns>	The number parameters. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual int GetNumParameters() override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets the parameters for each optimization iteration. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				///
				/// <param name="param">	The parameter. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void SetParameters(const af::array& param) override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets the parameters for each optimization iteration. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				///
				/// <param name="param">	The parameter. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual af::array GetParameters() override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Updates the parameters. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void UpdateParameters() override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets fixation for hyperparameters. </summary>
				///
				/// <remarks>	Hmetal T, 16/12/2019. </remarks>
				///
				/// <param name="isfixed">	True if isfixed. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void FixKernelParameters(bool isfixed);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Set fixation for inducing inputs. </summary>
				///
				/// <remarks>	Hmetal T, 16/12/2019. </remarks>
				///
				/// <param name="isfixed">	True if isfixed. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void FixInducing(bool isfixed);

				SparseGPSSMBaseModel();

			protected:

				int ik;										//!< number of inducing inputs
				SparseGPBaseLayer<Scalar>* dynLayer;		//!< sparse Gaussian Process dynamic layer
				SparseGPBaseLayer<Scalar>* gpEmissLayer;	//!< sparse Gaussian Process emission layer
				GaussEmission<Scalar>* gaussEmissLayer;		//!< Gaussian distibution emission layer

			private:
				friend class AEP::SGPLayer<Scalar>;
				friend class PowerEP::SGPLayer<Scalar>;
				//friend class VFE::SGPLayer/*<Scalar>*/;
				friend class GaussEmission<Scalar>;

				friend class boost::serialization::access;

				template<class Archive>
				void serialize(Archive& ar, unsigned int version)
				{
					ar& boost::serialization::base_object<GPSSBaseModel<Scalar>>(*this);

					//ar& boost::serialization::make_nvp("GPSSBaseModel", boost::serialization::base_object<GPSSBaseModel<Scalar>>(*this));

					ar.register_type<AEP::SGPLayer<Scalar>>();
					ar.register_type<PowerEP::SGPLayer<Scalar>>();
					//ar.register_type<VFE::SGPLayer>();
					ar.register_type<GaussEmission<Scalar>>();

					ar& BOOST_SERIALIZATION_NVP(ik);
					ar& BOOST_SERIALIZATION_NVP(dynLayer);
					ar& BOOST_SERIALIZATION_NVP(gpEmissLayer);
					ar& BOOST_SERIALIZATION_NVP(gaussEmissLayer);
				}
			};

		}
	}
}
