/**
File:		Golfswing.cpp

Author:		
Email:		
Site:       

Copyright (c) 2022 . All rights reserved.
*/

#include <NeCore.h>
#include <NeMachineLearning.h>
#include <iostream>
#include <filesystem>

using namespace NeuralEngine;
using namespace NeuralEngine::MachineLearning;
using namespace NeuralEngine::MachineLearning::GPModels;
using namespace af;

namespace fs = std::filesystem;

////////////////////////////////////////////////////////////////////////////////////////////////////
 /// <summary>	
 /// 	AEP::DeepGPSSM to learn golf swing from cmu database. 
 /// </summary>
 ///
 /// <remarks>	Hmetal T, 01/06/2018. </remarks>
 ///
 /// <param name="path">	Full pathname of the file. </param>
 ////////////////////////////////////////////////////////////////////////////////////////////////////
template<typename Scalar>
void Golfswing_AEP_DGPSSM_Stocatic(std::string path)
{
	af::dtype m_dType = CommonUtil<Scalar>::CheckDType();

	std::string mode = "CPU";

	af::array Y;

	int iq = 2, batchSize = 0, numIter = 1000;
	Scalar priorMean = 0.0, priorVariance = 1.0,  dAlpha = 0.5;

	std::vector<Scalar> valpha{ 0.1, 0.25, 0.5, 0.75, 0.9 };

	int hk = 60, hD = 50;
	HiddenLayerDescription description(hk, hD);

	Scalar ik = 30;
	std::vector<HiddenLayerDescription> descriptions;
	descriptions.push_back(HiddenLayerDescription(ik, 25));
	descriptions.push_back(HiddenLayerDescription(ik, 50));

	// get the Matlab .Mat file from the command line
	std::string fileNameTraining = path + "\\data\\golfswing.mat";

	// create a new reader
	MatlabIO matio;
	bool ok = matio.Open(fileNameTraining, "r");

	// read all of the variables in the file
	std::vector<MatlabIOContainer> variables;
	variables = matio.Read();

	// close the file
	matio.Close();

	// search for a variable named "im" and "gray"
	std::vector<cv::Mat> cvY;
	for (unsigned int n = 0; n < variables.size() - 3; ++n)
	{
			cvY.push_back(variables[n].Data<cv::Mat>());
	}

	for(auto i = 0; i < valpha.size(); i++)
	{
		dAlpha = valpha[i];

		Y = AfCv::MatToArray(cvY[0]);

		AEP::SDGPSSM<Scalar>* model = new AEP::SDGPSSM<Scalar>(Y, iq, description, dAlpha, priorMean, priorVariance);
		for (auto i = 1; i < cvY.size(); i++)
		{
			model->AddData(AfCv::MatToArray(cvY[i]));
			Y = CommonUtil<Scalar>::Join(Y, AfCv::MatToArray(cvY[i]));
		}
		std::cout << "\nImported " << Y.dims(0) << " trajectories with " << Y.dims(1) << " dimensions." << std::endl;

		Timer timer;
		model->Optimise(OptimizerType::ADAM, 0.0, false, numIter, 0, LineSearchType::MoreThuente);

		std::cout << "\nOptimization time: " << timer.GetSeconds() << " seconds.\n" << std::endl;

		std::string fileName = path + "\\resources\\" + mode + "Golfswing_DGPSSM_2Layer_alpha" + std::to_string(dAlpha) + "_time_" + std::to_string(timer.GetSeconds()) + "sec.dat";
		SaveModel<AEP::SDGPSSM<Scalar>>(fileName, model);

		delete model;
	}

	for (auto i = 0; i < valpha.size(); i++)
	{
		dAlpha = valpha[i];

		Y = AfCv::MatToArray(cvY[0]);

		AEP::SDGPSSM<Scalar>* model = new AEP::SDGPSSM<Scalar>(Y, iq, descriptions, dAlpha, priorMean, priorVariance);
		for (auto i = 1; i < cvY.size(); i++)
		{
			model->AddData(AfCv::MatToArray(cvY[i]));
			Y = CommonUtil<Scalar>::Join(Y, AfCv::MatToArray(cvY[i]));
		}
		std::cout << "\nImported " << Y.dims(0) << " trajectories with " << Y.dims(1) << " dimensions." << std::endl;

		Timer timer;
		model->Optimise(OptimizerType::ADAM, 0.0, false, numIter, 0, LineSearchType::MoreThuente);

		std::cout << "\nOptimization time: " << timer.GetSeconds() << " seconds.\n" << std::endl;

		std::string fileName = path + "\\resources\\" + mode + "Golfswing_DGPSSM_3Layer_alpha" + std::to_string(dAlpha) + "_time_" + std::to_string(timer.GetSeconds()) + "sec.dat";
		SaveModel<AEP::SDGPSSM<Scalar>>(fileName, model);

		delete model;
	}

	system("pause");
}

template<typename Scalar>
void LoadAndPredict(std::string path)
{
	af::dtype m_dType = CommonUtil<Scalar>::CheckDType();

	// get the Matlab .Mat file from the command line
	std::string fileNameTraining = path + "\\data\\golfswing.mat";

	// create a new reader
	MatlabIO matio;
	bool ok = matio.Open(fileNameTraining, "r");

	// read all of the variables in the file
	std::vector<MatlabIOContainer> variables;
	variables = matio.Read();

	// close the file
	matio.Close();

	af::array Y;

	// search for a variable named "im" and "gray"
	std::vector<cv::Mat> cvY;
	for (unsigned int n = 0; n < variables.size() - 3; ++n)
	{
		Y = CommonUtil<Scalar>::Join(Y, AfCv::MatToArray(variables[n].Data<cv::Mat>()));
	}

	Y = Y - af::tile(af::mean(Y), Y.dims(0));
	Y /= af::tile(af::stdev(Y), Y.dims(0));
	Y(af::isNaN(Y)) = 0.0;

	std::string file_Path = path + "\\resources\\Golfswing";

	CommonUtil<Scalar>::WriteTXT(Y(af::seq(0, 112), af::span), file_Path + "\\Y.txt", ' ');

	for (const auto& entry : fs::directory_iterator(file_Path))
	{
		af::array mx, vx, mf, vf;
		AEP::SDGPSSM<Scalar>* model= NeuralEngine::MachineLearning::LoadModel<AEP::SDGPSSM<Scalar>>(entry.path().string());

		model->GetLatents(mx, vx);
		model->PosteriorLatents(mx, vx);
		model->PredictY(mx(af::seq(0, 112), af::span), mf, vf);
		CommonUtil<Scalar>::WriteTXT(mf, entry.path().string() + ".txt", ' ');
		delete model;
	}

	system("pause");
}

 int main(int, char const*[])
 {
 #if defined(_DEBUG)
	LogReporter reporter(
		"LogReport.txt",
		Listener::LISTEN_FOR_ALL,
		Listener::LISTEN_FOR_ALL,
		Listener::LISTEN_FOR_ERROR,
		Listener::LISTEN_FOR_NOTHING);
 #endif

	Environment env;
	std::string nepath = env.GetVariable("NE_PATH");
	if (nepath == "")
	{
		LogError("You must create the environment variable NE_PATH.");
		return 0;
	}

	HWND consoleWindow = GetConsoleWindow();

	SetWindowPos(consoleWindow, 0, 10, 10, 0, 0, SWP_NOSIZE | SWP_NOZORDER);

	Golfswing_AEP_DGPSSM_Stocatic<double>(nepath);
	LoadAndPredict<double>(nepath);
	
	return 0;
}
