# Standard library
import logging
import time

# Third party
from rich.highlighter import NullHighlighter
from rich.logging import RichHandler

def T() -> float:
    """
    Returns the current time in seconds since the epoch.

    Returns:
        float: Current time in seconds.
    """
    return time.time()


def P(window: int, duration: float) -> str:
    """
    Formats a log prefix with the window number and duration.

    Args:
        window (int): The current window index.
        duration (float): The duration in seconds.

    Returns:
        str: A formatted string for log messages.
    """
    return f"[steel_blue]{window}[/steel_blue] ([grey63]{duration:.2f}s[/grey63])"


# Configure the root logger
FORMAT = "%(message)s"
logging.basicConfig(
    level=logging.INFO,
    format=FORMAT,
    datefmt="[%X]",
    handlers=[
        RichHandler(
            markup=True,  # Enable markup parsing to allow color rendering
            rich_tracebacks=True,
            highlighter=NullHighlighter(),
            show_level=False,
            show_time=True,
            show_path=False,
        )
    ],
)

# Create a logger instance
logger = logging.getLogger("core")
logger.setLevel(logging.INFO)


def debug() -> None:
    """
    Sets the logger level to DEBUG.
    """
    logger.setLevel(logging.DEBUG)


def trace() -> None:
    """
    Sets the logger level to TRACE.

    Note:
        The TRACE level is not standard in the logging module.
        You may need to add it explicitly if required.
    """
    TRACE_LEVEL_NUM = 5
    logging.addLevelName(TRACE_LEVEL_NUM, "TRACE")

    def trace_method(self, message, *args, **kws) -> None:
        if self.isEnabledFor(TRACE_LEVEL_NUM):
            self._log(TRACE_LEVEL_NUM, message, args, **kws)

    logging.Logger.trace = trace_method
    logger.setLevel(TRACE_LEVEL_NUM)


logger.setLevel(logging.INFO)
logger.propagate = True
logger.handlers.clear()
logger.addHandler(
    RichHandler(
        markup=True,
        rich_tracebacks=True,
        highlighter=NullHighlighter(),
        show_level=False,
        show_time=True,
        show_path=False,
    )
)


__all__ = [
    "logger",
    "debug",
    "trace",
]
