"""Run Off2OnRL Exp"""

import os
from typing import Tuple

os.environ["XLA_PYTHON_CLIENT_MEM_FRACTION"] = ".2"

import time

import d4rl
import gym
import ml_collections
import numpy as np
import pandas as pd
from models import Off2OnRLAgent
from tqdm import trange
from utils import (PrioritizedReplayBuffer, ReplayBuffer, get_logger,
                   load_ckpt, normalize_reward)


def normalize_rewards(replay_buffer: ReplayBuffer, env_name: str):
    if 'v2' in env_name:
        # mujoco environments
        normalize_info_df = pd.read_csv('configs/minmax_traj_reward.csv',
                                        index_col=0).set_index('env_name')
        min_traj_reward, max_traj_reward = normalize_info_df.loc[
            env_name, ['min_traj_reward', 'max_traj_reward']]
        replay_buffer.rewards = replay_buffer.rewards / (
            max_traj_reward - min_traj_reward) * 1000
    else:
        # antmaze environments
        replay_buffer.rewards -= 1.0


def eval_policy(agent, env, eval_episodes=10) -> Tuple[float, float]:
    t1 = time.time()
    avg_reward = 0.
    for _ in range(eval_episodes):
        obs, done = env.reset(), False
        while not done:
            action = agent.sample_action(obs, eval_mode=True)
            obs, reward, done, _ = env.step(action)
            avg_reward += reward
    avg_reward /= eval_episodes
    d4rl_score = env.get_normalized_score(avg_reward) * 100
    return d4rl_score, time.time() - t1


def train_and_evaluate(config: ml_collections.ConfigDict):
    start_time = time.time()
    timestamp = time.strftime("%Y%m%d_%H%M%S", time.localtime())
    if "v2" in config.env_name:
        log_dir = f"logs/exp_mujoco_baseline/off2onrl/{config.env_name}"
    else:
        log_dir = f"logs/exp_antmaze_baseline/off2onrl/{config.env_name}"
    os.makedirs(log_dir, exist_ok=True)
    exp_name = f"{config.algo}_s{config.seed}_{timestamp}"
    exp_info = f"# Running experiment for: {exp_name}_{config.env_name} #"
    print("#" * len(exp_info) + f"\n{exp_info}\n" + "#" * len(exp_info))

    logger = get_logger(f"{log_dir}/{exp_name}.log")
    logger.info(f"Exp configurations:\n{config}")

    # initialize the environment
    env = gym.make(config.env_name)
    eval_env = gym.make(config.env_name)

    obs_dim = env.observation_space.shape[0]
    act_dim = env.action_space.shape[0]
    max_action = env.action_space.high[0]

    # initialize agent
    agent = Off2OnRLAgent(obs_dim=obs_dim,
                          act_dim=act_dim,
                          max_action=max_action,
                          seed=config.seed,
                          expectile=config.expectile,
                          adv_temperature=config.adv_temperature,
                          std_temperature=config.std_temperature,
                          clip_priority=config.clip_priority,
                          max_timesteps=config.max_timesteps)

    # load checkpoint
    load_ckpt(agent, config.base_algo, config.env_name, cnt=200)
    logs = [{
        "step": 0,
        "reward": eval_policy(agent, env, config.eval_episodes)[0]
    }]

    # Off2OnRL uses 3 buffers
    offline_buffer = ReplayBuffer(obs_dim, act_dim, max_size=int(1e6))
    offline_buffer.convert_D4RL(d4rl.qlearning_dataset(env))
    normalize_rewards(offline_buffer, config.env_name)
    online_buffer = ReplayBuffer(obs_dim, act_dim, max_size=int(1e6))
    per_buffer = PrioritizedReplayBuffer(obs_dim, act_dim, max_size=int(1.5e6))
    for i in trange(offline_buffer.ptr, desc="[Add D4RL to PER]"):
        per_buffer.add(offline_buffer.observations[i],
                       offline_buffer.actions[i],
                       offline_buffer.next_observations[i],
                       offline_buffer.rewards[i],
                       1. - offline_buffer.discounts[i])
    per_buffer.max_priority = config.max_priority

    # fine-tuning
    obs, done = env.reset(), False
    episode_timesteps = 0
    for t in trange(1, config.max_timesteps + 1):
        episode_timesteps += 1
        action = agent.sample_action(obs, eval_mode=False)
        next_obs, reward, done, info = env.step(action)
        if config.base_algo == "iql" or "antmaze" in config.env_name:
            reward = normalize_reward(config.env_name, reward)
        done_bool = float(done) if "TimeLimit.truncated" not in info else 0

        online_buffer.add(obs, action, next_obs, reward, done_bool)
        per_buffer.add(obs, action, next_obs, reward, done_bool, flag=1.0)
        obs = next_obs

        if t > config.start_timesteps:
            online_batch = online_buffer.sample(config.batch_size)
            offline_batch = offline_buffer.sample(config.batch_size)
            batch = per_buffer.sample(config.batch_size)
            log_info = agent.update(batch, online_batch, offline_batch)
            per_buffer.update_priority(batch.idx, log_info["priority"])
            log_info["online_ratio"] = batch.flags.sum() / len(batch.flags)
            per_history = (per_buffer.ptr - batch.idx).mean()

        if done:
            obs, done = env.reset(), False
            episode_timesteps = 0

        if t % 2500 == 0:
            if t % config.eval_freq == 0:
                eval_reward, eval_time = eval_policy(agent, eval_env,
                                                     config.eval_episodes)
            else:
                eval_reward, eval_time = 0, 0
            if t > config.start_timesteps:
                log_info.update({
                    "step": t,
                    "reward": eval_reward,
                    "eval_time": eval_time,
                    "time": (time.time() - start_time) / 60,
                    "max_priority": per_buffer.max_priority,
                    "tree_level0": per_buffer.tree.levels[0].item(),
                    "per_history": per_history,
                    "buffer_size": per_buffer.size,
                    "buffer_ptr": per_buffer.ptr
                })
                if t % config.eval_freq == 0:
                    _ = log_info.pop("priority")
                    logs.append(log_info)
                agent.logger(t, logger, log_info)
            else:
                logs.append({"step": t, "reward": eval_reward})
                logger.info(
                    f"\n[#Step {t}] eval_reward: {eval_reward:.2f}, eval_time: {eval_time:.2f}\n"
                )

    log_df = pd.DataFrame(logs)
    log_df.to_csv(f"{log_dir}/{exp_name}.csv")
