"""Run PER Exp"""

import os
from typing import Tuple

os.environ["XLA_PYTHON_CLIENT_MEM_FRACTION"] = ".2"

import time

import d4rl
import gym
import ml_collections
import numpy as np
import pandas as pd
from models import PERIQLAgent
from tqdm import trange
from utils import (ReplayBuffer, NstepPrioritizedReplayBuffer, get_logger, load_ckpt, save_traj_data,
                   normalize_reward)


def eval_policy(agent, env, eval_episodes=10) -> Tuple[float, float]:
    t1 = time.time()
    avg_reward = 0.
    for _ in range(eval_episodes):
        obs, done = env.reset(), False
        while not done:
            action = agent.sample_action(obs, eval_mode=True)
            obs, reward, done, _ = env.step(action)
            avg_reward += reward
    avg_reward /= eval_episodes
    d4rl_score = env.get_normalized_score(avg_reward) * 100
    return d4rl_score, time.time() - t1


def train_and_evaluate(config: ml_collections.ConfigDict):
    start_time = time.time()
    timestamp = time.strftime("%Y%m%d_%H%M%S", time.localtime())
    if "v2" in config.env_name:
        log_dir = f"logs/exp_mujoco_sample/per/{config.env_name}"
    else:
        log_dir = f"logs/exp_antmaze_sample/per/{config.env_name}"
    os.makedirs(log_dir, exist_ok=True)
    exp_name = f"per_s{config.seed}_{timestamp}"
    exp_info = f"# Running experiment for: {exp_name}_{config.env_name} #"
    print("#" * len(exp_info) + f"\n{exp_info}\n" + "#" * len(exp_info))

    logger = get_logger(f"{log_dir}/{exp_name}.log")
    logger.info(f"Exp configurations:\n{config}")

    # initialize the environment
    env = gym.make(config.env_name)
    eval_env = gym.make(config.env_name)

    obs_dim = env.observation_space.shape[0]
    act_dim = env.action_space.shape[0]
    max_action = env.action_space.high[0]

    # initialize agent
    agent = PERIQLAgent(obs_dim=obs_dim,
                        act_dim=act_dim,
                        max_action=max_action,
                        seed=config.seed,
                     expectile=config.expectile,
                     adv_temperature=config.adv_temperature,
                     std_temperature=config.std_temperature,
                     max_timesteps=config.max_timesteps)

    # load checkpoint
    load_ckpt(agent, config.base_algo, config.env_name, cnt=200)
    logs = [{
        "step": 0,
        "reward": eval_policy(agent, env, config.eval_episodes)[0]
    }]

    # use offline buffer or not
    replay_buffer = NstepPrioritizedReplayBuffer(obs_dim,
                                                 act_dim,
                                                 max_size=int(2e6),
                                                 anneal_step=config.max_timesteps)
    if config.seed == 0:
        save_traj_data(config.env_name)

    dataset = np.load(f"saved_buffers/{config.env_name}/trajs.npz")
    ds_observations = dataset["observations"]
    ds_actions = dataset["actions"]
    ds_next_observations = dataset["next_observations"]
    ds_rewards = dataset["rewards"]
    ds_terminals = dataset["terminals"]
    ds_timeouts = dataset["timeouts"]

    for i in trange(len(dataset["rewards"]), desc="[Load tras to PER]"):
        replay_buffer.add(ds_observations[i],
                          ds_actions[i],
                          ds_next_observations[i],
                          normalize_reward(config.env_name, ds_rewards[i]),
                          float(ds_terminals[i]),
                          float(ds_timeouts[i]))

    # fine-tuning
    obs, done = env.reset(), False
    episode_timesteps = 0
    for t in trange(1, config.max_timesteps + 1):
        episode_timesteps += 1
        action = agent.sample_action(obs, eval_mode=False)
        next_obs, reward, done, info = env.step(action)
        if config.base_algo == "iql":
            reward = normalize_reward(config.env_name, reward)
        timeout = "TimeLimit.truncated" in info
        done_bool = float(done) if not timeout else 0

        replay_buffer.add(obs, action, next_obs, reward, done_bool, timeout, flag=1.0)
        obs = next_obs

        if t > config.start_timesteps:
            batch = replay_buffer.sample(config.batch_size)
            log_info = agent.update(batch)
            log_info["online_ratio"] = batch.flags.sum() / len(batch.flags)
            replay_buffer.update_priority(batch.idx, log_info["priority"])
            sample_age = (replay_buffer.ptr - batch.idx).mean()
        if done:
            obs, done = env.reset(), False
            episode_timesteps = 0

        if t % config.eval_freq == 0:
            eval_reward, eval_time = eval_policy(agent, eval_env,
                                                 config.eval_episodes)
            if t > config.start_timesteps:
                log_info.update({
                    "step": t,
                    "reward": eval_reward,
                    "eval_time": eval_time,
                    "time": (time.time() - start_time) / 60,
                    "buffer_size": replay_buffer.size,
                    "buffer_ptr": replay_buffer.ptr,
                    "sample_age": sample_age,
                    "batch_weights": batch.weights.mean(),
                    "batch_weights_max": batch.weights.max(),
                    "batch_weights_min": batch.weights.min(),
                })
                logs.append(log_info)
                agent.logger(t, logger, log_info)
            else:
                logs.append({"step": t, "reward": eval_reward})
                logger.info(
                    f"\n[#Step {t}] eval_reward: {eval_reward:.2f}, eval_time: {eval_time:.2f}\n"
                )

    log_df = pd.DataFrame(logs)
    log_df.to_csv(f"{log_dir}/{exp_name}.csv")