"""Train offline agent"""

import os
from typing import Tuple

os.environ["XLA_PYTHON_CLIENT_MEM_FRACTION"] = ".2"

import time

import d4rl
import gym
import ml_collections
import numpy as np
import pandas as pd
from models import IQLAgent
from tqdm import trange
from utils import ReplayBuffer, get_logger, load_ckpt, normalize_reward


def normalize_rewards(replay_buffer: ReplayBuffer, env_name: str):
    if 'v2' in env_name:
        # mujoco environments
        normalize_info_df = pd.read_csv('configs/minmax_traj_reward.csv',
                                        index_col=0).set_index('env_name')
        min_traj_reward, max_traj_reward = normalize_info_df.loc[
            env_name, ['min_traj_reward', 'max_traj_reward']]
        replay_buffer.rewards = replay_buffer.rewards / (
            max_traj_reward - min_traj_reward) * 1000
    else:
        # antmaze environments
        replay_buffer.rewards -= 1.0


def eval_policy(agent, env, eval_episodes=10) -> Tuple[float, float]:
    t1 = time.time()
    avg_reward = 0.
    for _ in range(eval_episodes):
        obs, done = env.reset(), False
        while not done:
            action = agent.sample_action(obs, eval_mode=True)
            obs, reward, done, _ = env.step(action)
            avg_reward += reward
    avg_reward /= eval_episodes
    d4rl_score = env.get_normalized_score(avg_reward) * 100
    return d4rl_score, time.time() - t1


def train_and_evaluate(config: ml_collections.ConfigDict):
    start_time = time.time()
    timestamp = time.strftime("%Y%m%d_%H%M%S", time.localtime())
    if "v2" in config.env_name:
        log_dir = f"logs/exp_mujoco_iql/{config.env_name}"
    else:
        log_dir = f"logs/exp_antmaze_iql/{config.env_name}"
    os.makedirs(log_dir, exist_ok=True)
    exp_name = f"{config.env_name}_s{config.seed}_{timestamp}"
    exp_info = f"# Running experiment for: {exp_name}_{config.env_name} #"
    print("#" * len(exp_info) + f"\n{exp_info}\n" + "#" * len(exp_info))

    logger = get_logger(f"{log_dir}/{exp_name}.log")
    logger.info(f"Exp configurations:\n{config}")

    # initialize the environment
    env = gym.make(config.env_name)
    eval_env = gym.make(config.env_name)

    obs_dim = env.observation_space.shape[0]
    act_dim = env.action_space.shape[0]
    max_action = env.action_space.high[0]

    # initialize agent
    agent = IQLAgent(obs_dim=obs_dim,
                     act_dim=act_dim,
                     max_action=max_action,
                     seed=config.seed,
                     expectile=config.expectile,
                     adv_temperature=config.adv_temperature,
                     max_timesteps=config.max_timesteps)

    # replay buffer
    replay_buffer = ReplayBuffer(obs_dim, act_dim)
    replay_buffer.convert_D4RL(d4rl.qlearning_dataset(env))
    normalize_rewards(replay_buffer, config.env_name)
    logs = [{
        "step": 0,
        "reward": eval_policy(agent, env, config.eval_episodes)[0]
    }]
    ckpt_dir = f"saved_ckpts/iql/{config.env_name}"
    os.makedirs(ckpt_dir, exist_ok=True)

    for t in trange(1, int(1e6) + 1):
        batch = replay_buffer.sample(config.batch_size)
        log_info = agent.update(batch)

        # Save checkpoints
        if (t >= int(9.8e5) and t % 5000 == 0):
            agent.save(f"{ckpt_dir}", t // 5000)

        if t % config.eval_freq == 0:
            eval_reward, eval_time = eval_policy(agent, env,
                                                 config.eval_episodes)
            log_info.update({
                "step": t,
                "reward": eval_reward,
                "eval_time": eval_time,
                "time": (time.time() - start_time) / 60
            })
            logs.append(log_info)
            agent.logger(t, logger, log_info)

    log_df = pd.DataFrame(logs)
    log_df.to_csv(f"{config.log_dir}/{config.env_name}/{exp_name}.csv")
    final_reward = log_df["reward"].iloc[-10:].mean()
    logger.info(f"\nAvg eval reward = {final_reward:.2f}\n")
