"""
Below is the copyright notice from Google.

Please also follow this license when you modify or distribute the code.
"""

"""
Copyright 2023 Google LLC

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    https://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
"""
from PIL import Image
import argparse
from tqdm.auto import tqdm
from diffusers import DiffusionPipeline, DDIMScheduler
from transformers import AutoTokenizer
import torch
from torch import autocast
import os

model_path = "outputs/D2Edit"
pipeline = DiffusionPipeline.from_pretrained(
            "models/models_runwayml_stable_diffusion_v1_5",   #/stabilityai/stable-diffusion-2-1-base
            torch_dtype=torch.float16,
        )#
pipeline.load_lora_weights(model_path)

token_embedding_path = os.path.join(model_path, 'token_embedding.pth')
token_embedding_state_dict = torch.load(token_embedding_path)
pipeline.text_encoder.get_input_embeddings().weight.data = \
token_embedding_state_dict['weight'].type(torch.float16)

pipeline.tokenizer = AutoTokenizer.from_pretrained(
            os.path.join(model_path, 'tokenizer'), 
            use_fast=False
        )

pipeline.scheduler = DDIMScheduler(
            beta_start=0.00085,
            beta_end=0.012,
            beta_schedule="scaled_linear",
            clip_sample=False,
            set_alpha_to_one=False,
        )
NUM_DDIM_STEPS = 50
GUIDANCE_SCALE = 5
pipeline.enable_vae_slicing()
pipeline.to("cuda")
MAX_NUM_WORDS = 77
device = torch.device('cuda') if torch.cuda.is_available() else torch.device('cpu')


def make_concept_projector(tokenizer, text_encoder, unet, scheduler, vae, prompt_plus, prompt_minus, prompt_perp):
    """
    Implement "concept projection" and return a function `_concept_proj`

    Args:
        prompt_plus, prompt_minus, prompt_perp: they are as defined in this paper
                (Algorithm 2, corresponding to \gamma_{+}, \gamma_{-} and \gamma_3)
    """

    def _concept_proj(tokenizer,text_encoder, unet, scheduler, vae, prompt, latents, num_inference_steps=50, guidance_scale=10):
        """
        Returns an image

        Args:
            prompt: it's as defined in this paper
                    (Algorithm 2, corresponding to \gamma_{1})
            latents: Pre-generated noisy latents, sampled from a Gaussian distribution,
                    to be used as inputs for image generation.
            num_inference_steps: The number of denoising steps. More denoising steps usually lead to a higher quality image at the
                    expense of slower inference.
            guidance_scale: Guidance scale as defined in [Classifier-Free Diffusion Guidance](https://arxiv.org/abs/2207.12598).
                    `guidance_scale` is defined as `w` of equation 2
        """
        max_length = tokenizer.model_max_length
        batch_size = latents.size()[0]
        uncond_input = tokenizer(
        [""] * batch_size, padding="max_length", max_length=max_length, return_tensors="pt"
        )
        uncond_embeddings = text_encoder(uncond_input.input_ids.to("cuda"))[0]

        text_input_tmp = tokenizer(prompt * batch_size, padding="max_length", max_length=max_length, return_tensors="pt")
        emb0 = text_encoder(text_input_tmp.input_ids.to("cuda"))[0]
        text_input_tmp = tokenizer(prompt_plus * batch_size, padding="max_length", max_length=max_length, return_tensors="pt")
        emb_z0 = text_encoder(text_input_tmp.input_ids.to("cuda"))[0]
        text_input_tmp = tokenizer(prompt_minus * batch_size, padding="max_length", max_length=max_length, return_tensors="pt")
        emb_z1 = text_encoder(text_input_tmp.input_ids.to("cuda"))[0]
        text_input_tmp = tokenizer(prompt_perp * batch_size, padding="max_length", max_length=max_length, return_tensors="pt")
        emb_z_target = text_encoder(text_input_tmp.input_ids.to("cuda"))[0]

        # latents = latents * scheduler.init_noise_sigma
        latents = latents.to(torch.float16).to("cuda")
        scheduler.set_timesteps(num_inference_steps)

        with autocast("cuda"):
            for i, t in tqdm(enumerate(scheduler.timesteps)):
                with torch.no_grad():
                    latent_model_input = torch.cat([latents] * 5)
                   
                    latent_model_input = scheduler.scale_model_input(latent_model_input, t)
                   
                    noise_pred = unet(latent_model_input, t,
                                    encoder_hidden_states=torch.cat([uncond_embeddings, emb0, emb_z0, emb_z1, emb_z_target])).sample
                    
                    noise_pred_uncond, noise_pred_text0, noise_pred_text_z0, noise_pred_text_z1, noise_pred_text_z_target = noise_pred.chunk(5)

                    ## score difference
                    noise_tmp = noise_pred_text0 - noise_pred_text_z_target
                    ## Z direction
                    u = noise_pred_text_z1 - noise_pred_text_z0 
                    u /= torch.sqrt((u**2).sum())


                    ## project out Z direction
                    noise_pred_text0 -=2 *(noise_tmp * u).sum() * u

                    noise_pred = noise_pred_uncond + guidance_scale * (noise_pred_text0 - noise_pred_uncond)
                    latents = scheduler.step(noise_pred, t, latents).prev_sample

        with torch.no_grad():
            latents = 1 / 0.18215 * latents
            image = vae.decode(latents).sample

        image = (image / 2 + 0.5).clamp(0, 1)
        image = image.detach().cpu().permute(0, 2, 3, 1).numpy()
        images = (image * 255).round().astype("uint8")

        return images

    return _concept_proj

def image_grid(imgs, rows, cols):
    w, h = imgs[0].size
    grid = Image.new('RGB', size=(cols*w, rows*h))
    grid_w, grid_h = grid.size

    for i, img in enumerate(imgs):
        grid.paste(img, box=(i%cols*w, i//cols*h))
    # save image
    grid.save("result.jpg")
    return grid


class D2EditInference:
    def __init__(self):
        self._parse_args()
        # self._load_pipeline()

    def _parse_args(self):
        parser = argparse.ArgumentParser()
        
        parser.add_argument("--prompt", type=str, required=True)
        parser.add_argument("--output_path", type=str, required=True)
        parser.add_argument("--device", type=str, default="cuda")
        self.args = parser.parse_args()

    

    @torch.no_grad()
    def infer_and_save(self, prompts):
        ims_proj = []
        
        latents = torch.randn([1,4,64,64], device=device).to(device).to(torch.float16) 
        prompt_source = "<v0> with sleeveless t-shirt." 
        prompt_plus ="a long-sleeved t-shirt" 
        prompt_minus ="a short-sleeved t-shirt" 

        sample_projected_concept = make_concept_projector(pipeline.tokenizer, pipeline.text_encoder, pipeline.unet, pipeline.scheduler, pipeline.vae, prompt_plus, prompt_minus, prompts)
        im_pj = sample_projected_concept(pipeline.tokenizer, pipeline.text_encoder, pipeline.unet, pipeline.scheduler, pipeline.vae, prompt_source, latents, num_inference_steps=NUM_DDIM_STEPS, guidance_scale=GUIDANCE_SCALE)
        ims_proj.append(Image.fromarray(im_pj[0]))
        image_grid(ims_proj, rows=1, cols=1)
       
        print(f"The genearated image is saved to: {self.args.output_path}")


if __name__ == "__main__":
    inference = D2EditInference()
    inference.infer_and_save(
        prompts=[inference.args.prompt]
    )
