# coding=utf-8
# Copyright 2020 The Google Research Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""All functions related to loss computation and optimization.
"""

from pickle import FALSE
import torch
import torch.optim as optim
import numpy as np
from models import utils as mutils
from sde_lib import VESDE, VPSDE
import logging
import torch
from matplotlib import pyplot as plt

def get_optimizer(config, params, lr):

  if config.optim.optimizer == 'Adam':
    optimizer = optim.Adam(params, lr=lr, betas=(config.optim.beta1, 0.999), eps=config.optim.eps,
                           weight_decay=config.optim.weight_decay)
  else:
    raise NotImplementedError(
      f'Optimizer {config.optim.optimizer} not supported yet!')

  return optimizer


def optimization_manager(config):

  def optimize_fn(optimizer, params, step, lr=config.optim.lr,
                  warmup=config.optim.warmup,
                  grad_clip=config.optim.grad_clip):
    if warmup > 0:
      for g in optimizer.param_groups:
        g['lr'] = lr * np.minimum((step+1) / warmup, 1.0)
    if grad_clip >= 0:
      torch.nn.utils.clip_grad_norm_(params, max_norm=grad_clip)
    optimizer.step()
  return optimize_fn


def get_sde_loss_fn(sde, train, reduce_mean=True, continuous=True, likelihood_weighting=True, eps=1e-5):
  reduce_op = torch.mean if reduce_mean else lambda *args, **kwargs: 0.5 * torch.sum(*args, **kwargs)

  def loss_fn(model, batch):
    score_fn = mutils.get_score_fn(sde, model, train=train, continuous=continuous)
    t = torch.rand(batch.shape[0], device=batch.device) * (sde.T - eps) + eps
    z = torch.randn_like(batch)
    mean, std = sde.marginal_prob(batch, t)
    perturbed_data = mean + std[:, None] * z

    score = score_fn(perturbed_data, t)
    
    if not likelihood_weighting:
      losses = torch.square(score * std[:, None] + z) 
      losses = reduce_op(losses.reshape(losses.shape[0], -1), dim=-1)

    else:
      g2 = sde.sde(torch.zeros_like(batch), t)[1] ** 2
      losses = torch.square(score + z / std[:, None])
      losses = reduce_op(losses.reshape(losses.shape[0], -1), dim=-1) * g2

    # loss = torch.mean(losses)
    return losses

  return loss_fn


def get_smld_loss_fn(vesde, train, reduce_mean=False):
  assert isinstance(vesde, VESDE), "SMLD training only works for VESDEs."

  smld_sigma_array = torch.flip(vesde.discrete_sigmas, dims=(0,))
  reduce_op = torch.mean if reduce_mean else lambda *args, **kwargs: 0.5 * torch.sum(*args, **kwargs)

  def loss_fn(model, batch):
    model_fn = mutils.get_model_fn(model, train=train)
    labels = torch.randint(0, vesde.N, (batch.shape[0],), device=batch.device)
    sigmas = smld_sigma_array.to(batch.device)[labels]
    noise = torch.randn_like(batch) * sigmas[:, None]
    perturbed_data = noise + batch
    score = model_fn(perturbed_data, labels)
    target = -noise / (sigmas ** 2)[:, None]
    losses = torch.square(score - target)
    losses = reduce_op(losses.reshape(losses.shape[0], -1), dim=-1) * sigmas ** 2
    loss = torch.mean(losses)
    return loss

  return loss_fn


def get_ddpm_loss_fn(vpsde, train, reduce_mean=True):
  assert isinstance(vpsde, VPSDE), "DDPM training only works for VPSDEs."

  reduce_op = torch.mean if reduce_mean else lambda *args, **kwargs: 0.5 * torch.sum(*args, **kwargs)

  def loss_fn(model, batch):
    model_fn = mutils.get_model_fn(model, train=train)
    labels = torch.randint(0, vpsde.N, (batch.shape[0],), device=batch.device)
    sqrt_alphas_cumprod = vpsde.sqrt_alphas_cumprod.to(batch.device)
    sqrt_1m_alphas_cumprod = vpsde.sqrt_1m_alphas_cumprod.to(batch.device)
    noise = torch.randn_like(batch)
    perturbed_data = sqrt_alphas_cumprod[labels, None] * batch + \
                     sqrt_1m_alphas_cumprod[labels, None] * noise
    score = model_fn(perturbed_data, labels)
    losses = torch.square(score - noise)
    losses = reduce_op(losses.reshape(losses.shape[0], -1), dim=-1)
    loss = torch.mean(losses)
    return loss

  return loss_fn


def get_step_fn(sde, train, range, spcl, optimize_fn=None, reduce_mean=False, continuous=True, likelihood_weighting=False, writer=None):
  if continuous:
    loss_fn = get_sde_loss_fn(sde, train, reduce_mean=reduce_mean,
                              continuous=True, likelihood_weighting=likelihood_weighting)
  else:
    assert not likelihood_weighting, "Likelihood weighting is not supported for original SMLD/DDPM training."
    if isinstance(sde, VESDE):
      loss_fn = get_smld_loss_fn(sde, train, reduce_mean=reduce_mean)
    elif isinstance(sde, VPSDE):
      loss_fn = get_ddpm_loss_fn(sde, train, reduce_mean=reduce_mean)
    else:
      raise ValueError(f"Discrete training for {sde.__class__.__name__} is not recommended.")



  def step_fn(state, batch, prob, belong, id_):
    model = state['model']
    if train:
      optimizer = state['optimizer']
      optimizer.zero_grad()

      losses = loss_fn(model, batch)

      if spcl:
        v_scaled = min_max_scaling(prob, belong.to(bool), state['epoch'], range)
        loss = torch.mean(v_scaled*losses)

        if state['epoch'] % 1000 == 0:
          writer.add_histogram(f"weight_hist/{id_}", v_scaled.detach().cpu(), state['epoch'])

          loss_ = losses.detach().cpu().numpy()
          density_ = prob.detach().cpu().numpy()

          fig = plt.figure()
          loss_density = plt.scatter(loss_, density_)

          writer.add_figure(f"loss_density_scatter/{id_}", fig, state['epoch'])

      else:
        # loss = torch.mean(losses * min_max_scaling(prob, scale=(min, 1)))
        loss = torch.mean(losses)
        v_scaled = torch.zeros_like(loss)
        
      loss.backward()
      optimize_fn(optimizer, model.parameters(), step=state['step'])
      state['step'] += 1
      state['ema'].update(model.parameters())

      return loss, v_scaled      


    else:
      with torch.no_grad():
        ema = state['ema']
        ema.store(model.parameters())
        ema.copy_to(model.parameters())
        losses, score = loss_fn(model, batch)
        ema.restore(model.parameters())
        loss = torch.mean(losses)

      return loss

  return step_fn

def min_max_scaling(factor, belong, epoch, scale):
  std = (factor - scale[0]) / (scale[1] - scale[0])
  std[std < 0] = 0

  temp = torch.cat([std, torch.tensor([1., 0.]).to(std.device)])

  log_scaled = (temp * (np.e - 1.) + 1.)[:len(std)]
  log_step = (np.e - log_scaled[belong]) / 10000
  v_log = log_scaled.clone() 
  v_log[belong] = log_scaled[belong] + log_step * epoch

  v = (v_log - 1.) / (np.e - 1.)
  return v.clip_(0., 1.)

