""" Packages import """
import numpy as np
from numba import jit
from scipy.signal import convolve
import bottleneck as bn
import scipy.stats as sc


@jit(nopython=True)
def rd_argmax(vector):
    """
    Compute random among eligible maximum indices
    :param vector: np array
    :return: int, random index among eligible maximum indices
    """
    m = np.amax(vector)
    indices = np.nonzero(vector == m)[0]
    return np.random.choice(indices)

# def rd_argmax(vector):
#     """
#     Compute random among eligible maximum indices
#     :param vector: np.array
#     :return: int, random index among eligible maximum indices
#     """
#     m = np.amax(vector)
#     indices = np.nonzero(vector == m)[0]
#     return np.random.choice(indices)


@jit(nopython=True)
def rd_choice(vec, size):
    return np.random.choice(vec, size=size, replace=False)

#
# @jit(nopython=True)
# def hypergeom_sample(s1, n1, n2):
#     return np.random.hypergeometric(s1, n1 - s1, nsample=n2)


def rollavg_convolve(a, n):
    """
    :param a: array
    :param n: window of the rolling average
    :return:
    """
    return convolve(a, np.ones(n, dtype='float')/n, 'same')[n//2:-n//2+1]


def rollavg_bottlneck(a, n):
    """
    :param a: array
    :param n: window of the rolling average
    :return:
    """
    return bn.move_mean(a, window=n, min_count=n)


@jit(nopython=True)
def get_leader(Na, Sa, l_prev):
    """
    :param Na: Number of pulls of each arm (array)
    :param Sa: Sum of rewards of each arm (array)
    :param l_prev: Leader of the previous round
    :return: New leader for duelling algorithms
    """
    m = np.amax(Na)
    n_argmax = np.nonzero(Na == m)[0]
    if n_argmax.shape[0] == 1:
        l = n_argmax[0]
        return l
    else:
        s_max = Sa[n_argmax].max()
        s_argmax = np.nonzero(Sa[n_argmax] == s_max)[0]
        if np.nonzero(n_argmax[s_argmax] == l_prev)[0].shape[0] > 0:
            return l_prev
    return n_argmax[np.random.choice(s_argmax)]


def convert_tg_mean(mu, scale, step=1e-7):
    X = np.arange(0, 1, step)
    return (X * sc.norm.pdf(X, loc=mu, scale=scale)).mean() + 1 - sc.norm.cdf(1, loc=mu, scale=scale)
