from datetime import datetime
import json
from pathlib import Path
import shutil

import numpy as np

results_dir = Path(__file__).parent / "results"
backup_dir = results_dir / ".backup"


def write_result(
    model: str, dataset: str, div_order: str, rmse=[], time=[], log_like=[], config={}
):
    model_dir = results_dir / model
    dataset_dir = model_dir / dataset
    dataset_dir.mkdir(exist_ok=True, parents=True)

    res_file = model_dir / f"{dataset}.json"
    existing = {}
    if res_file.exists():
        existing = json.load(res_file.open())
        if div_order in existing:
            backup(model, dataset)

    mfs = {
        measure: dataset_dir / f"{div_order}_{measure}.txt"
        for measure in ["rmse", "time", "log_like", "config"]
    }

    np.array(rmse).tofile(mfs["rmse"].resolve(), sep=",")
    np.array(time).tofile(mfs["time"].resolve(), sep=",")
    np.array(log_like).tofile(mfs["log_like"].resolve(), sep=",")
    json.dump(config, mfs["config"].open("w"))

    json.dump(
        {
            div_order: {
                "config": config,
                **{name: str(fp.relative_to(results_dir)) for name, fp in mfs.items()},
            },
            **existing,
        },
        res_file.open("w"),
    )


def backup(model, dataset):
    backup_mdir = backup_dir / model / dataset
    backup_mdir.mkdir(exist_ok=True, parents=True)

    res_file = results_dir / model / f"{dataset}.json"
    res = json.load(res_file.open())

    backup_res = {}
    ts = datetime.today().strftime("%d%m%Y%H%M%S")
    for order, measures in res.items():

        entry = {}
        for name, loc in measures.items():
            source_mf = results_dir / loc
            dest_mf = backup_mdir / f"{ts}_{source_mf.name}"
            shutil.copy(source_mf, dest_mf)
            entry[name] = str(dest_mf.relative_to(backup_dir))

        backup_res[order] = entry
    json.dump(backup_res, (backup_dir / model / f"{ts}_{dataset}.json").open("w"))


def read_result(model, dataset, ts=None):
    res_file = results_dir / model / f"{dataset}.json"
    base_dir = results_dir
    if ts is not None:
        res_file = backup_dir / model / f"{ts}_{dataset}.json"
        base_dir = backup_dir

    res_file = json.load(res_file.open())

    result = {}
    for order, entries in res_file.items():
        result[order] = {
            name: json.load((base_dir / loc).open())
            if name == "config"
            else np.loadtxt((base_dir / loc).open(), delimiter=",")
            for name, loc in entries.items()
        }
    return result


def backup_ts(model, dataset):
    model_dir = backup_dir / model
    model_dir.mkdir(exist_ok=True, parents=True)
    tss = []
    for file in model_dir.iterdir():
        if file.is_file() and dataset in file.name:
            tss.append(file.stem.split("_")[0])
    return tss
