import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.optim as optim


class PPO():
    def __init__(self,
                 actor_critic,
                 clip_param,
                 ppo_epoch,
                 num_mini_batch,
                 value_loss_coef,
                 entropy_coef,
                 lr=None,
                 eps=None,
                 max_grad_norm=None,
                 use_clipped_value_loss=True):

        self.actor_critic = actor_critic

        self.clip_param = clip_param
        self.ppo_epoch = ppo_epoch
        self.num_mini_batch = num_mini_batch

        self.value_loss_coef = value_loss_coef
        self.entropy_coef = entropy_coef

        self.max_grad_norm = max_grad_norm
        self.use_clipped_value_loss = use_clipped_value_loss

        self.optimizer = optim.Adam(actor_critic.parameters(), lr=lr, eps=eps)
        
        # exploration network
        self.pred_net = None
        self.target_net = None
        self.pred_optimizer = None
        self.obs_size = None

    def update(self, rollouts):
        advantages = rollouts.returns[:-1] - rollouts.value_preds[:-1]
        advantages = (advantages - advantages.mean()) / (
            advantages.std() + 1e-5)
        '''
        for i in range(int(torch.max(rollouts.subset_idx).item())):
            # index = (rollouts.subset_idx == i).nonzero()
            index = (rollouts.subset_idx == i)
            if torch.sum(index.float()) > 1:
                # data = torch.gather(advantages, dim=0, index=index)
                data = advantages[index]
                equal = (rollouts.subset_idx == i).float()
                advantages = advantages * (1 - equal) + equal * (advantages - data.mean()) / (data.std() + 1e-5)
        '''

        value_loss_epoch = 0
        action_loss_epoch = 0
        dist_entropy_epoch = 0

        for e in range(self.ppo_epoch):
            if self.actor_critic.is_recurrent:
                data_generator = rollouts.recurrent_generator(
                    advantages, self.num_mini_batch)
            else:
                data_generator = rollouts.feed_forward_generator(
                    advantages, self.num_mini_batch)

            for sample in data_generator:
                obs_batch, recurrent_hidden_states_batch, actions_batch, \
                   value_preds_batch, return_batch, masks_batch, old_action_log_probs_batch, \
                        adv_targ = sample

                # Reshape to do in a single forward pass for all steps
                values, action_log_probs, dist_entropy, _ = self.actor_critic.evaluate_actions(
                    obs_batch, recurrent_hidden_states_batch, masks_batch,
                    actions_batch)

                ratio = torch.exp(action_log_probs -
                                  old_action_log_probs_batch)
                surr1 = ratio * adv_targ
                surr2 = torch.clamp(ratio, 1.0 - self.clip_param,
                                    1.0 + self.clip_param) * adv_targ
                action_loss = -torch.min(surr1, surr2).mean()

                if self.use_clipped_value_loss:
                    value_pred_clipped = value_preds_batch + \
                        (values - value_preds_batch).clamp(-self.clip_param, self.clip_param)
                    value_losses = (values - return_batch).pow(2)
                    value_losses_clipped = (
                        value_pred_clipped - return_batch).pow(2)
                    value_loss = 0.5 * torch.max(value_losses,
                                                 value_losses_clipped).mean()
                else:
                    value_loss = 0.5 * (return_batch - values).pow(2).mean()

                self.optimizer.zero_grad()
                (value_loss * self.value_loss_coef + action_loss -
                 dist_entropy * self.entropy_coef).backward()
                nn.utils.clip_grad_norm_(self.actor_critic.parameters(),
                                         self.max_grad_norm)
                self.optimizer.step()

                value_loss_epoch += value_loss.item()
                action_loss_epoch += action_loss.item()
                dist_entropy_epoch += dist_entropy.item()

                # Used for exploration
                if self.obs_size is not None:
                    norm_obs = self.actor_critic.base.normalizer.normalize(obs_batch)
                    pred_emb = self.pred_net(norm_obs[:, :self.obs_size].to(obs_batch.device))
                    target_emb = self.target_net(norm_obs[:, :self.obs_size].to(obs_batch.device))
                  
                    # norm_obs = actor_critic.base.normalizer.normalize(obs)
                    # pred_emb = pred_net(norm_obs.to(device))
                    # target_emb = target_net(norm_obs.to(device))
                    loss = torch.mean((target_emb - pred_emb)**2)
                    self.pred_optimizer.zero_grad()
                    loss.backward()
                    self.pred_optimizer.step()

        num_updates = self.ppo_epoch * self.num_mini_batch

        value_loss_epoch /= num_updates
        action_loss_epoch /= num_updates
        dist_entropy_epoch /= num_updates

        print(value_loss_epoch, action_loss_epoch, dist_entropy_epoch, flush=True)
        return value_loss_epoch, action_loss_epoch, dist_entropy_epoch
