import os
os.environ['OPENBLAS_NUM_THREADS'] = '1'
import argparse
from typing import Tuple

import torch
import torchvision.transforms as transforms
from torchvision.transforms import Compose

from xad.datasets.transformations import RandomGaussianNoise
from xad.datasets import GTSDB_CSETS
from xad.models.cnn import CNN32
from xad.models.resnets.concept_resnets import ConceptResNet64
from xad.models.resnets.resgan import WideResNetGenerator32, \
    WideSNResNetProjectionDiscriminator32
from xad.main.bases import default_comment, main
from xad.models.bases import ADNN, ConditionalGenerator, ConditionalDiscriminator, ConceptNN


def modify_parser(parser):
    parser.set_defaults(
        comment=default_comment(__file__),
        objective='bce',
        dataset='gtsdb',
        oe_dataset=['cifar100'],
        epochs=200,
        learning_rate=1e-3,
        weight_decay=0,
        milestones=[100, 150],
        batch_size=32,
        devices=[0],
        classes=['18+19+20+21+22+23+24+25+26+27+28+29+30+31', "38+39", "1+2", "15+32", ],
        iterations=2,
        x_discrete_anomaly_scores=3,
        # x_normal_training_only=True,
        x_batch_size=32,
        x_epochs=350,
        x_learning_rate=2e-4,
        x_milestones=[300, 325],
        x_lamb_gen=1,
        x_lamb_asc=1,
        x_lamb_cyc=100,
        x_lamb_conc=10,
        x_gen_every=5,
        # x_cluster_ncc=True,
    )


def modify_args(args: argparse.Namespace):
    args.comment = args.comment.replace("{NormCls}", "Norm-" + "-".join(args.classes))
    args.classes = [
        cset_str if cset_str not in GTSDB_CSETS else
        "+".join((str(class_id) for class_id in sorted(GTSDB_CSETS[cset_str])))
        for cset_str in args.classes
    ]
    if len(args.oe_dataset) > 0 and args.oe_dataset[0] == 'gtsdb' and isinstance(args.oe_classes, (list, tuple)):
        args.oe_classes = [
            cset_str if cset_str not in GTSDB_CSETS else
            "+".join((str(class_id) for class_id in sorted(GTSDB_CSETS[cset_str])))
            for cset_str in args.oe_classes
        ]


def get_transforms() -> Tuple[Compose, Compose]:
    train_transform = transforms.Compose([
        transforms.Resize((32, 32)),
        # transforms.RandomCrop(32, padding=1),
        # transforms.RandomHorizontalFlip(p=0.5),
        transforms.ToTensor(),
        # 'normalize'
    ])
    val_transform = Compose([
        transforms.Resize((32, 32)),
        transforms.ToTensor(),
        # 'normalize'
    ])
    return train_transform, val_transform


def get_models(args) -> Tuple[ADNN, ConditionalGenerator, ConditionalDiscriminator, ConceptNN]:
    model = CNN32(bias=args.objective not in ('dsvdd', 'mdsvdd'), clf=args.objective in ('bce', 'focal'))
    gen = WideResNetGenerator32(1024, torch.Size([args.x_discrete_anomaly_scores, args.x_concepts]))
    disc = WideSNResNetProjectionDiscriminator32(1024, torch.Size([args.x_discrete_anomaly_scores]))
    concept_classifier = ConceptResNet64(args.x_concepts)  # 64 is fine
    return model, gen, disc, concept_classifier


if __name__ == '__main__':
    main(modify_parser, get_transforms, get_models, modify_args)
