import os
os.environ['OPENBLAS_NUM_THREADS'] = '1'
from typing import Tuple

import torch
import torchvision.transforms as transforms
from torchvision.transforms import Compose

from xad.models.cnn import GenericCNN28
from xad.models.resnets.concept_resnets import ConceptResNet64
from xad.models.resnets.resgan import WideResNetGenerator32, \
    WideSNResNetProjectionDiscriminator32
from xad.main.bases import default_comment, main
from xad.models.bases import ADNN, ConditionalGenerator, ConditionalDiscriminator, ConceptNN


def modify_parser(parser):
    parser.add_argument('--cnn28-conv-layers', nargs='+', type=int, default=(4, 8, 16, 32))
    parser.add_argument('--cnn28-linear-layers', nargs='+', type=int, default=(64, 32))
    parser.add_argument('--cnn28-ksize', type=int, default=5)
    parser.add_argument('--label-smoothing', type=float, default=0.0)
    parser.set_defaults(
        comment=default_comment(__file__),
        objective='bce',
        dataset='mnist',
        oe_dataset=['emnist'],
        epochs=80,
        learning_rate=1e-4,
        weight_decay=0,
        milestones=[60, ],
        batch_size=128,
        devices=[0],
        classes=["1+7"],
        iterations=2,
        x_discrete_anomaly_scores=3,
        # x_normal_training_only=True,
        x_batch_size=64,
        x_epochs=350,
        x_learning_rate=2e-4,
        x_milestones=[300, 325],
        x_lamb_gen=1,
        x_lamb_asc=1,
        x_lamb_cyc=100,
        x_lamb_conc=10,
        x_gen_every=5,
        # x_cluster_ncc=True,
    )


def get_transforms() -> Tuple[Compose, Compose]:
    train_transform = transforms.Compose([])
    val_transform = Compose([])
    return train_transform, val_transform


def get_models(args) -> Tuple[ADNN, ConditionalGenerator, ConditionalDiscriminator, ConceptNN]:
    model = GenericCNN28(
        conv_layer=args.cnn28_conv_layers, fc_layer=args.cnn28_linear_layers, ksize=args.cnn28_ksize,
        bias=args.objective not in ('dsvdd', 'mdsvdd'), clf=args.objective in ('bce', 'focal')
    )
    gen = WideResNetGenerator32(1024, torch.Size([args.x_discrete_anomaly_scores, args.x_concepts]), grayscale=True)
    disc = WideSNResNetProjectionDiscriminator32(1024, torch.Size([args.x_discrete_anomaly_scores]), grayscale=True)
    concept_classifier = ConceptResNet64(args.x_concepts, grayscale=True)  # 64 is fine
    return model, gen, disc, concept_classifier


if __name__ == '__main__':
    main(modify_parser, get_transforms, get_models)
