import torch.nn as nn

from torchvision.models import wide_resnet50_2
from xad.models.cbam import CBAM
from xad.models.bases import ADNN
from torch.nn import init


class WideResNet50Pretrained(ADNN):
    def __init__(self, rep_dim=256, clf=False):
        self.clf = clf
        super().__init__()
        self.model = wide_resnet50_2(pretrained=True)
        del self.model.fc
        self.model.fc = nn.Linear(2048, rep_dim)
        if self.clf:
            self.linear = nn.Linear(rep_dim, 1)

    def forward(self, x, return_encoding=False):
        fx = self.model(x)
        if self.clf:
            logits = self.linear(x)
        else:
            logits = fx
            if return_encoding:
                raise NotImplementedError()
        if return_encoding:
            return logits, fx
        else:
            return logits


# Credits to: https://github.com/hendrycks/ss-ood
class WideResNet(ADNN):

    def __init__(self, rep_dim=256, clf=False):
        super().__init__()
        self.inplanes = 64
        self.clf = clf

        self.rep_dim = rep_dim
        att_type = 'CBAM'
        layers = [2, 2, 2, 2]

        self.conv1 = nn.Conv2d(3, 64, kernel_size=7, stride=2, padding=3, bias=False)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        self.avgpool = nn.AvgPool2d(7)
        self.bn1 = nn.BatchNorm2d(64)
        self.relu = nn.ReLU(inplace=True)

        self.bam1, self.bam2, self.bam3 = None, None, None

        self.layer1 = self._make_layer(BasicBlock, 64, layers[0], att_type=att_type)
        self.layer2 = self._make_layer(BasicBlock, 128, layers[1], stride=2, att_type=att_type)
        self.layer3 = self._make_layer(BasicBlock, 256, layers[2], stride=2, att_type=att_type)
        self.layer4 = self._make_layer(BasicBlock, 512, layers[3], stride=2, att_type=att_type)

        self.fc = nn.Linear(512 * BasicBlock.expansion, self.rep_dim)

        if self.clf:
            self.linear = nn.Linear(self.rep_dim, 1)
            init.kaiming_normal_(self.linear.weight)

        init.kaiming_normal_(self.fc.weight)
        for key in self.state_dict():
            if key.split('.')[-1] == "weight":
                if "conv" in key:
                    init.kaiming_normal_(self.state_dict()[key], mode='fan_out')
                if "bn" in key:
                    if "SpatialGate" in key:
                        self.state_dict()[key][...] = 0
                    else:
                        self.state_dict()[key][...] = 1
            elif key.split(".")[-1] == 'bias':
                self.state_dict()[key][...] = 0

    def _make_layer(self, block, planes, blocks, stride=1, att_type=None):
        downsample = None
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                nn.Conv2d(self.inplanes, planes * block.expansion,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, stride, downsample, use_cbam=att_type == 'CBAM'))
        self.inplanes = planes * block.expansion
        for i in range(1, blocks):
            layers.append(block(self.inplanes, planes, use_cbam=att_type == 'CBAM'))

        return nn.Sequential(*layers)

    def forward(self, x, return_encoding=False):
        x = x.reshape(-1, 3, 224, 224)
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x)
        if not self.bam1 is None:
            x = self.bam1(x)

        x = self.layer2(x)
        if not self.bam2 is None:
            x = self.bam2(x)

        x = self.layer3(x)
        if not self.bam3 is None:
            x = self.bam3(x)

        x = self.layer4(x)
        x = self.avgpool(x)
        x = x.reshape(x.size(0), -1)
        encoding = x
        x = self.fc(x)

        if self.clf:
            encoding = x
            logits = self.linear(x)
        else:
            logits = x

        if return_encoding:
            return logits, x
        else:
            return logits


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, inplanes, planes, stride=1, downsample=None, use_cbam=False):
        super(BasicBlock, self).__init__()
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.bn1 = nn.BatchNorm2d(planes)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = nn.BatchNorm2d(planes)
        self.downsample = downsample
        self.stride = stride

        if use_cbam:
            self.cbam = CBAM(planes, 16)
        else:
            self.cbam = None

    def forward(self, x):
        residual = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)

        if self.downsample is not None:
            residual = self.downsample(x)

        if not self.cbam is None:
            out = self.cbam(out)

        out += residual
        out = self.relu(out)

        return out


def conv3x3(in_planes, out_planes, stride=1):
    """3x3 convolution with padding"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride, padding=1, bias=False)
