from dataclasses import dataclass, field
from optparse import Option
from typing import ClassVar, Optional

@dataclass(frozen=True)
class AvalonObservation:

    QUEST_PRESET: ClassVar =   {5 : [[3,2] , [2,3,2,3,3], [1,1,1,1,1], ] , 
                                6 : [[4,2] , [2,3,4,3,4], [1,1,1,1,1],] , 
                                7 : [[4,3] , [2,3,3,4,4], [1,1,1,2,1],] , 
                                8 : [[5,3] , [3,4,4,5,5], [1,1,1,2,1],] , 
                                9 : [[6,3] , [3,4,4,5,5], [1,1,1,2,1],] , 
                                10 : [[6,4] , [3,4,4,5,5], [1,1,1,2,1],]}
    
    MAX_ROUNDS: ClassVar = 5
    PHASES: ClassVar = {0 : "Team Selection", 1 : "Team Voting", 2 : "Quest Voting", 3 : "Assassination", 4: "Discussion"}
    ROLES: ClassVar = {0 : "Merlin", 5 : "Servant", 6 : "Minion", 7 : "Assassin"}
    ROLES_TO_IS_GOOD: ClassVar = {0: True, 5: True, 6: False, 7: False}
    
    num_players: int # number of players in the game, in [5, 10]
    quest_leader_id: int # player id of the current quest leader, in [0, num_players)
    phase: int # current phase of the game, in PHASES
    quest_number: int # which quest the game is currently on, in [0, 5)
    discussion_round: int # current discussion round of the game, in [0, MAX_ROUNDS)
    current_quest_team: Optional[frozenset[int]] # team selected for the current quest if any, subset of [0, num_players)
    known_is_good: frozenset[int] # set of player ids known to be good
    known_is_evil: frozenset[int] # set of player ids known to be evil
    historical_quest_results: tuple[bool, ...] # list of results of previous quests in the game that were approved

    historical_quest_teams_discussion_and_approval_votes: tuple[tuple[frozenset[int], tuple[tuple[int, str], ...], tuple[bool]], ...] # list of teams selected previously in the game, discussion that round regarding that team in a list of (player_id, utterance) format, and the approval votes for each team

    observed_role: int # secret role of the player who observed this, in ROLES
    observed_player_id: int # player id of the player who observed this, in [0, num_players)

    def num_good_players(self) -> int:
        return self.QUEST_PRESET[self.num_players][0][0]
    
    def num_evil_players(self) -> int:
        return self.QUEST_PRESET[self.num_players][0][1]
    
    def num_players_on_quest(self, quest_number: int) -> int:
        return self.QUEST_PRESET[self.num_players][1][quest_number]
    
    def num_fails_required(self, quest_number: int) -> int:
        return self.QUEST_PRESET[self.num_players][2][quest_number]

@dataclass(frozen=True)
class AvalonHiddenState(AvalonObservation):
    roles: tuple[int, ...] # secret roles of all players
    current_team_votes: Optional[tuple[Optional[bool], ...]] = field(default=None) # current simultaneous votes for team approval, None if not yet voted
    current_quest_votes: Optional[tuple[Optional[bool], ...]] = field(default=None) # current simultaneous votes for quest success/failure, None if not yet voted
    
    def is_good_player(self, player_id: int) -> bool:
        return self.ROLES_TO_IS_GOOD[self.roles[player_id]]
    
    def known_roles(self) -> dict[int, int]:
        """Returns known roles based on the viewing perspective of the player."""
        if self.observed_role == 0:
            # Merlin knows all evil players
            return {i: self.roles[i] for i in range(self.num_players) if not self.is_good_player(i)}
        elif self.observed_role in [6, 7]:
            # Minions and Assassin know each other
            return {i: self.roles[i] for i in range(self.num_players) if self.roles[i] not in [0, 5]}
        else:
            # Servants know nothing
            return {}
    