from dataclasses import dataclass
from operator import is_
from typing import Optional

from responses import target

from src.searchlight.headers import *


@dataclass(frozen=True)
class ObservedState:
    is_clue_giver: bool # whether the player observing the state is the guesser or the clue giver
    target_word: Optional[str] # the target word that the guesser is trying to guess. None if is_guesser is False
    taboo_words: Optional[tuple[str, ...]] # the taboo words that the clue giver must avoid. None if is_guesser is False
    turn: int # the turn number in [0, 5)
    guesses: tuple[str, ...] # the guesses made by the guesser so far, same length as turn
    clues: tuple[str, ...] # the clues given by the clue giver so far, same length as turn


@dataclass(frozen=True)
class HiddenState:
    target_word: str # the target word that the guesser is trying to guess
    taboo_words: tuple[str, ...] # the taboo words that the clue giver must avoid
    turn: int # the turn number in [0, 5)
    guesses: tuple[str, ...] # the guesses made by the guesser so far, same length as turn
    clues: tuple[str, ...] # the clues given by the clue giver so far, same length as turn
    is_guesser_turn: bool # whether it is the guesser's turn to guess the target word or the clue giver's turn to give a clue

@dataclass(frozen=True)
class TabooForwardTransitor(ForwardTransitor):

    def _transition(self, state: HiddenState, action: str, actor: int) ->tuple[HiddenState, dict[int, float]]: # type: ignore
        if state.is_guesser_turn:
            assert actor == 1
            new_guesses = state.guesses + (action,)
            new_turn = state.turn + 1
            new_is_guesser_turn = False
            return HiddenState(state.target_word, state.taboo_words, new_turn, new_guesses, state.clues, new_is_guesser_turn), {0: 0.0, 1: 0.0}
        else:
            assert actor == 0
            new_clues = state.clues + (action,)
            new_turn = state.turn
            new_is_guesser_turn = True
            return HiddenState(state.target_word, state.taboo_words, new_turn, state.guesses, new_clues, new_is_guesser_turn), {0: 0.0, 1: 0.0}
        
    def 