import numpy as np
from gym import utils
from gym.envs.mujoco import mujoco_env

class HopperEnv(mujoco_env.MujocoEnv, utils.EzPickle):
    def __init__(self, backward=False, task_one_hot_id=None):
        self.backward = backward
        self.task_one_hot_id = task_one_hot_id
        mujoco_env.MujocoEnv.__init__(self, 'hopper.xml', 4)
        utils.EzPickle.__init__(self)

    def step(self, a):
        if not hasattr(self, "init_zpos"):
            self.init_zpos = self.sim.data.get_body_xpos('torso')[2]
        posbefore = self.sim.data.qpos[0]
        self.do_simulation(a, self.frame_skip)
        posafter, height, ang = self.sim.data.qpos[0:3]
        alive_bonus = 1.0
        if not self.backward:
            reward_run = ((posafter - posbefore) / self.dt)
        else:
            reward_run = -((posafter - posbefore) / self.dt)
        reward_ctrl = -1e-3 * np.square(a).sum()
        reward_jump = 10.0*(height - self.init_zpos)
        # reward = (posafter - posbefore) / self.dt
        # reward += alive_bonus
        # reward -= 1e-3 * np.square(a).sum()
        reward = alive_bonus + reward_run + reward_ctrl - np.abs(reward_jump)
        s = self.state_vector()
        done = not (np.isfinite(s).all() and (np.abs(s[2:]) < 100).all() and
                    (height > .7) and (abs(ang) < .2))
        ob = self._get_obs()
        return ob, reward, done, dict(reward_run=reward_run, reward_ctrl=reward_ctrl, reward_jump=reward_jump)

    def _get_obs(self):
        if self.task_one_hot_id is not None:
            return np.concatenate([
                self.sim.data.qpos.flat[1:],
                np.clip(self.sim.data.qvel.flat, -10, 10),
                self.task_one_hot_id
                ])
        return np.concatenate([
            self.sim.data.qpos.flat[1:],
            np.clip(self.sim.data.qvel.flat, -10, 10)
        ])

    def reset_model(self):
        qpos = self.init_qpos + self.np_random.uniform(low=-.005, high=.005, size=self.model.nq)
        qvel = self.init_qvel + self.np_random.uniform(low=-.005, high=.005, size=self.model.nv)
        self.set_state(qpos, qvel)
        self.init_zpos = self.sim.data.get_body_xpos('torso')[2]
        return self._get_obs()

    def viewer_setup(self):
        self.viewer.cam.trackbodyid = 2
        self.viewer.cam.distance = self.model.stat.extent * 0.75
        self.viewer.cam.lookat[2] = 1.15
        self.viewer.cam.elevation = -20
