import numpy as np
from gym.spaces import Box

from metaworld.envs.mujoco.sawyer_xyz.v2.sawyer_door_v2 import SawyerDoorEnvV2


class SawyerDoorCloseEnvV2(SawyerDoorEnvV2):
    def __init__(self, task_one_hot_id=None):

        goal_low = (.2, 0.65, 0.1499)
        goal_high = (.3, 0.75, 0.1501)

        super().__init__(task_one_hot_id=task_one_hot_id)

        self._last_rand_vec = np.array([0.0, 1.0, .1525])  
        self._set_task_called = True
        self.init_config = {
            'obj_init_angle': 0.3,
            'obj_init_pos': np.array([0.1, 0.95, 0.15], dtype=np.float32),
            # 'hand_init_pos': np.array([0, 0.6, 0.2], dtype=np.float32),
            'hand_init_pos': np.array([0, 0.6, 0.4], dtype=np.float32),
        }
        self.goal = np.array([0.2, 0.8, 0.15])
        self.obj_init_pos = self.init_config['obj_init_pos']
        self.obj_init_angle = self.init_config['obj_init_angle']
        self.hand_init_pos = self.init_config['hand_init_pos']

        self.goal_space = Box(np.array(goal_low), np.array(goal_high))

    def _set_hand_to_handle(self, steps=50):
        # initialize the hand near the handle
        # hand_pos = self.hand_init_pos
        # hand_pos[:-1] = self._get_pos_objects()[:-1]
        # hand_pos[0] -= 0.05
        # hand_pos = np.array([0.23146,  0.52192,  0.2189])
        hand_pos = np.array([0.3,  0.5,  0.2189])
        for _ in range(steps):
            self.data.set_mocap_pos('mocap', hand_pos)
            self.data.set_mocap_quat('mocap', np.array([1, 0, 1, 0]))
            self.do_simulation([-1, 1], self.frame_skip)

    def reset_model(self):
        self._reset_hand()
        self._target_pos = self.goal.copy()
        self.objHeight = self.data.get_geom_xpos('handle')[2]

        if self.random_init:
            obj_pos = self._get_state_rand_vec()
            self.obj_init_pos = obj_pos
            goal_pos = obj_pos.copy() + np.array([-0.2, -0.3, 0.])
            self._target_pos = goal_pos

        self.sim.model.body_pos[self.model.body_name2id('door')] = self.obj_init_pos
        self.sim.model.site_pos[self.model.site_name2id('goal')] = self._target_pos

        # keep the door open after resetting initial positions
        # print('Door initial position', self.obj_init_pos)
        # print('Handle initial position', self._get_pos_objects())
        self._set_hand_to_handle()    
        self._set_obj_xyz(-1.5708)
        # print('Door after resetting position', self.sim.model.body_pos[self.model.body_name2id('door')])
        # print('Handle after resetting position', self._get_pos_objects())  
        # print('Hand after resetting position', (self._get_site_pos('rightEndEffector')+self._get_site_pos('leftEndEffector'))/2)  
        self.maxPullDist = np.linalg.norm(self.data.get_geom_xpos('handle')[:-1] - self._target_pos[:-1])
        self.target_reward = 1000*self.maxPullDist + 1000*2

        return self._get_obs()

    def compute_reward(self, actions, obs):
        del actions
        objPos = obs[3:6]

        rightFinger, leftFinger = self._get_site_pos('rightEndEffector'), self._get_site_pos('leftEndEffector')
        fingerCOM  =  (rightFinger + leftFinger)/2

        pullGoal = self._target_pos

        pullDist = np.linalg.norm(objPos[:-1] - pullGoal[:-1])
        reachDist = np.linalg.norm(objPos - fingerCOM)
        reachRew = -reachDist

        self.reachCompleted = reachDist < 0.05

        def pullReward():
            c1 = 1000
            c2 = 0.01
            c3 = 0.001

            if self.reachCompleted:
                pullRew = 1000*(self.maxPullDist - pullDist) + c1*(np.exp(-(pullDist**2)/c2) + np.exp(-(pullDist**2)/c3))
                pullRew = max(pullRew,0)
                return pullRew
            else:
                return 0

        pullRew = pullReward()
        reward = reachRew + pullRew

        return [reward, reachDist, pullDist]
