"""
FNO code borrowed from (https://arxiv.org/pdf/2010.08895.pdf)
"""

import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
from scipy.io import savemat
from torch.nn.parameter import Parameter

import matplotlib.pyplot as plt

import operator
import sys
from functools import reduce
from functools import partial

from timeit import default_timer
from utilities3 import *

from Adam import Adam

torch.manual_seed(0)
np.random.seed(0)

torch.set_default_dtype(torch.float64)

# torch.set_default_tensor_type(torch.DoubleTensor)


################################################################
# fourier layer
################################################################
class SpectralConv2d(nn.Module):
    def __init__(self, in_channels, out_channels, modes1, modes2):
        super(SpectralConv2d, self).__init__()

        """
        2D Fourier layer. It does FFT, linear transform, and Inverse FFT.    
        """

        self.in_channels = in_channels
        self.out_channels = out_channels
        self.modes1 = modes1 #Number of Fourier modes to multiply, at most floor(N/2) + 1
        self.modes2 = modes2

        self.scale = (1 / (in_channels * out_channels))
        self.weights1 = nn.Parameter(self.scale * torch.rand(in_channels, out_channels, self.modes1, self.modes2, dtype=torch.cdouble))
        self.weights2 = nn.Parameter(self.scale * torch.rand(in_channels, out_channels, self.modes1, self.modes2, dtype=torch.cdouble))

    # Complex multiplication
    def compl_mul2d(self, input, weights):
        # (batch, in_channel, x,y ), (in_channel, out_channel, x,y) -> (batch, out_channel, x,y)
        return torch.einsum("bixy,ioxy->boxy", input, weights)

    def forward(self, x):
        batchsize = x.shape[0]
        #Compute Fourier coeffcients up to factor of e^(- something constant)
        x_ft = torch.fft.rfft2(x)

        # Multiply relevant Fourier modes
        out_ft = torch.zeros(batchsize, self.out_channels,  x.size(-2), x.size(-1)//2 + 1, dtype=torch.cdouble, device=x.device)
        out_ft[:, :, :self.modes1, :self.modes2] = \
            self.compl_mul2d(x_ft[:, :, :self.modes1, :self.modes2], self.weights1)
        out_ft[:, :, -self.modes1:, :self.modes2] = \
            self.compl_mul2d(x_ft[:, :, -self.modes1:, :self.modes2], self.weights2)

        #Return to physical space
        x = torch.fft.irfft2(out_ft, s=(x.size(-2), x.size(-1)))
        return x
    
    
class KernelCorrection1d(nn.Module):
    def __init__(self, layer):
        super(KernelCorrection1d, self).__init__()

        """
        1D Corrected Layer. It modifies the kernel to enforce a Dirichlet boundary condition.   
        """

        self.layer = layer
        
        
    def forward(self, x , bdy_left, bdy_right):
        """
        Forward layer to enforce the boundary as well as the correction terms for
            a general operator K
            
        We apply T^{1}_Dirichlet, K and T^{2}_{Dirichlet} on x
        """
        
#         tol = 10**(-4)
#         e1 = torch.zeros_like(x)
#         e1[:, :, 0, :] = 1
#         K00 = self.layer(e1)[0,0,0, :]
        
#         indices_K00_less_tol = torch.nonzero(K00 < tol)
        
#         if len(indices_K00_less_tol) != 0:
#             K00[indices_K00_less_tol] = tol
        
#         Kx = self.layer(x)
#         T1 = x.clone()
        
#         T1[:, :, 0, :] = 2*T1[:, :, 0, :] - Kx[:, :, 0, :]/K00   
#         T1 = self.layer(T1.clone())
        
#         T1[:, :, 0, :] = x[:, :, 0, :]
        
        tol = 10**(-4)
        
        e1 = torch.zeros_like(x)
        eN = torch.zeros_like(x)
        
        e1[:, :, 0, :] = 1
        eN[:, :, -1, :] = 1
        
        K00 = self.layer(e1)[0,0,0, :]
        KN0 = self.layer(e1)[0,0,-1, :]
        K0N = self.layer(eN)[0,0,0, :]
        KNN = self.layer(eN)[0,0,-1, :]
        
        indices_K00_less_tol = torch.nonzero(K00 < tol)
        indices_KN0_less_tol = torch.nonzero(KN0 < tol)
        indices_K0N_less_tol = torch.nonzero(K0N < tol)
        indices_KNN_less_tol = torch.nonzero(KNN < tol)
        
        if len(indices_K00_less_tol) != 0:
            K00[indices_K00_less_tol] = tol
            
        if len(indices_KNN_less_tol) != 0:
            KNN[indices_KNN_less_tol] = tol
                      
    
        Kx = self.layer(x)
        T2_left_x = x.clone()
        
        tilde_K00 = K00 - KN0*K0N/KNN
        
        indices_tilde_K00_less_tol = torch.nonzero(tilde_K00 < tol)
        
        if len(indices_tilde_K00_less_tol) != 0:
            tilde_K00[indices_tilde_K00_less_tol] = tol
        

        T2_left_x[:, :, 0, :] = 2*x[:, :, 0, :] - (1/tilde_K00)*(Kx[:, :, 0, :] - Kx[:, :, -1, :]*K0N/KNN + K0N*x[:, :, -1, :])

        Ky = self.layer(T2_left_x.clone())
        T2_right_y = T2_left_x.clone()
        
        T2_right_y[:, :, -1, :] = 2*T2_right_y[:, :, 0, :] - Ky[:, :, -1, :]/KNN
        
        Kv = self.layer(T2_right_y.clone())
        
        T1 = Kv.clone()
        
        T1[:, :, 0, :] = x[:, :, 0, :]
        T1[:, :, -1, :] = x[:, :, -1, :]
        
        
             
  
        return T1
    

class FNO2d(nn.Module):
    def __init__(self, modes1, modes2,  width):
        super(FNO2d, self).__init__()

        """
        The overall network. It contains 4 layers of the Fourier layer.
        1. Lift the input to the desire channel dimension by self.fc0 .
        2. 4 layers of the integral operators u' = (W + K)(u).
            W defined by self.w; K defined by self.conv .
        3. Project from the channel space to the output space by self.fc1 and self.fc2 .
        
        input: the solution of the coefficient function and locations (a(x, y), x, y)
        input shape: (batchsize, x=s, y=s, c=3)
        output: the solution 
        output shape: (batchsize, x=s, y=s, c=1)
        """

        self.modes1 = modes1
        self.modes2 = modes2
        self.width = width
        self.padding = 9 # pad the domain if input is non-periodic
        self.fc0 = nn.Linear(3, self.width).double() # input channel is 3: (a(x, y), x, y)

        self.conv0 = SpectralConv2d(self.width, self.width, self.modes1, self.modes2).double()
        self.conv1 = SpectralConv2d(self.width, self.width, self.modes1, self.modes2).double()
        self.conv2 = SpectralConv2d(self.width, self.width, self.modes1, self.modes2).double()
        self.conv3 = SpectralConv2d(self.width, self.width, self.modes1, self.modes2).double()
        
        self.kernel_correction0 = KernelCorrection1d(self.conv0).double()
        self.kernel_correction1 = KernelCorrection1d(self.conv1).double()
        self.kernel_correction2 = KernelCorrection1d(self.conv2).double()
        self.kernel_correction3 = KernelCorrection1d(self.conv3).double()
        
        self.w0 = nn.Conv2d(self.width, self.width, 1).double()
        self.w1 = nn.Conv2d(self.width, self.width, 1).double()
        self.w2 = nn.Conv2d(self.width, self.width, 1).double()
        self.w3 = nn.Conv2d(self.width, self.width, 1).double()
        
        self.w0_correction = KernelCorrection1d(self.w0).double()
        self.w1_correction = KernelCorrection1d(self.w1).double()
        self.w2_correction = KernelCorrection1d(self.w2).double()
        self.w3_correction = KernelCorrection1d(self.w3).double()

        self.fc1 = nn.Linear(self.width, 128).double()
        self.fc2 = nn.Linear(128, 1).double()

    def forward(self, x, bdy_left, bdy_right, h_bdy_right, coeffs_finite_difference_right):
        
#         bdy_left = bdy_left.reshape(x.shape[0], 1, x.shape[2])
#         bdy_right = bdy_right.reshape(x.shape[0], 1, x.shape[2])
        
#         bdy_left_padded = bdy_left
#         bdy_right_padded = bdy_right
        
#         bdy_left_padded = F.pad(bdy_left, [0,self.padding])
#         bdy_right_padded =  F.pad(bdy_right, [0,self.padding])
        
        grid = self.get_grid(x.shape, x.device)
        x = torch.cat((x, grid), dim=-1)
        x = self.fc0(x)
        x = x.permute(0, 3, 1, 2)
        x = F.pad(x, [0,self.padding, 0,self.padding])
        
        x1 = self.kernel_correction0(x, bdy_left, bdy_right)
        x2 = self.w0_correction(x, bdy_left, bdy_right)
        x = (x1 + x2)/2
        x[:, :, 1:x.shape[2]-1] = F.gelu(x[:, :, 1:x.shape[2]-1].clone())

        x1 = self.kernel_correction1(x, bdy_left, bdy_right)
        x2 = self.w1_correction(x, bdy_left, bdy_right)
        x = (x1 + x2)/2
        x[:, :, 1:x.shape[2]-1] = F.gelu(x[:, :, 1:x.shape[2]-1].clone())

        x1 = self.kernel_correction2(x, bdy_left, bdy_right)
        x2 = self.w2_correction(x, bdy_left, bdy_right)
        x = (x1 + x2)/2
        x[:, :, 1:x.shape[2]-1] = F.gelu(x[:, :, 1:x.shape[2]-1].clone())

        x1 = self.kernel_correction3(x, bdy_left, bdy_right)
        x2 = self.w3_correction(x, bdy_left, bdy_right)
        x = (x1 + x2)/2

        x = x[..., :-self.padding, :-self.padding]
        x = x.permute(0, 2, 3, 1)
        x = self.fc1(x)
        x[:, 1:x.shape[1]-1, :] = F.gelu(x[:, 1:x.shape[1]-1, :].clone())
        
        x = self.fc2(x)
        
        x[:, -1, :] = x[:, -2, :] + h_bdy_right.reshape(1, x.shape[2],1).repeat([x.shape[0],1,1]) 
        
#         x[:, -1, :] = (bdy_right.reshape(1, x.shape[2],1).repeat([x.shape[0],1,1]) - coeffs_finite_difference_right[-2]*x[:, -2, :] - coeffs_finite_difference_right[-3]*x[:, -3, :] - coeffs_finite_difference_right[-4]*x[:, -4, :])/coeffs_finite_difference_right[-1]
        x[:, 0, :] = x[:, 1, :] - h*bdy_left
        
        
        return x
    
    def get_grid(self, shape, device):
        batchsize, size_x, size_y = shape[0], shape[1], shape[2]
        gridx = torch.tensor(np.linspace(0, 1, size_x), dtype=torch.float)
        gridx = gridx.reshape(1, size_x, 1, 1).repeat([batchsize, 1, size_y, 1])
        gridy = torch.tensor(np.linspace(0, 1, size_y), dtype=torch.float)
        gridy = gridy.reshape(1, 1, size_y, 1).repeat([batchsize, size_x, 1, 1])
        return torch.cat((gridx, gridy), dim=-1).to(device)

################################################################
# configs
################################################################
# TRAIN_PATH = 'piececonst_r421_N1024_smooth1.mat'
# TEST_PATH = 'piececonst_r421_N1024_smooth2.mat'

ntrain = 1000
ntest = 200

resolution_data = 500

batch_size = 20
learning_rate = 0.001

epochs = 500
step_size = 100
gamma = 0.5

modes = 12
width = 32

sub = int(sys.argv[1]) #subsampling rate
N = 2048 // sub #total grid size divided by the subsampling rate
# N = 1500 // sub #total grid size divided by the subsampling rate
s = N

T_in = 1

T = 25



data_all = scipy.io.loadmat('data_2D_heat_neumann.mat')


x_train = data_all['a'][:ntrain,::sub,:T_in]
x_train = x_train.astype(np.float64)
x_train = torch.from_numpy(x_train)

y_train = data_all['u'][:ntrain,::sub,-T:]
y_train = y_train.astype(np.float64)
y_train = torch.from_numpy(y_train)

x_test = data_all['a'][-ntest:,::sub,:T_in]
x_test = x_test.astype(np.float64)
x_test = torch.from_numpy(x_test)

y_test = data_all['u'][-ntest:,::sub,-T:]
y_test = y_test.astype(np.float64)
y_test = torch.from_numpy(y_test)


# x_normalizer = UnitGaussianNormalizer(x_train)
# x_train = x_normalizer.encode(x_train)
# x_test = x_normalizer.encode(x_test)

# y_normalizer = UnitGaussianNormalizer(y_train)
# y_train = y_normalizer.encode(y_train)

x_train = x_train.reshape(ntrain,s,1,T_in).repeat([1,1,T,1])
print(x_train.shape)
x_test = x_test.reshape(ntest,s,1,T_in).repeat([1,1,T,1])


train_loader = torch.utils.data.DataLoader(torch.utils.data.TensorDataset(x_train, y_train), batch_size=batch_size, shuffle=True)
test_loader = torch.utils.data.DataLoader(torch.utils.data.TensorDataset(x_test, y_test), batch_size=batch_size, shuffle=False)

################################################################
# training and evaluation
################################################################
model = FNO2d(modes, modes, width).cuda()
print(count_params(model))

optimizer = Adam(model.parameters(), lr=learning_rate, weight_decay=1e-4)
scheduler = torch.optim.lr_scheduler.StepLR(optimizer, step_size=step_size, gamma=gamma)

myloss = LpLoss(size_average=False)

h = 1/(N-1)

coeffs_finite_difference_left = np.array([-11/(6*h), 3/h, -3/(2*h), 1/(3*h)])
coeffs_finite_difference_left = coeffs_finite_difference_left.astype(np.float32)
coeffs_finite_difference_left = torch.from_numpy(coeffs_finite_difference_left).cuda()

coeffs_finite_difference_right = np.array([-1/(3*h), 3/(2*h), -3/h , 11/(6*h)])
coeffs_finite_difference_right = coeffs_finite_difference_right.astype(np.float32)
coeffs_finite_difference_right = torch.from_numpy(coeffs_finite_difference_right).cuda()

bdy_left = 0.0

U = np.array([5])
U = torch.from_numpy(U).cuda()

t_final = np.array([0.5])
num_dt = 200
t = np.linspace(0, t_final, num_dt)
t = t[-T:]
t = torch.from_numpy(t).cuda()
bdy_right = U*torch.sin(torch.pi*t)

h_bdy_right = h*bdy_right

# y_normalizer.cuda()
for ep in range(epochs):
    model.train()
    t1 = default_timer()
    train_l2 = 0
    for x, y in train_loader:
        x, y = x.cuda(), y.cuda()

        optimizer.zero_grad()
        out = model(x, bdy_left, bdy_right, h_bdy_right, coeffs_finite_difference_right).reshape(batch_size, s, T)
#         print(out.shape)
#         print(y.shape)
#         out = y_normalizer.decode(out)
#         y = y_normalizer.decode(y)

        loss = myloss(out.view(batch_size,-1), y.view(batch_size,-1))
        loss.backward()

        optimizer.step()
        train_l2 += loss.item()

    scheduler.step()

    model.eval()
    test_l2 = 0.0
    with torch.no_grad():
        for x, y in test_loader:
            x, y = x.cuda(), y.cuda()

            out = model(x, bdy_left, bdy_right, h_bdy_right, coeffs_finite_difference_right).reshape(batch_size, s, T)
#             out = y_normalizer.decode(out)

            test_l2 += myloss(out.view(batch_size,-1), y.view(batch_size,-1)).item()
#             print(out.view(batch_size,-1))

    train_l2/= ntrain
    test_l2 /= ntest

    t2 = default_timer()
    print(ep, t2-t1, train_l2, test_l2)
    
pred = torch.zeros(y_test.shape)

test_loss_average = 0.0
test_loss_neumann = 0.0
test_solutions_plot = torch.zeros(1, s, 5)

index = 0
tol_bdy_error = 10**(-10)
test_loader = torch.utils.data.DataLoader(torch.utils.data.TensorDataset(x_test, y_test), batch_size=1, shuffle=False)
with torch.no_grad():
    for x, y in test_loader:
        test_l2 = 0
        x, y = x.cuda(), y.cuda()
        
        out = model(x, bdy_left, bdy_right, h_bdy_right, coeffs_finite_difference_right).reshape(1, s, T)
#         out = y_normalizer.decode(out)
#         out = model(x)[0][:,:,0]
#         pred[index] = out

        test_l2 += myloss(out.view(1, -1), y.view(1, -1)).item()
        test_loss_average += test_l2
                
#         error_neumann_right_bdy_vec = (out[0, -1, :] - out[0, -2, :])/h - bdy_right
#         indices_error_neumann_right_bdy_less_tol = torch.nonzero(torch.abs(error_neumann_right_bdy_vec) < tol_bdy_error)
#         error_neumann_right_bdy_vec[indices_error_neumann_right_bdy_less_tol] = 0
#         error_neumann_right_bdy = torch.norm(error_neumann_right_bdy_vec)
#         error_neumann_right_bdy_vec = coeffs_finite_difference_right[-1]*out[0, -1, :] + coeffs_finite_difference_right[-2]*out[0, -2, :] + coeffs_finite_difference_right[-3]*out[0, -3, :] + coeffs_finite_difference_right[-4]*out[0, -4, :] - bdy_right
#         indices_error_neumann_right_bdy_less_tol = torch.nonzero(torch.abs(error_neumann_right_bdy_vec) < tol_bdy_error)
#         error_neumann_right_bdy_vec[indices_error_neumann_right_bdy_less_tol] = 0
#         error_neumann_right_bdy = torch.norm(error_neumann_right_bdy_vec)
        
        
        error_neumann_right_bdy = torch.norm((out[0, -1, :] - out[0, -2, :])/h - bdy_right)
        error_neumann_left_bdy = torch.norm((out[0, 1, :] - out[0, 0, :])/h - bdy_left)
        test_loss_neumann += error_neumann_left_bdy + error_neumann_right_bdy
        print(index, test_l2)
            
        index = index + 1
     
    test_solutions_plot = out[:, :, 1::5]
    np.savetxt('results_heat_N_500_third_order_fd.csv', np.array([test_loss_average/ntest]))
#     dict_data = {"u_plot": test_solutions_plot.cpu().detach().numpy(), 'u_plot_all':out.cpu().detach().numpy()}
#     savemat("heat_neumann_boon_plot_data.mat", dict_data)
    print("Test error is {}".format(test_loss_average/ntest))
    print("Error at Neumann bdy is {}".format(test_loss_neumann/ntest))

# with torch.no_grad():
#     for x, y in test_loader:
#         test_l2 = 0
#         x, y = x.cuda(), y.cuda()

#         out = model(x).view(-1).reshape(s, s)
#         pred[index] = out

#         test_l2 += myloss(out.view(1, -1), y.view(1, -1)).item()
#         print(index, test_l2)
#         index = index + 1
