"""
@author: Zongyi Li
FNO code borrowed from (https://arxiv.org/pdf/2010.08895.pdf)
"""


import torch
import numpy as np
import torch.nn as nn
import torch.nn.functional as F
import sys

import matplotlib.pyplot as plt
from utilities3 import *

import operator
from functools import reduce
from functools import partial

from timeit import default_timer

from Adam import Adam

torch.manual_seed(0)
np.random.seed(0)


################################################################
# 3d fourier layers
################################################################

class SpectralConv3d(nn.Module):
    def __init__(self, in_channels, out_channels, modes1, modes2, modes3):
        super(SpectralConv3d, self).__init__()

        """
        3D Fourier layer. It does FFT, linear transform, and Inverse FFT.    
        """

        self.in_channels = in_channels
        self.out_channels = out_channels
        self.modes1 = modes1 #Number of Fourier modes to multiply, at most floor(N/2) + 1
        self.modes2 = modes2
        self.modes3 = modes3

        self.scale = (1 / (in_channels * out_channels))
        self.weights1 = nn.Parameter(self.scale * torch.rand(in_channels, out_channels, self.modes1, self.modes2, self.modes3, dtype=torch.cfloat))
        self.weights2 = nn.Parameter(self.scale * torch.rand(in_channels, out_channels, self.modes1, self.modes2, self.modes3, dtype=torch.cfloat))
        self.weights3 = nn.Parameter(self.scale * torch.rand(in_channels, out_channels, self.modes1, self.modes2, self.modes3, dtype=torch.cfloat))
        self.weights4 = nn.Parameter(self.scale * torch.rand(in_channels, out_channels, self.modes1, self.modes2, self.modes3, dtype=torch.cfloat))

    # Complex multiplication
    def compl_mul3d(self, input, weights):
        # (batch, in_channel, x,y,t ), (in_channel, out_channel, x,y,t) -> (batch, out_channel, x,y,t)
        return torch.einsum("bixyz,ioxyz->boxyz", input, weights)

    def forward(self, x):
        batchsize = x.shape[0]
        #Compute Fourier coeffcients up to factor of e^(- something constant)
        x_ft = torch.fft.rfftn(x, dim=[-3,-2,-1])

        # Multiply relevant Fourier modes
        out_ft = torch.zeros(batchsize, self.out_channels, x.size(-3), x.size(-2), x.size(-1)//2 + 1, dtype=torch.cfloat, device=x.device)
        out_ft[:, :, :self.modes1, :self.modes2, :self.modes3] = \
            self.compl_mul3d(x_ft[:, :, :self.modes1, :self.modes2, :self.modes3], self.weights1)
        out_ft[:, :, -self.modes1:, :self.modes2, :self.modes3] = \
            self.compl_mul3d(x_ft[:, :, -self.modes1:, :self.modes2, :self.modes3], self.weights2)
        out_ft[:, :, :self.modes1, -self.modes2:, :self.modes3] = \
            self.compl_mul3d(x_ft[:, :, :self.modes1, -self.modes2:, :self.modes3], self.weights3)
        out_ft[:, :, -self.modes1:, -self.modes2:, :self.modes3] = \
            self.compl_mul3d(x_ft[:, :, -self.modes1:, -self.modes2:, :self.modes3], self.weights4)

        #Return to physical space
        x = torch.fft.irfftn(out_ft, s=(x.size(-3), x.size(-2), x.size(-1)))
        return x

class KernelCorrection1d(nn.Module):
    def __init__(self, layer):
        super(KernelCorrection1d, self).__init__()

        """
        1D Corrected Layer. It modifies the kernel to enforce a Dirichlet boundary condition.   
        """

        self.layer = layer
        
    
    def forward(self, x , bdy_left, bdy_right, bdy_down, bdy_up, h):
        """
        Forward layer to enforce the boundary as well as the correction terms for
            a general operator K
            
        We apply T^{1}_Neumann, K and T^{2}_{Neumann} on x
        """
        
        tol = 10**(-4)
        
        e1_x = torch.zeros_like(x)
        eN_x = torch.zeros_like(x)
                
        e1_x[:, :, 0, :, :] = 1
        eN_x[:, :, -1, :, :] = 1
                
        K00_x = self.layer(e1_x)[0,0,0, 0, :]
        KN0_x = self.layer(e1_x)[0,0,-1, 0, :]
        K0N_x = self.layer(eN_x)[0,0,0, 0, :]
        KNN_x = self.layer(eN_x)[0,0,-1, 0, :]
        
        
        indices_K00_x_less_tol = torch.nonzero(K00_x < tol)
        indices_KNN_x_less_tol = torch.nonzero(KNN_x < tol)
        
        
        if len(indices_K00_x_less_tol) != 0:
            K00_x[indices_K00_x_less_tol] = tol
            
        if len(indices_KNN_x_less_tol) != 0:
            KNN_x[indices_KNN_x_less_tol] = tol
                   
        
        tilde_K00_x = K00_x - KN0_x*K0N_x/KNN_x
        
        indices_tilde_K00_x_less_tol = torch.nonzero(tilde_K00_x < tol)
        
        if len(indices_tilde_K00_x_less_tol) != 0:
            tilde_K00_x[indices_tilde_K00_x_less_tol] = tol
            
    
        Kx = self.layer(x)
        
        T2_left_x = x.clone()
        

        T2_left_x[:, :, 0, :, :] = 2*x[:, :, 0, :, :] - (1/tilde_K00_x)*(Kx[:, :, 0, :, :] - Kx[:, :, -1, :, :]*K0N_x/KNN_x + K0N_x*x[:, :, -1, :, :])

        Ky = self.layer(T2_left_x.clone())
        T2_right_y = T2_left_x.clone()
        
        T2_right_y[:, :, -1, :, :] = 2*T2_right_y[:, :, 0, :, :] - Ky[:, :, -1, :, :]/KNN_x
        
        Kv = self.layer(T2_right_y.clone())
        
        T1 = Kv.clone()
        
        T1[:, :, 0, :, :] = x[:, :, 0, :, :]
        T1[:, :, -1, :, :] = x[:, :, -1, :, :]
        
        
        return T1

    
class FNO3d(nn.Module):
    def __init__(self, modes1, modes2, modes3, width):
        super(FNO3d, self).__init__()

        """
        The overall network. It contains 4 layers of the Fourier layer.
        1. Lift the input to the desire channel dimension by self.fc0 .
        2. 4 layers of the integral operators u' = (W + K)(u).
            W defined by self.w; K defined by self.conv .
        3. Project from the channel space to the output space by self.fc1 and self.fc2 .
        
        input: the solution of the first 10 timesteps + 3 locations (u(1, x, y), ..., u(10, x, y),  x, y, t). It's a constant function in time, except for the last index.
        input shape: (batchsize, x=64, y=64, t=40, c=13)
        output: the solution of the next 40 timesteps
        output shape: (batchsize, x=64, y=64, t=40, c=1)
        """

        self.modes1 = modes1
        self.modes2 = modes2
        self.modes3 = modes3
        self.width = width
        self.padding = 6 # pad the domain if input is non-periodic
        self.fc0 = nn.Linear(4, self.width)
        # input channel is 12: the solution of the first 10 timesteps + 3 locations (u(1, x, y), ..., u(10, x, y),  x, y, t)

        self.conv0 = SpectralConv3d(self.width, self.width, self.modes1, self.modes2, self.modes3)
        self.conv1 = SpectralConv3d(self.width, self.width, self.modes1, self.modes2, self.modes3)
        self.conv2 = SpectralConv3d(self.width, self.width, self.modes1, self.modes2, self.modes3)
        self.conv3 = SpectralConv3d(self.width, self.width, self.modes1, self.modes2, self.modes3)
        
        self.kernel_correction0 = KernelCorrection1d(self.conv0)
        self.kernel_correction1 = KernelCorrection1d(self.conv1)
        self.kernel_correction2 = KernelCorrection1d(self.conv2)
        self.kernel_correction3 = KernelCorrection1d(self.conv3)
        
        self.w0 = nn.Conv3d(self.width, self.width, 1)
        self.w1 = nn.Conv3d(self.width, self.width, 1)
        self.w2 = nn.Conv3d(self.width, self.width, 1)
        self.w3 = nn.Conv3d(self.width, self.width, 1)
        
        self.w0_correction = KernelCorrection1d(self.w0)
        self.w1_correction = KernelCorrection1d(self.w1)
        self.w2_correction = KernelCorrection1d(self.w2)
        self.w3_correction = KernelCorrection1d(self.w3)
        
        
        self.bn0 = torch.nn.BatchNorm3d(self.width)
        self.bn1 = torch.nn.BatchNorm3d(self.width)
        self.bn2 = torch.nn.BatchNorm3d(self.width)
        self.bn3 = torch.nn.BatchNorm3d(self.width)

        self.fc1 = nn.Linear(self.width, 128)
        self.fc2 = nn.Linear(128, 1)

    def forward(self, x, bdy_left, bdy_right, bdy_down, bdy_up, h):
        
        bdy_left = bdy_left.reshape(x.shape[0], 1, x.shape[2], x.shape[3])
        bdy_right = bdy_right.reshape(x.shape[0], 1, x.shape[2], x.shape[3])
        
        bdy_down = bdy_down.reshape(x.shape[0], x.shape[1], 1 , x.shape[3])
        bdy_up = bdy_up.reshape(x.shape[0], x.shape[1], 1 , x.shape[3])
        
        grid = self.get_grid(x.shape, x.device)
        x = torch.cat((x, grid), dim=-1)
        x = self.fc0(x)
        x = x.permute(0, 4, 1, 2, 3)
        x = F.pad(x, [0,self.padding]) # pad the domain if input is non-periodic

#         x1 = self.conv0(x)
#         x2 = self.w0(x)
#         x = x1 + x2
#         x = F.gelu(x)

        x1 = self.kernel_correction0(x, bdy_left, bdy_right, bdy_down, bdy_up, h)
        x2 = self.w0_correction(x, bdy_left, bdy_right, bdy_down, bdy_up, h)
        x = (x1 + x2)/2
        x[:, :, 1:x.shape[2]-1, 1:x.shape[3]-1, :] = F.gelu(x[:, :, 1:x.shape[2]-1, 1:x.shape[3]-1, :].clone())

#         x1 = self.conv1(x)
#         x2 = self.w1(x)
#         x = x1 + x2
#         x = F.gelu(x)

        x1 = self.kernel_correction1(x, bdy_left, bdy_right, bdy_down, bdy_up, h)
        x2 = self.w1_correction(x, bdy_left, bdy_right, bdy_down, bdy_up, h)
        x = (x1 + x2)/2
        x[:, :, 1:x.shape[2]-1, 1:x.shape[3]-1, :] = F.gelu(x[:, :, 1:x.shape[2]-1, 1:x.shape[3]-1, :].clone())

#         x1 = self.conv2(x)
#         x2 = self.w2(x)
#         x = x1 + x2
#         x = F.gelu(x)
        
        x1 = self.kernel_correction2(x, bdy_left, bdy_right, bdy_down, bdy_up, h)
        x2 = self.w2_correction(x, bdy_left, bdy_right, bdy_down, bdy_up, h)
        x = (x1 + x2)/2
#         x = x1 + x2
        x[:, :, 1:x.shape[2]-1, 1:x.shape[3]-1, :] = F.gelu(x[:, :, 1:x.shape[2]-1, 1:x.shape[3]-1, :].clone())
#         x = F.gelu(x)

#         x1 = self.conv3(x)
#         x2 = self.w3(x)
#         x = x1 + x2
        
        x1 = self.kernel_correction3(x, bdy_left, bdy_right, bdy_down, bdy_up, h)
        x2 = self.w3_correction(x, bdy_left, bdy_right, bdy_down, bdy_up, h)
        x = (x1 + x2)/2
#         x = x1 + x2

        x = x[..., :-self.padding]
        x = x.permute(0, 2, 3, 4, 1) # pad the domain if input is non-periodic
        x = self.fc1(x)
#         x = F.gelu(x)
        x[:, 1:x.shape[1]-1, 1:x.shape[2]-1, : ,:] = F.gelu(x[:, 1:x.shape[1]-1, 1:x.shape[2]-1, : ,:].clone())
        x = self.fc2(x)
        
        x[:, -1, :, :] = x[:, -2, :, :] + h*bdy_right 
        x[:, 0, :, :] = x[:, 1, :, :] - h*bdy_left
        
        x[:, :, -1, :] = x[:, :, -2, :] + h*bdy_up 
        x[:, :, 0, :] = x[:, :, 1, :] - h*bdy_down
        
        
        return x

    def get_grid(self, shape, device):
        batchsize, size_x, size_y, size_z = shape[0], shape[1], shape[2], shape[3]
        gridx = torch.tensor(np.linspace(0, 1, size_x), dtype=torch.float)
        gridx = gridx.reshape(1, size_x, 1, 1, 1).repeat([batchsize, 1, size_y, size_z, 1])
        gridy = torch.tensor(np.linspace(0, 1, size_y), dtype=torch.float)
        gridy = gridy.reshape(1, 1, size_y, 1, 1).repeat([batchsize, size_x, 1, size_z, 1])
        gridz = torch.tensor(np.linspace(0, 1, size_z), dtype=torch.float)
        gridz = gridz.reshape(1, 1, 1, size_z, 1).repeat([batchsize, size_x, size_y, 1, 1])
        return torch.cat((gridx, gridy, gridz), dim=-1).to(device)

################################################################
# configs
################################################################

# TRAIN_PATH = 'data/ns_data_V100_N1000_T50_1.mat'
# TEST_PATH = 'data/ns_data_V100_N1000_T50_2.mat'

ntrain = 1000
ntest = 200

modes = 8
width = 20

batch_size = 10
batch_size2 = batch_size

epochs = 500
learning_rate = 0.001
scheduler_step = 100
scheduler_gamma = 0.5

print(epochs, learning_rate, scheduler_step, scheduler_gamma)

# path = 'test'
# # path = 'ns_fourier_V100_N'+str(ntrain)+'_ep' + str(epochs) + '_m' + str(modes) + '_w' + str(width)
# path_model = 'model/'+path
# path_train_err = 'results/'+path+'train.txt'
# path_test_err = 'results/'+path+'test.txt'
# path_image = 'image/'+path


runtime = np.zeros(2, )
t1 = default_timer()

sub = int(sys.argv[1]) #subsampling rate
N = 100 // sub #total grid size divided by the subsampling rate
S = N

T_in = 1
T = 25

################################################################
# load data
################################################################

data_all = scipy.io.loadmat('wave_2D_data.mat')

train_a = data_all['a'][:ntrain,::sub, ::sub,:T_in]
train_a = train_a.astype(np.float32)
train_a = torch.from_numpy(train_a)

# y_train = data_all['u'][:ntrain,::sub,T_in:T+T_in]
train_u = data_all['u'][:ntrain,::sub, ::sub,-T:]
train_u = train_u.astype(np.float32)
train_u = torch.from_numpy(train_u)

test_a = data_all['a'][-ntest:,::sub, ::sub,:T_in]
test_a = test_a.astype(np.float32)
test_a = torch.from_numpy(test_a)

test_u = data_all['u'][-ntest:,::sub, ::sub,-T:]
test_u = test_u.astype(np.float32)
test_u = torch.from_numpy(test_u)

# train_a = reader.read_field('u')[:ntrain,::sub,::sub,:T_in]
# train_u = reader.read_field('u')[:ntrain,::sub,::sub,T_in:T+T_in]

# reader = MatReader(TEST_PATH)
# test_a = reader.read_field('u')[-ntest:,::sub,::sub,:T_in]
# test_u = reader.read_field('u')[-ntest:,::sub,::sub,T_in:T+T_in]

print(train_u.shape)
print(test_u.shape)
assert (S == train_u.shape[-2])
assert (T == train_u.shape[-1])


# a_normalizer = UnitGaussianNormalizer(train_a)
# train_a = a_normalizer.encode(train_a)
# test_a = a_normalizer.encode(test_a)

# y_normalizer = UnitGaussianNormalizer(train_u)
# train_u = y_normalizer.encode(train_u)

train_a = train_a.reshape(ntrain,S,S,1,T_in).repeat([1,1,1,T,1])
test_a = test_a.reshape(ntest,S,S,1,T_in).repeat([1,1,1,T,1])

train_loader = torch.utils.data.DataLoader(torch.utils.data.TensorDataset(train_a, train_u), batch_size=batch_size, shuffle=True)
test_loader = torch.utils.data.DataLoader(torch.utils.data.TensorDataset(test_a, test_u), batch_size=batch_size, shuffle=False)

t2 = default_timer()

print('preprocessing finished, time used:', t2-t1)
device = torch.device('cuda')

################################################################
# training and evaluation
################################################################
model = FNO3d(modes, modes, modes, width).cuda()
# model = torch.load('model/ns_fourier_V100_N1000_ep100_m8_w20')

print(count_params(model))
optimizer = Adam(model.parameters(), lr=learning_rate, weight_decay=1e-4)
scheduler = torch.optim.lr_scheduler.StepLR(optimizer, step_size=scheduler_step, gamma=scheduler_gamma)

h = 1/(S-1)

coeffs_finite_difference_right = np.array([-1/(3*h), 3/(2*h), -3/h , 11/(6*h)])
coeffs_finite_difference_right = coeffs_finite_difference_right.astype(np.float32)
coeffs_finite_difference_right = torch.from_numpy(coeffs_finite_difference_right).cuda()

bdy_left = 0.0
bdy_right = 0.0
bdy_down = 0.0
bdy_up = 0.0

myloss = LpLoss(size_average=False)
# y_normalizer.cuda()
for ep in range(epochs):
    model.train()
    t1 = default_timer()
    train_mse = 0
    train_l2 = 0
    for x, y in train_loader:
        x, y = x.cuda(), y.cuda()

        optimizer.zero_grad()
        out = model(x, bdy_left, bdy_right, bdy_down, bdy_up, h).view(batch_size, S, S, T)

        mse = F.mse_loss(out, y, reduction='mean')
        # mse.backward()

#         y = y_normalizer.decode(y)
#         out = y_normalizer.decode(out)
        l2 = myloss(out.view(batch_size, -1), y.view(batch_size, -1))
        l2.backward()

        optimizer.step()
        train_mse += mse.item()
        train_l2 += l2.item()

    scheduler.step()

    model.eval()
    test_l2 = 0.0
    with torch.no_grad():
        for x, y in test_loader:
            x, y = x.cuda(), y.cuda()
#             print(model(x).shape)
#             print(y.shape)

            out = model(x, bdy_left, bdy_right, bdy_down, bdy_up, h).view(batch_size, S, S, T)
#             out = y_normalizer.decode(out)
            test_l2 += myloss(out.view(batch_size, -1), y.view(batch_size, -1)).item()

    train_mse /= len(train_loader)
    train_l2 /= ntrain
    test_l2 /= ntest

    t2 = default_timer()
    print(ep, t2-t1, train_mse, train_l2, test_l2)
# torch.save(model, path_model)

test_loss_average = 0.0
test_loss_neumann = 0.0
pred = torch.zeros(test_u.shape)
index = 0
test_loader = torch.utils.data.DataLoader(torch.utils.data.TensorDataset(test_a, test_u), batch_size=1, shuffle=False)
with torch.no_grad():
    for x, y in test_loader:
        test_l2 = 0
        x, y = x.cuda(), y.cuda()

        out = model(x, bdy_left, bdy_right, bdy_down, bdy_up, h)
#         out = y_normalizer.decode(out)
#         pred[index] = out

        test_l2 += myloss(out.view(1, -1), y.view(1, -1)).item()
#         error_neumann_right_bdy = (coeffs_finite_difference_right[-1]*out[0, -1, :, 0] + coeffs_finite_difference_right[-2]*out[0, -2, :, 0] + coeffs_finite_difference_right[-3]*out[0, -3, :, 0] + coeffs_finite_difference_right[-4]*out[0, -4, :, 0] - neumann_bdy_right)**2
        test_loss_average += test_l2
        error_neumann_right_bdy = torch.norm((out[0, -1, :, 0] - out[0, -2, :, 0])/h - bdy_right)
        error_neumann_left_bdy = torch.norm((out[0, 1, :, 0] - out[0, 0, :, 0])/h - bdy_left)
        error_neumann_up_bdy = torch.norm((out[0, :, -1, 0] - out[0, :, -2, 0])/h - bdy_up)
        error_neumann_down_bdy = torch.norm((out[0, :, 1, 0] - out[0, :, 0, 0])/h - bdy_down)
        
        test_loss_neumann += (error_neumann_left_bdy + error_neumann_right_bdy + error_neumann_up_bdy + error_neumann_down_bdy)
        print(index, test_l2)
        index = index + 1
        
    np.savetxt('results_wave_2D_N+100.csv', np.array([test_loss_average/ntest, (test_loss_neumann/ntest).cpu().detach().numpy()]))
    print("Test error is {}".format(test_loss_average/ntest))
    print("Error at Neumann bdy is {}".format(test_loss_neumann/ntest))

# scipy.io.savemat('pred/'+path+'.mat', mdict={'pred': pred.cpu().numpy()})




