from abc import abstractmethod, ABC

from sklearn.cluster import KMeans as KM, DBSCAN as DS
from sklearn.neighbors import NearestCentroid, NearestNeighbors


class Clusterer(ABC):

    @abstractmethod
    def fit(self, X):
        pass

    @abstractmethod
    def predict(self, X):
        pass


class AgglomerativeClustering(Clusterer):

    def __init__(self, distance_threshold=0, n_clusters=None):
        self._agg = AgglomerativeClustering(distance_threshold=distance_threshold, n_clusters=n_clusters)
        self._clusterer = NearestCentroid()

    def fit(self, X):
        self._agg.fit(X)
        self._clusterer.fit(X, self._agg.labels_)

    def predict(self, X):
        return self._clusterer.predict(X)[0]


class KMeans(Clusterer):

    def __init__(self, n_clusters=2):
        self._clusterer = KM(n_clusters, init="random")

    def fit(self, X):
        self._clusterer.fit(X)

    def predict(self, X):
        return self._clusterer.predict(X)


class DBSCAN(Clusterer):

    def __init__(self, eps=4.0, min_cluster_size=1, metric='euclidean'):
        self._clusterer = DS(eps=eps, min_samples=min_cluster_size, metric=metric)
        self._train_labels = None
        self._nn = NearestNeighbors(n_neighbors=1)
        self._eps = eps

    def fit(self, X):
        self._clusterer.fit(X)
        self._train_labels = self._clusterer.labels_
        self._nn.fit(X)

    def predict(self, X):
        distances, indices = self._nn.kneighbors(X)
        distance, index = distances[0], indices[0]
        if distance > self._eps:
            return -1  # noise
        return self._train_labels[index[0]]
