import time

import multiprocessing
import os
import shutil
import warnings
from typing import Callable, List, Iterable, Any

import pandas as pd
import numpy as np
import pickle
import textwrap


class UnionFind:
    """Union-find data structure.

    Each unionFind instance X maintains a family of disjoint sets of
    hashable objects, supporting the following two methods:

    - X[item] returns a name for the set containing the given item.
      Each set is named by an arbitrarily-chosen one of its members; as
      long as the set remains unchanged it will keep the same name. If
      the item is not yet part of a set in X, a new singleton set is
      created for it.

    - X.union(item1, item2, ...) merges the sets containing each item
      into a single larger set.  If any item is not yet part of a set
      in X, it is added to X as one of the members of the merged set.

      Based on Josiah Carlson's code,
      http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/215912
      with significant additional changes by D. Eppstein.

    """

    def __init__(self, objects):
        """Create a new empty union-find structure."""
        self.weights = {}
        self.parents = {}
        for object in objects:
            self.parents[object] = object
            self.weights[object] = 1

    def __getitem__(self, object):
        """Find and return the name of the set containing the object."""

        # check for previously unknown object
        if object not in self.parents:
            self.parents[object] = object
            self.weights[object] = 1
            return object

        # find path of objects leading to the root
        path = [object]
        root = self.parents[object]
        while root != path[-1]:
            path.append(root)
            root = self.parents[root]

        # compress the path and return
        for ancestor in path:
            self.parents[ancestor] = root
        return root

    def __iter__(self):
        """Iterate through all items ever found or unioned by this structure."""
        return iter(self.parents)

    def merge(self, *objects):
        """Find the sets containing the objects and merge them all."""
        roots = [self[x] for x in objects]
        heaviest = max([(self.weights[r], r) for r in roots])[1]
        for r in roots:
            if r != heaviest:
                self.weights[heaviest] += self.weights[r]
                self.parents[r] = heaviest


class Vividict(dict):
    """
    Autovivification like perl
    """

    def __missing__(self, key):
        value = self[key] = type(self)()
        return value


def if_not_none(value: Any, default_value: Any) -> Any:
    """
    If the value is not none, returns it. Otherwise, returns the default value
    :param value: teh value to check
    :param default_value: the default value
    """
    return value if value is not None else default_value


def now():
    """
    Return the current time in milliseconds
    """
    return int(round(time.time() * 1000))


def indent(x: Any, count: int = 1) -> str:
    """
    Indent an object
    :param x: the object
    :param count: the number of tab spaces
    """
    return textwrap.indent(str(x), '\t' * count)


def select_rows(data: pd.DataFrame, indices: Iterable, reset_index=True) -> pd.DataFrame:
    """
    Select a set of rows from a data frame
    :param data: the data frame
    :param indices: the row indices
    :param reset_index: whether the indices should be reset
    :return: the subframe
    """""
    frame = data.loc[indices]
    if reset_index:
        return frame.reset_index(drop=True)
    return frame


def run_parallel(functions: List[Callable]):
    """
    Run the list of function in parallel and return the results in a list
    :param functions: the functions to execute
    :return: a list of results
    """

    # if iit's just one function, don't waste time creating new processes
    if len(functions) == 1:
        return [f() for f in functions]

    n_procs = len(functions)
    pool = multiprocessing.Pool(processes=n_procs)
    processes = [pool.apply_async(functions[i]) for i in range(n_procs)]
    return [p.get() for p in processes]


def range_without(start, end, *skips):
    """
    Create a range while omitting certain values
    :param start: the start of the range
    :param end: the end of the range (excluded)
    :param skips: the numbers to skip
    """
    skip = set(skips)
    return [x for x in range(start, end) if x not in skip]


def pd2np(data: pd.Series):
    """
    Convert a Pandas series of arrays to a numpy 2D array
    :param data: the series
    :return: a numpy 2D array
    """
    return np.stack(data.values)


def show(data: Any, verbose: bool):
    """
    Convenience method to display data, if verbose is set to true
    :param data: the information to print
    :param verbose: the mode
    """
    if verbose:
        print(data)


def save(object, filename=None, binary=True):
    """
    Convenience method for saving object to file without risking wrong mode overwrites
    :param binary: whether to save as a binary file
    :param object: the object to pickle
    :param filename: the filename
    """
    if filename is None:
        warnings.warn("No filename specified, saving to temp.pkl...")
        filename = 'temp.pkl'
    mode = 'wb' if binary else 'w'
    with open(filename, mode) as file:
        if binary:
            pickle.dump(object, file)
        else:
            file.write(str(object))
    return filename


def load(filename=None, binary=True):
    """
    Convenience method for loading object from file without risking wrong mode overwrites
    :param binary: whether to load as a binary file
    :param filename: the filename
    """
    if filename is None:
        warnings.warn("No filename specified, loading from temp.pkl...")
        filename = 'temp.pkl'
    mode = 'rb' if binary else 'r'
    with open(filename, mode) as file:
        return pickle.load(file)


def get_dir_name(file):
    """
    Get the directory of the given file
    :param file: the file
    :return: the file's directory
    """
    return os.path.dirname(os.path.realpath(file))


def get_sibling_file(file,
                     sibling_name):
    """
    Get the full path to a file within the same directory as another file
    :param file: the file whose directory we care about
    :param sibling_name: the name of the file we're looking for
    :return: a full path to the sibling file
    """
    return make_path(get_dir_name(file), sibling_name)


def make_path(root,
              *args):
    """
    Creates a path from the given parameters
    :param root: the root of the path
    :param args: the elements of the path
    :return: a string, each element separated by a forward slash.
    """
    path = root
    if path.endswith('/'):
        path = path[0:-1]
    for element in args:
        if not isinstance(element, str):
            element = str(element)
        if element[0] != '/':
            path += '/'
        path += element
    return path


def exists(path):
    return os.path.exists(path)


def make_dir(path,
             clean=True):
    """
    Create a new directory (create the entire tree if necessary)
    :param path: the directory's path
    :param clean: whether the directory should made empty (if it already exists)
    """

    exists = os.path.exists(path)
    if not exists:
        os.makedirs(path)
    elif clean:
        # Delete directory
        shutil.rmtree(path)
        os.makedirs(path)


def copy_directory(source, dest):
    """
    Copy a directory tree from source to a destination. The directory must not exist in the destination
    :param source:
    :param dest:
    """
    if os.path.exists(dest):
        raise ValueError(dest + " already exists!")
    shutil.copytree(source, dest)


def merge(input_files, output_file):
    """
    Merge a list of files into one single file
    :param input_files: the list of input files
    :param output_file: the output file
    """
    with open(output_file, 'w') as wfd:
        for f in input_files:
            with open(f, 'r') as fd:
                shutil.copyfileobj(fd, wfd)


def files_in_dir(directory):
    """
    Generate the files inside a directory tree
    :param directory: the directory tree to traverse
    :return: the path and filename of files within the tree
    """

    for dirpath, dirnames, filenames in os.walk(directory):
        for f in filenames:
            file = make_path(dirpath, f)
            if os.path.isfile(file):
                yield dirpath, f
