import gym
import datetime
import numpy as np
import os
import sys
from collections import deque
from ..utils import file_io, get_parser
from .controller import SkillController, actions, options
from ..ataritools.envs import MontezumaEnv
from .plans import Plans

FPS = 600

def RunEpisode(env, seed, plan=None, render=False, quiet=False):
    """Run one episode in the specified MontezumaEnv using a SkillController
    """
    np.random.seed(seed)

    skillController = SkillController(initial_plan=plan)

    images = []
    states = []
    experiences = []

    last_update = datetime.datetime.now()
    global FPS

    done = False
    timestep = 0
    alpha = 0.99
    actual_fps = 0
    if not quiet:
        print()
    while not done:
        now = datetime.datetime.now()
        update_time = datetime.timedelta(milliseconds=1000 / FPS)
        if not render or (now - last_update > update_time):
            actual_fps = alpha * actual_fps + (1 - alpha) / (now - last_update).total_seconds()
            if not quiet:
                print('fps = {:4.1f}'.format(actual_fps), end='\r')
            last_update = now

            if render:
                env.render()

            image = env.getFrame()
            state = env.getState()

            valid_skills = skillController.getValidSkills(state)

            action, current_option, option_frame, op_done = skillController.runSkillPolicy(state)

            ob, reward, done, _ = env.step(action.value)

            experience = {
                'timestep': timestep,
                'action': action,
                'reward': reward,
                'current_option': current_option,
                'option_frame': option_frame,
                'op_done': op_done,
                'valid_skills': valid_skills,
            }
            timestep += 1
            images.append(image)
            states.append(state)
            experiences.append(experience)

            if skillController.isInitialized:
                break

    image = env.getFrame()
    state = env.getState()
    images.append(image)
    states.append(state)
    if not quiet:
        print()
    return images, states, experiences

def main(seed, output_dir=None, single_life=False, save_mp4=False, render=False):
    """Initialize a MontezumaEnv, run an episode, and save the results
    """
    env = MontezumaEnv(seed, single_life=single_life, single_screen=False)
    images, states, experiences = RunEpisode(env, seed, plan=Plans.GetFullRunPlan(), render=render)
    env.close()
    print('Observed frames:', len(images))

    seed_str = 'seed_{:05d}'.format(seed)
    if output_dir:
        path = os.path.join(output_dir, seed_str)
        file_io.saveData(path, images, states, experiences)
    if save_mp4:
        file_io.saveMP4('./output/videos/interact/{}.mp4'.format(seed_str), images)

if __name__ == '__main__':
    parser = get_parser()
    parser.add_argument('-d', '--output_dir', help='Output directory', type=str, default='')
    parser.add_argument('-s', '--seed', help='Random seed', type=int, required=True)
    parser.add_argument('--render', help='Render environment', action='store_true')
    parser.add_argument('--save_mp4', help='Save video as .mp4', action='store_true')
    parser.add_argument('--single_life',
                        help='Terminate episode on loss of life',
                        action='store_true')
    args = parser.parse_args()
    main(args.seed, args.output_dir, args.single_life, args.save_mp4, args.render)
