import copy
from collections import deque
import numpy as np

from .options import options as o

WAIT_50 = [o.WAIT_10, o.WAIT_10, o.WAIT_10, o.WAIT_10, o.WAIT_10]

def makeWait(x):
    waits = []
    while x > 50:
        waits += WAIT_50
        x -= 50
    while x > 10:
        waits.append(o.WAIT_10)
        x -= 10
    while x > 1:
        waits.append(o.WAIT_1)
        x -= 1
    return waits

class Plans:
    """Class for conveniently specifying an initial plan to a SkillController
    """
    #https://www.youtube.com/watch?v=sYbBgkP9aMo

    # Movement
    S0_TO_NW = [o.RUN_LEFT, o.JUMP_LEFT]
    S0_TO_NE = [o.RUN_RIGHT, o.JUMP_RIGHT]
    S0_TO_MID = [o.CLIMB_DOWN, o.CLIMB_DOWN]
    S0_TO_E = S0_TO_MID + [o.RUN_RIGHT, o.JUMP_RIGHT, o.JUMP_RIGHT]
    S0_TO_SE = S0_TO_E + [o.CLIMB_DOWN, o.CLIMB_DOWN, o.RUN_LEFT]
    S0_TO_SW = S0_TO_SE + [o.JUMP_LEFT, o.WAIT_FOR_SKULL, o.JUMP_LEFT, o.RUN_LEFT]
    S0_TO_W = S0_TO_SW + [o.RUN_LEFT, o.CLIMB_UP, o.CLIMB_UP]
    S0_TO_KEY = S0_TO_W + [o.JUMP_LEFT]

    KEY_TO_W = [o.RUN_RIGHT]
    KEY_TO_SW = KEY_TO_W + [o.CLIMB_DOWN, o.CLIMB_DOWN, o.RUN_RIGHT]
    KEY_TO_SE = KEY_TO_SW + [o.WAIT_FOR_SKULL, o.JUMP_RIGHT, o.RUN_RIGHT]
    KEY_TO_E = KEY_TO_SE + [o.RUN_RIGHT, o.CLIMB_UP, o.CLIMB_UP]
    KEY_TO_MID = KEY_TO_E + [o.JUMP_LEFT, o.JUMP_LEFT, o.RUN_LEFT]
    KEY_TO_S0 = KEY_TO_MID + [o.CLIMB_UP]
    KEY_TO_NW = KEY_TO_S0 + S0_TO_NW
    KEY_TO_NE = KEY_TO_S0 + S0_TO_NE

    S0_TO_SKULL = S0_TO_SE + [o.RUN_LEFT]
    KEY_TO_SKULL = KEY_TO_SW + [o.RUN_RIGHT]
    S0_TO_KEY_NOSKULL = S0_TO_SE + [o.RUN_LEFT, o.RUN_LEFT, o.CLIMB_UP, o.JUMP_LEFT]

    # Base positions
    S0 = []
    NE = S0_TO_NE
    NW = S0_TO_NW
    MID = S0_TO_MID
    E = S0_TO_E
    SE = S0_TO_SE
    SW = S0_TO_SW
    W = S0_TO_W

    # Positions w/ key
    S0_KEY = S0_TO_KEY + KEY_TO_S0
    NW_KEY = S0_TO_KEY + KEY_TO_NW
    NE_KEY = S0_TO_KEY + KEY_TO_NE
    MID_KEY = S0_TO_KEY + KEY_TO_MID
    E_KEY = S0_TO_KEY + KEY_TO_E
    SE_KEY = S0_TO_KEY + KEY_TO_SE
    SW_KEY = S0_TO_KEY + KEY_TO_SW
    W_KEY = S0_TO_KEY + KEY_TO_W

    # Positions w/o skull
    S0_SKULL = S0_TO_SKULL
    NW_SKULL = S0_TO_SKULL + S0_TO_NW
    NE_SKULL = S0_TO_SKULL + S0_TO_NE
    MID_SKULL = S0_TO_SKULL + S0_TO_MID
    E_SKULL = S0_TO_SKULL + S0_TO_E
    SE_SKULL = S0_TO_SKULL + S0_TO_SE
    SW_SKULL = SE_SKULL + [o.RUN_LEFT]
    W_SKULL = SW_SKULL + [o.RUN_LEFT, o.CLIMB_UP]

    # Positions w/ key + w/o skull
    S0_BOTH = S0_TO_KEY + KEY_TO_SKULL
    NE_BOTH = S0_TO_KEY + KEY_TO_SKULL + S0_TO_NE
    NW_BOTH = S0_TO_KEY + KEY_TO_SKULL + S0_TO_NW
    MID_BOTH = S0_TO_KEY + KEY_TO_SKULL + S0_TO_MID
    E_BOTH = S0_TO_KEY + KEY_TO_SKULL + S0_TO_E
    SE_BOTH = S0_TO_KEY + KEY_TO_SKULL + S0_TO_SE
    SW_BOTH = S0_TO_SKULL + S0_TO_KEY_NOSKULL + KEY_TO_SW
    W_BOTH = S0_TO_SKULL + S0_TO_KEY_NOSKULL + KEY_TO_W

    # Intentionally lose a life. Lose N lives with (S0_TO_DIE * N)
    S0_TO_DIE = [o.RUN_RIGHT, o.RUN_RIGHT]

    JUMP_5_TIMES = [o.JUMP] * 5
    WAIT = [o.NONE] * 5
    WAIT_FOR_SKULL = [o.WAIT_FOR_SKULL]

    RUN_LEFT = [o.RUN_LEFT]
    RUN_RIGHT = [o.RUN_RIGHT]
    RUN_LEFT3 = [o.RUN_LEFT3]
    RUN_RIGHT3 = [o.RUN_RIGHT3]
    JUMP_LEFT = [o.JUMP_LEFT]
    JUMP_RIGHT = [o.JUMP_RIGHT]
    JUMP = [o.JUMP]
    DOWN = [o.CLIMB_DOWN, o.CLIMB_DOWN]
    UP = [o.CLIMB_UP, o.CLIMB_UP]

    SMALL_STEP_RIGHT = [o.STEP_RIGHT]
    STEP_RIGHT = (SMALL_STEP_RIGHT + SMALL_STEP_RIGHT + SMALL_STEP_RIGHT + SMALL_STEP_RIGHT +
                  SMALL_STEP_RIGHT + SMALL_STEP_RIGHT + SMALL_STEP_RIGHT)
    MIDDLE_STEP_RIGHT = SMALL_STEP_RIGHT + SMALL_STEP_RIGHT + SMALL_STEP_RIGHT
    SMALL_STEP_LEFT = [o.STEP_LEFT]
    STEP_LEFT = (SMALL_STEP_LEFT + SMALL_STEP_LEFT + SMALL_STEP_LEFT + SMALL_STEP_LEFT +
                 SMALL_STEP_LEFT + SMALL_STEP_LEFT + SMALL_STEP_LEFT)
    MIDDLE_STEP_LEFT = SMALL_STEP_LEFT + SMALL_STEP_LEFT + SMALL_STEP_LEFT

    SAVE = [o.SAVE]
    LOAD = [o.LOAD]

    FORCE_LEFT = [o.RUN_LEFT, o.RUN_LEFT, o.RUN_LEFT]
    FORCE_UP = UP + UP + UP + UP + UP + UP + UP + UP + UP + UP
    FORCE_DOWN = DOWN + DOWN + DOWN + DOWN + DOWN + DOWN + DOWN + DOWN + DOWN + DOWN

    BARRIER_PASS_LEFT = makeWait(110) + RUN_LEFT
    BARRIER_PASS_RIGHT = makeWait(110) + RUN_RIGHT

    R1_START_SKULL = [
        o.CLIMB_DOWN, o.CLIMB_DOWN, o.RUN_RIGHT, o.JUMP_RIGHT, o.JUMP_RIGHT, o.RUN_RIGHT,
        o.CLIMB_DOWN, o.CLIMB_DOWN, o.RUN_LEFT, o.JUMP_LEFT
    ]
    R1_SKULL_KEY = [o.RUN_LEFT, o.CLIMB_UP, o.CLIMB_UP, o.RUN_LEFT, o.JUMP]
    R1_KEY_SKULL = [o.RUN_RIGHT, o.CLIMB_DOWN, o.CLIMB_DOWN, o.RUN_RIGHT, o.JUMP_RIGHT]
    R1_SKULL_START = [
        o.RUN_RIGHT, o.CLIMB_UP, o.CLIMB_UP, o.RUN_LEFT, o.JUMP_LEFT, o.JUMP_LEFT, o.RUN_LEFT,
        o.CLIMB_UP, o.CLIMB_UP
    ]
    R1_START_RIGHT = [o.RUN_RIGHT, o.JUMP_RIGHT, o.RUN_RIGHT, o.JUMP_RIGHT, o.RUN_RIGHT]
    R1_START_LEFT = [o.RUN_LEFT, o.JUMP_LEFT, o.RUN_LEFT, o.JUMP_LEFT, o.RUN_LEFT]

    # NOTE: Rooms in this plan are 1-indexed, but rooms in the state are 0-indexed!
    ROOM_2 = R1_START_SKULL + R1_SKULL_KEY + R1_KEY_SKULL + R1_SKULL_START + R1_START_LEFT
    ROOM_1 = (RUN_LEFT + RUN_LEFT + BARRIER_PASS_LEFT + BARRIER_PASS_LEFT + RUN_LEFT + RUN_LEFT +
              makeWait(60) + RUN_LEFT + [o.JUMP] + makeWait(75) + RUN_RIGHT + DOWN)
    ROOM_5 = makeWait(100) + DOWN + RUN_LEFT + makeWait(230) + RUN_LEFT + JUMP_LEFT + RUN_LEFT
    ROOM_4 = makeWait(40) + RUN_LEFT + RUN_LEFT + DOWN
    ROOM_10 = (DOWN + DOWN + RUN_LEFT + RUN_LEFT + RUN_LEFT + JUMP_LEFT + RUN_LEFT + RUN_LEFT +
               RUN_LEFT + JUMP_LEFT + RUN_LEFT)
    ROOM_9 = (RUN_LEFT + JUMP_LEFT + FORCE_UP + UP + UP + JUMP + FORCE_DOWN + FORCE_DOWN +
              FORCE_DOWN + FORCE_DOWN + RUN_RIGHT + makeWait(50) + RUN_RIGHT + JUMP + JUMP + JUMP +
              makeWait(75) + JUMP_5_TIMES + RUN_RIGHT)
    ROOM_10_B = (STEP_RIGHT + JUMP_RIGHT + STEP_RIGHT + STEP_RIGHT + JUMP_RIGHT + STEP_RIGHT +
                 STEP_RIGHT + STEP_RIGHT + UP)
    ROOM_4_B = UP + RUN_RIGHT
    ROOM_5_B = RUN_RIGHT + RUN_RIGHT + JUMP_RIGHT + makeWait(5) + RUN_LEFT + DOWN
    ROOM_11 = (DOWN + makeWait(100) + DOWN + RUN_LEFT + JUMP_RIGHT + RUN_RIGHT + STEP_RIGHT +
               STEP_RIGHT + UP + DOWN + DOWN + makeWait(60) + DOWN + RUN_RIGHT)
    ROOM_12 = RUN_RIGHT + MIDDLE_STEP_RIGHT + JUMP_RIGHT + RUN_RIGHT + RUN_RIGHT + UP
    ROOM_6 = (UP + RUN_RIGHT + JUMP_RIGHT + STEP_RIGHT + MIDDLE_STEP_RIGHT + STEP_RIGHT + JUMP +
              FORCE_UP + JUMP_LEFT + RUN_LEFT + JUMP_LEFT + RUN_LEFT + JUMP_LEFT + UP + UP + UP +
              MIDDLE_STEP_RIGHT + JUMP_RIGHT + JUMP_RIGHT + STEP_RIGHT + STEP_RIGHT + STEP_RIGHT +
              STEP_RIGHT + JUMP_RIGHT + FORCE_DOWN + JUMP_LEFT + RUN_LEFT + DOWN)
    ROOM_12_B = DOWN + RUN_RIGHT + MIDDLE_STEP_RIGHT + JUMP_RIGHT + RUN_RIGHT
    ROOM_13 = (makeWait(20) + RUN_RIGHT + makeWait(75) + RUN_RIGHT + makeWait(110) + RUN_RIGHT +
               makeWait(115) + RUN_RIGHT + makeWait(110) + RUN_RIGHT)
    ROOM_14 = makeWait(100) + RUN_RIGHT + RUN_RIGHT + JUMP_RIGHT + RUN_RIGHT + UP + UP
    ROOM_8 = makeWait(130) + RUN_LEFT + makeWait(40) + RUN_LEFT + makeWait(110) + RUN_LEFT
    ROOM_7 = (RUN_LEFT + RUN_LEFT + STEP_LEFT + STEP_LEFT + RUN_LEFT + RUN_RIGHT + JUMP_RIGHT +
              RUN_RIGHT + STEP_RIGHT + STEP_RIGHT + RUN_RIGHT)
    ROOM_8_B = (makeWait(65) + RUN_RIGHT + BARRIER_PASS_RIGHT + RUN_RIGHT + makeWait(60) +
                RUN_RIGHT + JUMP + makeWait(80) + RUN_LEFT + DOWN)
    ROOM_14_B = DOWN + STEP_RIGHT + JUMP_RIGHT + JUMP_RIGHT + RUN_RIGHT
    ROOM_15 = (RUN_RIGHT + RUN_RIGHT + JUMP_RIGHT + JUMP_RIGHT + JUMP_LEFT + FORCE_DOWN +
               FORCE_DOWN + FORCE_DOWN + FORCE_DOWN + RUN_RIGHT + DOWN + DOWN)
    ROOM_23 = makeWait(70) + DOWN + RUN_RIGHT
    ROOM_24 = RUN_RIGHT + RUN_RIGHT + STEP_LEFT + JUMP_LEFT + STEP_LEFT + JUMP_LEFT + RUN_LEFT
    ROOM_23_B = (makeWait(120) + RUN_LEFT + [o.CLIMB_UP] + makeWait(100) + DOWN + RUN_LEFT +
                 MIDDLE_STEP_LEFT + JUMP_LEFT + RUN_LEFT)
    ROOM_22 = makeWait(60) + RUN_LEFT + JUMP_LEFT + UP
    ROOM_14_C = DOWN + RUN_LEFT
    ROOM_13_B = (makeWait(130) + RUN_LEFT + makeWait(20) + RUN_LEFT + makeWait(120) + RUN_LEFT +
                 makeWait(110) + RUN_LEFT + makeWait(110) + RUN_LEFT)
    ROOM_12_C = RUN_LEFT + RUN_LEFT + MIDDLE_STEP_LEFT + JUMP_LEFT + RUN_LEFT + DOWN + DOWN
    ROOM_20 = DOWN + RUN_RIGHT
    ROOM_21 = (makeWait(70) + RUN_RIGHT + JUMP_RIGHT + STEP_RIGHT + JUMP_RIGHT + RUN_RIGHT +
               makeWait(100) + JUMP_LEFT + JUMP_LEFT + RUN_LEFT)
    ROOM_20_B = (RUN_LEFT + RUN_LEFT + STEP_LEFT + STEP_LEFT + RUN_LEFT + RUN_RIGHT + JUMP_RIGHT +
                 RUN_LEFT)
    TO_END = (makeWait(40) + RUN_LEFT + JUMP_LEFT + RUN_LEFT + JUMP_LEFT + RUN_LEFT + JUMP_LEFT +
              RUN_LEFT + JUMP_LEFT + RUN_LEFT + RUN_LEFT + RUN_LEFT)

    END_A = (JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT + JUMP_LEFT + JUMP +
             JUMP_RIGHT + JUMP_LEFT + JUMP_RIGHT + JUMP_LEFT + JUMP_LEFT)
    END_B = JUMP_LEFT + JUMP_LEFT + JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT
    END = END_A + END_B

    BROOM_2 = ROOM_2
    BROOM_1 = (makeWait(30) + RUN_LEFT + makeWait(110) + RUN_LEFT + JUMP_LEFT + RUN_LEFT +
               makeWait(50) + RUN_LEFT + JUMP + makeWait(80) + RUN_RIGHT + DOWN)
    BROOM_5 = DOWN + RUN_LEFT + MIDDLE_STEP_LEFT + JUMP_LEFT + RUN_LEFT
    BROOM_4 = (RUN_LEFT + RUN_LEFT + JUMP_LEFT + RUN_LEFT + JUMP_RIGHT + JUMP_RIGHT + RUN_RIGHT +
               DOWN)
    BROOM_10 = DOWN + DOWN + RUN_LEFT + JUMP_LEFT
    BROOM_9 = (RUN_LEFT + RUN_LEFT + MIDDLE_STEP_RIGHT + JUMP_LEFT + FORCE_DOWN + FORCE_DOWN +
               RUN_LEFT + JUMP_LEFT + makeWait(60) + JUMP + JUMP + JUMP + makeWait(60) +
               JUMP_5_TIMES + RUN_RIGHT + MIDDLE_STEP_LEFT + SMALL_STEP_LEFT + JUMP_RIGHT +
               FORCE_DOWN + RUN_RIGHT + JUMP_RIGHT + JUMP + JUMP + JUMP + makeWait(30) +
               JUMP_5_TIMES + RUN_RIGHT)
    BROOM_10_B = RUN_RIGHT + RUN_RIGHT + JUMP_RIGHT + RUN_RIGHT + STEP_RIGHT + STEP_RIGHT + UP
    BROOM_4_B = UP + RUN_RIGHT
    BROOM_5_B = RUN_RIGHT + RUN_RIGHT + JUMP_RIGHT + RUN_RIGHT + STEP_RIGHT + STEP_RIGHT + DOWN
    BROOM_11 = (DOWN + DOWN + STEP_LEFT + STEP_LEFT + STEP_LEFT + STEP_LEFT + JUMP_LEFT +
                JUMP_LEFT + RUN_RIGHT + STEP_RIGHT + STEP_RIGHT + [o.CLIMB_UP] + makeWait(70) +
                DOWN + RUN_RIGHT)
    BROOM_12 = RUN_RIGHT + STEP_RIGHT + STEP_RIGHT + UP
    BROOM_6 = ROOM_6
    BROOM_12_B = DOWN + RUN_RIGHT
    BROOM_13 = (makeWait(20) + RUN_RIGHT + RUN_RIGHT + makeWait(105) + RUN_RIGHT +
                BARRIER_PASS_RIGHT + makeWait(10) + BARRIER_PASS_RIGHT + BARRIER_PASS_RIGHT)
    BROOM_14 = RUN_RIGHT + RUN_RIGHT + JUMP_RIGHT + RUN_RIGHT + STEP_RIGHT + STEP_RIGHT + UP
    BROOM_8 = (UP + makeWait(20) + RUN_RIGHT + makeWait(110) + RUN_RIGHT + JUMP + makeWait(75) +
               RUN_LEFT + RUN_LEFT + makeWait(70) + RUN_LEFT + BARRIER_PASS_LEFT)
    BROOM_7 = RUN_LEFT + RUN_LEFT + UP
    BROOM_3 = UP + RUN_RIGHT + JUMP_LEFT + RUN_LEFT + DOWN
    BROOM_7_B = DOWN + DOWN + RUN_RIGHT
    BROOM_8_B = makeWait(90) + RUN_RIGHT + BARRIER_PASS_RIGHT + DOWN
    BROOM_14_B = DOWN + RUN_RIGHT + JUMP_RIGHT + RUN_RIGHT
    BROOM_15 = (RUN_RIGHT + RUN_RIGHT + JUMP_RIGHT + FORCE_UP + FORCE_UP + STEP_RIGHT + JUMP +
                STEP_LEFT + FORCE_DOWN + FORCE_DOWN + FORCE_DOWN + FORCE_DOWN + RUN_RIGHT + DOWN)
    BROOM_23 = DOWN + DOWN + RUN_RIGHT
    BROOM_24 = RUN_RIGHT + RUN_RIGHT + JUMP_LEFT + JUMP_LEFT + RUN_LEFT
    BROOM_23_B = (makeWait(80) + RUN_LEFT + JUMP_LEFT + JUMP_LEFT + JUMP_LEFT + JUMP_LEFT +
                  JUMP_LEFT + RUN_LEFT)
    BROOM_22 = (RUN_LEFT + RUN_LEFT + JUMP_LEFT + RUN_LEFT + JUMP_RIGHT + RUN_RIGHT + STEP_RIGHT +
                STEP_RIGHT + UP)
    BROOM_14_C = UP + DOWN + RUN_LEFT + MIDDLE_STEP_LEFT + JUMP_LEFT + RUN_LEFT
    BROOM_13_B = (RUN_RIGHT + makeWait(20) + RUN_LEFT + makeWait(110) + RUN_LEFT + makeWait(110) +
                  RUN_LEFT + makeWait(115) + RUN_LEFT + makeWait(110) + RUN_LEFT)
    BROOM_12_C = RUN_LEFT + RUN_LEFT + DOWN + DOWN
    BROOM_20 = DOWN + RUN_RIGHT
    BROOM_21 = (makeWait(60) + RUN_RIGHT + JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT +
                JUMP_RIGHT + JUMP_LEFT + RUN_LEFT)
    BROOM_20_B = RUN_LEFT + RUN_LEFT + UP
    BROOM_12_D = (UP + DOWN + RUN_LEFT + RUN_RIGHT + JUMP_RIGHT + RUN_RIGHT + STEP_RIGHT +
                  STEP_RIGHT + DOWN)
    BROOM_20_C = DOWN + DOWN + RUN_LEFT
    BROOM_19 = makeWait(40) + JUMP_LEFT + JUMP_LEFT + RUN_LEFT + JUMP_LEFT + RUN_LEFT
    TO_BEND = RUN_LEFT + JUMP_LEFT + RUN_LEFT + JUMP_LEFT + RUN_LEFT + RUN_LEFT + RUN_LEFT
    BEND = (JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT +
            JUMP_LEFT + JUMP_LEFT + JUMP_LEFT + JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT +
            JUMP_LEFT + JUMP_LEFT + JUMP + JUMP + JUMP_RIGHT + JUMP + JUMP)

    CROOM_2 = (DOWN + RUN_RIGHT + JUMP_RIGHT + JUMP_RIGHT + RUN_RIGHT + DOWN + makeWait(30) +
               RUN_LEFT3 + JUMP_LEFT + RUN_LEFT + UP + RUN_LEFT + JUMP + RUN_RIGHT + DOWN +
               makeWait(10) + RUN_RIGHT3 + JUMP_RIGHT + RUN_RIGHT + UP + RUN_LEFT + JUMP_LEFT +
               JUMP_LEFT + RUN_LEFT + UP + RUN_LEFT + JUMP_LEFT + RUN_LEFT + JUMP_LEFT + RUN_LEFT)
    CROOM_1 = RUN_LEFT + makeWait(280) + RUN_LEFT + makeWait(105) + RUN_LEFT + DOWN
    CROOM_4 = (JUMP_LEFT + JUMP_LEFT + JUMP_LEFT + JUMP_LEFT + JUMP_LEFT + JUMP_LEFT + RUN_RIGHT +
               DOWN)
    CROOM_5 = (DOWN + RUN_LEFT + MIDDLE_STEP_LEFT + JUMP_LEFT + RUN_LEFT + MIDDLE_STEP_LEFT +
               JUMP_LEFT + RUN_LEFT)
    CROOM_10 = (DOWN + DOWN + DOWN + JUMP_LEFT + MIDDLE_STEP_LEFT + MIDDLE_STEP_LEFT +
                MIDDLE_STEP_LEFT + makeWait(10) + JUMP_LEFT + RUN_LEFT)
    CROOM_9 = (RUN_LEFT + RUN_LEFT + STEP_RIGHT + JUMP_LEFT + FORCE_DOWN + RUN_LEFT + JUMP_LEFT +
               JUMP + JUMP + JUMP + makeWait(40) + JUMP_5_TIMES + JUMP_RIGHT + RUN_RIGHT +
               STEP_LEFT + JUMP_RIGHT + DOWN + DOWN + RUN_RIGHT + JUMP_RIGHT + JUMP + JUMP + JUMP +
               makeWait(40) + JUMP_5_TIMES + RUN_RIGHT)
    CROOM_10_B = (MIDDLE_STEP_RIGHT + MIDDLE_STEP_RIGHT + MIDDLE_STEP_RIGHT + JUMP_RIGHT +
                  RUN_RIGHT + STEP_RIGHT + STEP_RIGHT + UP)
    CROOM_4_B = UP + RUN_RIGHT
    CROOM_5_B = (STEP_RIGHT + JUMP_RIGHT + RUN_RIGHT + JUMP_RIGHT + RUN_RIGHT + STEP_RIGHT +
                 STEP_RIGHT + DOWN)
    CROOM_11 = (makeWait(91) + DOWN + RUN_LEFT + JUMP_RIGHT + RUN_RIGHT + JUMP_RIGHT + RUN_RIGHT +
                JUMP_RIGHT + STEP_RIGHT + STEP_RIGHT + STEP_RIGHT + JUMP_RIGHT + RUN_RIGHT)
    CROOM_12 = makeWait(45) + RUN_RIGHT + RUN_RIGHT + UP
    CROOM_6 = (UP + RUN_RIGHT + JUMP_RIGHT + RUN_RIGHT + JUMP_LEFT + JUMP_LEFT + FORCE_UP +
               makeWait(30) + JUMP_LEFT + JUMP_LEFT + STEP_LEFT + JUMP_LEFT + RUN_LEFT +
               JUMP_LEFT + UP + UP + UP + MIDDLE_STEP_RIGHT + JUMP_RIGHT + JUMP_RIGHT +
               STEP_RIGHT + STEP_RIGHT + STEP_RIGHT + STEP_RIGHT + JUMP_RIGHT + FORCE_DOWN +
               JUMP_LEFT + RUN_LEFT + DOWN)
    CROOM_12_B = DOWN + DOWN
    CROOM_20_A = DOWN + DOWN + makeWait(80) + UP
    CROOM_12_BB = DOWN + RUN_RIGHT
    CROOM_13 = (RUN_RIGHT + RUN_RIGHT + makeWait(60) + RUN_RIGHT + makeWait(110) + RUN_RIGHT +
                makeWait(120) + RUN_RIGHT + makeWait(110) + RUN_RIGHT)
    CROOM_14 = RUN_RIGHT + RUN_RIGHT + UP
    CROOM_8 = UP + makeWait(130) + RUN_RIGHT + JUMP + makeWait(75) + RUN_LEFT + DOWN
    CROOM_14_B = DOWN + RUN_RIGHT
    CROOM_15 = ROOM_15
    CROOM_23 = makeWait(90) + DOWN + RUN_LEFT
    CROOM_22 = RUN_LEFT + RUN_LEFT + JUMP_LEFT + RUN_LEFT + UP
    CROOM_14_C = UP + DOWN + RUN_LEFT
    CROOM_13_B = (RUN_LEFT + makeWait(80) + RUN_LEFT + makeWait(115) + RUN_LEFT + makeWait(115) +
                  RUN_LEFT + makeWait(110) + RUN_LEFT)
    CROOM_12_C = makeWait(30) + RUN_LEFT + RUN_LEFT + DOWN
    CROOM_20 = DOWN + DOWN + RUN_RIGHT
    CROOM_21 = (makeWait(50) + RUN_RIGHT + JUMP_RIGHT + RUN_RIGHT + JUMP_LEFT + makeWait(90) +
                JUMP_LEFT + RUN_LEFT)
    CROOM_20_B = RUN_LEFT + RUN_LEFT + JUMP_LEFT + RUN_LEFT
    CROOM_19 = (makeWait(90) + JUMP_LEFT + JUMP_LEFT + JUMP_LEFT + JUMP_LEFT + JUMP_LEFT +
                MIDDLE_STEP_LEFT + MIDDLE_STEP_LEFT + JUMP_LEFT + RUN_LEFT)
    TO_CEND = RUN_LEFT + JUMP_LEFT + RUN_LEFT + JUMP_LEFT + RUN_LEFT + RUN_LEFT + RUN_LEFT
    CEND = (JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT +
            JUMP_RIGHT + JUMP_LEFT + JUMP_LEFT + JUMP_LEFT + JUMP_RIGHT + JUMP_LEFT + JUMP_RIGHT +
            JUMP + JUMP_LEFT + JUMP_RIGHT + JUMP_RIGHT + JUMP_RIGHT + JUMP_LEFT + JUMP_LEFT)

    LEVEL_1 = (ROOM_2 + ROOM_1 + ROOM_5 + ROOM_4 + ROOM_10 + ROOM_9 + ROOM_10_B + ROOM_4_B +
               ROOM_5_B + ROOM_11 + ROOM_12 + ROOM_6 + ROOM_12_B + ROOM_13 + ROOM_14 + ROOM_8 +
               ROOM_7 + ROOM_8_B + ROOM_14_B + ROOM_15 + ROOM_23 + ROOM_24 + ROOM_23_B + ROOM_22 +
               ROOM_14_C + ROOM_13_B + ROOM_12_C + ROOM_20 + ROOM_21 + ROOM_20_B + TO_END + END)

    LEVEL_2 = (BROOM_2 + BROOM_1 + BROOM_5 + BROOM_4 + BROOM_10 + BROOM_9 + BROOM_10_B +
               BROOM_4_B + BROOM_5_B + BROOM_11 + BROOM_12 + BROOM_6 + BROOM_12_B + BROOM_13 +
               BROOM_14 + BROOM_8 + BROOM_7 + BROOM_3 + BROOM_7_B + BROOM_8_B + BROOM_14_B +
               BROOM_15 + BROOM_23 + BROOM_24 + BROOM_23_B + BROOM_22 + BROOM_14_C + BROOM_13_B +
               BROOM_12_C + BROOM_20 + BROOM_21 + BROOM_20_B + BROOM_12_D + BROOM_20_C + BROOM_19 +
               TO_BEND + BEND)

    LEVEL_3 = (CROOM_2 + CROOM_1 + CROOM_5 + CROOM_4 + CROOM_10 + CROOM_9 + CROOM_10_B +
               CROOM_4_B + CROOM_5_B + CROOM_11 + CROOM_12 + CROOM_6 + CROOM_12_B + CROOM_20_A +
               CROOM_12_BB + CROOM_13 + CROOM_14 + CROOM_8 + CROOM_14_B + CROOM_15 + CROOM_23 +
               CROOM_22 + CROOM_14_C + CROOM_13_B + CROOM_12_C + CROOM_20 + CROOM_21 + CROOM_20_B +
               CROOM_19 + TO_CEND + CEND)

    FULL_RUN = LEVEL_1 + LEVEL_2 + LEVEL_3
    FROM_SAVE = LOAD + CEND + SAVE

    safe_starts = [
        S0, NW, NE, MID, E, SE, SW, W, S0_KEY, NW_KEY, NE_KEY, MID_KEY, E_KEY, SE_KEY, SW_KEY,
        W_KEY
    ]
    death_starts = [
        S0_SKULL,
        NW_SKULL,
        NE_SKULL,
        MID_SKULL,
        E_SKULL,
        SE_SKULL,
        SW_SKULL,
        W_SKULL,
        S0_BOTH,
        NW_BOTH,
        NE_BOTH,
        MID_BOTH,
        E_BOTH,
        SE_BOTH,
        SW_BOTH,
        W_BOTH,
    ]
    all_starts = safe_starts + death_starts

    base_plan = [o.NONE]

    n_lives = 5
    n_starts = n_lives * len(all_starts) + len(safe_starts)

    @classmethod
    def GetFullRunPlan(cls):
        """Choose the hand-coded full run plan

        Returns the plan as a collections.deque
        """
        return deque(cls.FULL_RUN)

    @classmethod
    def GetStartByIndex(cls, i):
        """Choose the initial plan associated with index 'i'

        Returns the plan as a collections.deque
        """
        assert i >= 0 and i < cls.n_starts
        n_deaths = int(i // len(cls.all_starts))
        i %= len(cls.all_starts)
        p = cls.base_plan
        p += cls.all_starts[i]
        return deque(p)

    @classmethod
    def RandomStart(cls):
        """Choose a random initial plan and return it as a collections.deque
        """
        i = np.random.randint(0, Plans.n_starts)
        return cls.GetStartByIndex(i)

    @classmethod
    def GetDefaultStart(cls):
        return cls.GetStartByIndex(0)

    @classmethod
    def WrapPlanSteps(cls, plan_steps):
        return deque(cls.base_plan + plan_steps)

    @classmethod
    def ToRoomNumber(cls, room_number):
        # NOTE: room_number argument is 0-indexed
        # but the implementation is 1-indexed for legacy reasons
        translated_room_number = room_number + 1

        # Here and below, (parentheses) denote multiple room plans that we combine together, since
        # some don't reach a new room on their own. We only care about reaching specific rooms in
        # this function.
        # yapf: disable
        rooms_with_existing_plans = [
            2, 1, 5, 4, 10, 9,
            (11),
            12, 6,
            (13),
            14, 8, 7,
            (15),
            23, 24,
            (22),
            20,
            (21),
            19,
            16
        ]
        partial_plan_to_subsequent_rooms = [
            [],
            cls.ROOM_2, cls.ROOM_1, cls.ROOM_5, cls.ROOM_4, cls.ROOM_10,
            (cls.ROOM_9 + cls.ROOM_10_B + cls.ROOM_4_B + cls.ROOM_5_B),
            cls.ROOM_11, cls.ROOM_12,
            (cls.ROOM_6 + cls.ROOM_12_B),
            cls.ROOM_13, cls.ROOM_14, cls.ROOM_8,
            (cls.ROOM_7 + cls.ROOM_8_B + cls.ROOM_14_B),
            cls.ROOM_15, cls.ROOM_23,
            cls.ROOM_24 + cls.ROOM_23_B,
            (cls.ROOM_22 + cls.ROOM_14_C + cls.ROOM_13_B + cls.ROOM_12_C),
            cls.ROOM_20,
            (cls.ROOM_21 + cls.ROOM_20_B),
            cls.TO_END,
            []
        ]
        # yapf: enable

        if translated_room_number in rooms_with_existing_plans:
            idx = rooms_with_existing_plans.index(translated_room_number)
            plan_fragments = partial_plan_to_subsequent_rooms[:idx + 1]
            combined_plan = sum(plan_fragments, [])
            return deque(combined_plan)

        raise NotImplementedError('Room {} does not have an existing plan'.format(room_number))

    @classmethod
    def ToSpiderRoom(cls):
        return cls.ToRoomNumber(4)

    @classmethod
    def ToSpiderRoomFloor(cls):
        plan = cls.ToRoomNumber(4)
        [plan.extend(item) for item in [makeWait(100) + cls.DOWN]]
        return plan

    @classmethod
    def ToJumpSkullRoom(cls):
        return cls.ToRoomNumber(3)

    @classmethod
    def ToJumpSkullRoomLadder(cls):
        plan = cls.ToRoomNumber(3)
        plan += makeWait(40) + cls.RUN_LEFT + cls.RUN_LEFT + cls.DOWN + [o.CLIMB_UP] * 2
        return plan

    @classmethod
    def ToJumpSkullRoomLadder2(cls):
        plan = deque([o.NONE] + Plans.NE_KEY + [o.JUMP_RIGHT] * 4)
        return plan

    @classmethod
    def ToStartingRoom(cls):
        return deque([o.NONE] * 5)

    @classmethod
    def ToStartingRoomBottom(cls):
        return deque(cls.S0_TO_SE) + deque(cls.RUN_RIGHT)

    @classmethod
    def ToTorchRoom(cls):
        plan = cls.ToRoomNumber(5)
        return plan

    @classmethod
    def ToLaserRoom(cls, location='laser_screen-0_right'):

        # Room 0
        s0_r = cls.ToRoomNumber(0)
        s0_l = cls.ToRoomNumber(0)
        for item in [makeWait(110) + cls.RUN_LEFT]:
            s0_l.extend(item)
        s0_m = cls.ToRoomNumber(0)
        for item in [makeWait(110) + cls.RUN_LEFT + makeWait(110) + cls.RUN_LEFT]:
            s0_m.extend(item)

        # Room 7
        s7_m = cls.ToRoomNumber(7)
        for item in [cls.UP]:
            s7_m.extend(item)

        # Room 12
        s12_l = cls.ToRoomNumber(12)
        s12_r = cls.ToRoomNumber(12)
        for item in [makeWait(110) + cls.RUN_RIGHT]:
            s12_r.extend(item)

        valid_locations = {
            'laser_screen-0_right': s0_r,
            'laser_screen-0_left': s0_l,
            'laser_screen-0_middle': s0_m,
            'laser_screen-7_middle': s7_m,
            'laser_screen-12_right': s12_r,
            'laser_screen-12_left': s12_l,
        }
        if location not in valid_locations:
            raise ValueError("No such location {} in platform room".format(location))
        return valid_locations[location]

    @classmethod
    def ToPlatformRoom(cls, location='enter'):

        ENTER_ROOM = cls.ToRoomNumber(8)
        TOP_RIGHT_EDGE = ENTER_ROOM + deque(cls.RUN_LEFT)
        ROPE = TOP_RIGHT_EDGE + deque(cls.JUMP_LEFT)
        TOP_CENTER = ROPE + deque(cls.FORCE_UP + cls.UP + cls.UP + cls.JUMP)
        BOTTOM_CENTER = TOP_CENTER + deque(cls.FORCE_DOWN + cls.FORCE_DOWN + cls.FORCE_DOWN +
                                           cls.FORCE_DOWN)
        BOTTOM_LEFT = BOTTOM_CENTER + deque(cls.RUN_LEFT + makeWait(50) + cls.RUN_LEFT)
        BOTTOM_RIGHT = BOTTOM_CENTER + deque(cls.RUN_RIGHT + makeWait(50) + cls.RUN_RIGHT)
        TOP_LEFT = BOTTOM_LEFT + deque(cls.JUMP + cls.JUMP + cls.JUMP + makeWait(75) +
                                       cls.JUMP_5_TIMES)
        TOP_LEFT_EDGE = TOP_LEFT + deque(cls.RUN_RIGHT)
        TOP_RIGHT = BOTTOM_RIGHT + deque(cls.JUMP + cls.JUMP + cls.JUMP + makeWait(75) +
                                         cls.JUMP_5_TIMES)
        valid_locations = {
            'enter': ENTER_ROOM,
            'top-right-edge': TOP_RIGHT_EDGE,
            'rope': ROPE,
            'top-center': TOP_CENTER,
            'bottom-center': BOTTOM_CENTER,
            'bottom-left': BOTTOM_LEFT,
            'bottom-right': BOTTOM_RIGHT,
            'top-left': TOP_LEFT,
            # below are unused for now
            'top-left-edge': TOP_LEFT_EDGE,
            'top-right': TOP_RIGHT,
        }
        if location not in valid_locations:
            raise ValueError("No such location {} in platform room".format(location))
        return valid_locations[location]

    @classmethod
    def ToTorchRoom(cls):
        plan = cls.ToRoomNumber(5)
        plan.extend([o.CLIMB_UP, o.CLIMB_UP, o.CLIMB_UP])
        return plan

    @classmethod
    def ToBridgeRoom(cls):
        return cls.ToRoomNumber(10)
