from collections import defaultdict

from ..ataritools.actions import actions

class Skill:
    SCREEN_TOP = 0xfe
    UPPER_LEVEL = 0xeb
    MIDDLE_LEVEL_1 = 0xc0
    MIDDLE_LEVEL_5 = 0xc6
    LOWER_LEVEL_1 = 0x94
    LOWER_LEVEL_5 = 0x9d
    LOWER_LEVEL_8 = 0x9b
    LOWER_LEVEL_14 = 0xa0
    SCREEN_BOTTOM = 0x86
    JUMP_DISTANCE = 19  # pixels

    def can_run(self, state):
        raise NotImplementedError

    def policy(self, state):
        raise NotImplementedError

    def is_done(self, state):
        raise NotImplementedError

    def get_object_x_positions(self, state):
        if state['object_type'] == 'none':
            return []
        obj_config = state['object_configuration']
        dist_str = obj_config.split('_')[1]
        obj_dist = defaultdict(lambda: 0, {
                'near': 0x10,
                'mid': 0x20,
                'far': 0x40,
                # 'double': 0x05,
                # 'triple': 0x05,
            })[dist_str]  # yapf: disable
        n_objects = {'one': 1, 'two': 2, 'three': 3}[obj_config.split('_')[0]]
        # if dist_str == 'double':
        #     n_objects = 2
        # elif dist_str == 'triple':
        #     n_objects = 3
        object_positions = [(state['object_x'] + obj_dist * i) for i in range(n_objects)]
        return object_positions

    def get_object_y_position(self, state):
        y_position = defaultdict(
            lambda: self.UPPER_LEVEL, {
                1: self.MIDDLE_LEVEL_1,
                14: self.MIDDLE_LEVEL_1 if state['level'] != 1 else self.UPPER_LEVEL,
            })[state['screen']]
        return y_position

    def at_object(self, state, object_x=None):
        if object_x is None: object_x = state['object_x']
        return object_x == state['player_x']

    def near_object(self, state, object_x=None, margin=None):
        if object_x is None: object_x = state['object_x']
        if margin is None: margin = self.object_margin
        return abs(object_x - state['player_x']) <= margin

    def left_of_object(self, state, object_x=None, margin=None):
        if object_x is None: object_x = state['object_x']
        if margin is None: margin = self.object_margin
        return object_x - margin > state['player_x']

    def right_of_object(self, state, object_x=None, margin=None):
        if object_x is None: object_x = state['object_x']
        if margin is None: margin = self.object_margin
        return object_x + margin < state['player_x']

    def for_all_objects(self, state, fn):
        return all(map(fn, self.get_object_x_positions(state)))

    def for_any_objects(self, state, fn):
        return any(map(fn, self.get_object_x_positions(state)))

    def player_static(self, state):
        return not (
            state['player_jumping']
            or state['player_status'] in ['mid-air']
            or state['player_falling']
            or state['respawning']
        )# yapf: disable

    def level_changing(self, state):
        return state['screen'] == 15 and state['player_y'] < self.UPPER_LEVEL

    def has_tall_bottom_ladder(self, screen):
        return screen in [0, 2, 3, 4, 7, 11, 13]

    def has_short_top_ladder(self, screen):
        return screen in [4, 6, 9, 10, 11, 13, 19, 21, 22]

    def get_platforms(self, state):
        screen = state['screen']
        platforms = defaultdict(list)
        if screen == 1:
            platforms.update({
                self.UPPER_LEVEL: [(0x00, 0x05), (0x05, 0x33), (0x43, 0x4d), (0x4d, 0x55),
                                   (0x66, 0x95), (0x95, 0x99)],
                self.MIDDLE_LEVEL_1: [(0x09, 0x15), (0x15, 0x20), (0x38, 0x4d), (0x39, 0x4d),
                                      (0x4d, 0x61), (0x4e, 0x61), (0x7b, 0x85), (0x85, 0x91)],
                # Running left on the treadmill changes x position by 2 every other step, so we need
                # to overlap the platforms so the left ends of the treadmill sections span 2px each.
                self.LOWER_LEVEL_1: [(0x15, 0x85)],
            })
        elif screen == 5:
            platforms.update({
                self.UPPER_LEVEL: [(0x00, 0x05), (0x05, 0x1a), (0x24, 0x2d), (0x38, 0x4d),
                                   (0x39, 0x4d), (0x4d, 0x64), (0x4e, 0x64), (0x6d, 0x76),
                                   (0x80, 0x95), (0x95, 0x99)],
                # Running left on the treadmill changes x position by 2 every other step, so we need
                # to overlap the platforms so the left ends of the treadmill sections span 2px each.
                self.MIDDLE_LEVEL_5: [(0x2d, 0x6d)],
                self.LOWER_LEVEL_5: [(0x05, 0x4d), (0x4d, 0x7c), (0x7c, 0x95)],
            })
        elif screen == 8:
            platforms.update({
                self.UPPER_LEVEL: [(0x05, 0x2d), (0x48, 0x51), (0x6c, 0x95), (0x95, 0x99)],
                self.LOWER_LEVEL_8: [(0x05, 0x95)],
            })
        elif screen == 14:
            platforms.update({
                self.UPPER_LEVEL: [(0x00, 0x05), (0x05, 0x25), (0x40, 0x59)],
                self.LOWER_LEVEL_14: [(0x11, 0x44), (0x44, 0x4d), (0x4d, 0x89)],
            })
        elif state['has_lasers']:
            if screen in [0, 7]:
                platforms.update({
                    self.UPPER_LEVEL: [(0x00, 0x05), (0x05, 0x0a), (0x13, 0x1e), (0x2f, 0x4d),
                                       (0x4d, 0x6a), (0x7b, 0x86), (0x91, 0x95), (0x95, 0x99)],
                })
            else:  # screen == 12
                platforms.update({
                    self.UPPER_LEVEL: [(0x00, 0x05), (0x05, 0x1e), (0x2f, 0x36), (0x47, 0x52),
                                       (0x63, 0x6a), (0x7b, 0x95), (0x95, 0x99)],
                })
        elif state['has_bridge'] and state['has_ladder']:
            # yapf: disable
            platforms.update({
                self.UPPER_LEVEL: [(0x00, 0x05), (0x05, 0x21), (0x21, 0x21 + self.JUMP_DISTANCE),
                                   (0x21 + self.JUMP_DISTANCE, 0x4d), (0x4d, 0x79 - self.JUMP_DISTANCE),
                                   (0x79 - self.JUMP_DISTANCE, 0x79), (0x79, 0x95), (0x95, 0x99)],
            })
        elif state['has_bridge']:
            platforms.update({
                self.UPPER_LEVEL: [(0x00, 0x05), (0x05, 0x21), (0x21, 0x21 + self.JUMP_DISTANCE),
                                   (0x21 + self.JUMP_DISTANCE, 0x79 - self.JUMP_DISTANCE),
                                   (0x79 - self.JUMP_DISTANCE, 0x79), (0x79, 0x95), (0x95, 0x99)],
            })
            # yapf: enable
        elif state['has_ladder']:  # excluding special cases above
            platforms.update({
                self.UPPER_LEVEL: [(0x00, 0x05), (0x05, 0x4d), (0x4d, 0x95), (0x95, 0x99)],
            })
        else:
            platforms.update({
                self.UPPER_LEVEL: [(0x00, 0x05), (0x05, 0x95), (0x95, 0x99)],
            })
        return platforms

class _MovingEnemySkill(Skill):
    object_margin = 12
    object_type = None  #'has_spider' or 'has_skull' or 'has_jump_skull'
    object_x = 'object_x'
    object_dir = 'object_dir'

    def out_of_path(self, state):
        return (state['player_y'] > 0xf5 or state['player_y'] < 0xd9)

    def valid_context(self, state):
        return (state[self.object_type] and self.player_static(state))

class _WaitForEnemyMovingTowards(_MovingEnemySkill):
    def can_run(self, state):
        if not self.valid_context(state):
            return False

        if ((self.right_of_object(state, object_x=state[self.object_x], margin=0)
             and state[self.object_dir] == 'left')
                or (self.left_of_object(state, object_x=state[self.object_x], margin=0)
                    and state[self.object_dir] == 'right') or
            (self.near_object(state, object_x=state[self.object_x]) and self.out_of_path(state))):
            # Safely moving past or away from player
            return True
        return False

    def policy(self, state):
        return actions.NOOP

    def is_done(self, state):
        if ((self.right_of_object(state, object_x=state[self.object_x])
             and state[self.object_dir] == 'right')
                or (self.left_of_object(state, object_x=state[self.object_x])
                    and state[self.object_dir] == 'left')):
            # Moving towards player
            return True
        # invalid context
        if not state[self.object_type]:
            return True
        return False

class _WaitForEnemyMovingAway(_MovingEnemySkill):
    def can_run(self, state):
        if not self.valid_context(state):
            return False
        if (
            self.out_of_path(state)
            and (
                (self.right_of_object(state, object_x=state[self.object_x],margin=0) and state[self.object_dir] == 'right')
                or (self.left_of_object(state, object_x=state[self.object_x],margin=0) and state[self.object_dir] == 'left')
                or self.near_object(state)
            )
        ):  # yapf: disable
            # Safely moving past or towards player
            return True
        return False

    def policy(self, state):
        return actions.NOOP

    def is_done(self, state):
        if ((self.right_of_object(state, object_x=state[self.object_x])
             and state[self.object_dir] == 'left')
                or (self.left_of_object(state, object_x=state[self.object_x])
                    and state[self.object_dir] == 'right')):
            # Moving away from player
            return True
        # invalid context
        if not state[self.object_type]:
            return True
        return False

class WaitForSpiderMovingTowards(_WaitForEnemyMovingTowards):
    object_type = 'has_spider'

class WaitForSpiderMovingAway(_WaitForEnemyMovingAway):
    object_type = 'has_spider'

class WaitForSkullMovingTowards(_WaitForEnemyMovingTowards):
    object_type = 'has_skull'
    object_x = 'skull_x'
    object_dir = 'skull_dir'

    def out_of_path(self, state):
        return abs(state['player_y'] - state['object_y']) > 15

class WaitForSkullMovingAway(_WaitForEnemyMovingAway):
    object_type = 'has_skull'
    object_x = 'skull_x'
    object_dir = 'skull_dir'

    def out_of_path(self, state):
        return abs(state['player_y'] - state['object_y']) > 15

class WaitForJumpSkullMovingTowards(_WaitForEnemyMovingTowards):
    object_type = 'has_jump_skull'

class WaitForJumpSkullMovingAway(_WaitForEnemyMovingAway):
    object_type = 'has_jump_skull'

class _JumpSkullSkill(Skill):
    """
    Players runs until "holding range"
    Stays in holding range until skull is high enough
    Then passes under the skulls

    Ranges are different because the skulls are not centered on object_x
    """
    object_margin = 12
    SKULL_PASSABLE_HEIGHT = 250
    PLAYER_TOO_FAR = None  #number distance
    PLAYER_TOO_CLOSE = None  #number distance

    def out_of_path(self, state):
        return state[
            'player_y'] < self.UPPER_LEVEL - self.object_margin  #Below the floor but on the ladder

    def valid_context(self, state):
        return state['has_jump_skull'] and self.player_static(state)

    def has_passed_all_skulls(self, state):
        if self.for_all_objects(state, fn=lambda x: self.has_passed_skull(state, x, margin=0)):
            return True

    def has_passed_any_skulls(self, state):
        if self.for_any_objects(state, fn=lambda x: self.has_passed_skull(state, x, margin=0)):
            return True

    def policy(self, state):
        margin = 0
        pass_height = self.SKULL_PASSABLE_HEIGHT
        if state['level'] == 2:
            margin = -5
        if self.has_passed_all_skulls(state):
            return self.direction_toward
        if (abs(state['player_x'] - state['object_x']) > self.PLAYER_TOO_FAR + margin):
            return self.direction_toward
        if state['object_y'] < pass_height or (state['object_vertical_dir'] != 'up'
                                               and not self.has_passed_any_skulls(state)):
            if (abs(state['player_x'] - state['object_x']) < self.PLAYER_TOO_CLOSE + margin):
                #step back if too close
                return self.direction_away
            #Stop running if skull is at player level
            return actions.NOOP
        #run under the skull when it goes up
        return self.direction_toward

    def can_run(self, state):
        if not self.valid_context(state):
            return False
        if not self.has_passed_all_skulls(state) and state['object_dir'] == self.towards_player:
            return True
        return False

    def is_done(self, state):
        if self.has_passed_all_skulls(state):
            return True
        return False

class PassToLeftOfJumpSkull(_JumpSkullSkill):
    PLAYER_TOO_FAR = 40
    PLAYER_TOO_CLOSE = 25

    direction_toward = actions.LEFT
    direction_away = actions.RIGHT
    has_passed_skull = _JumpSkullSkill.left_of_object
    towards_player = 'right'

class PassToRightOfJumpSkull(_JumpSkullSkill):
    PLAYER_TOO_FAR = 30
    PLAYER_TOO_CLOSE = 15

    direction_toward = actions.RIGHT
    direction_away = actions.LEFT
    has_passed_skull = _JumpSkullSkill.right_of_object
    towards_player = 'left'

class _LaserSkill(Skill):
    outer_margin = 12
    inner_margin = 3
    width = 10

    laser_pos = {
        'screen_0': [
            [140, Skill.UPPER_LEVEL],
            [119, Skill.UPPER_LEVEL],
            [111, Skill.UPPER_LEVEL],
            [43, Skill.UPPER_LEVEL],
            [35, Skill.UPPER_LEVEL],
            [14, Skill.UPPER_LEVEL],
        ],
        'screen_12': [
            [119, Skill.UPPER_LEVEL],
            [111, Skill.UPPER_LEVEL],
            [95, Skill.UPPER_LEVEL],
            [87, Skill.UPPER_LEVEL],
            [66, Skill.UPPER_LEVEL],
            [58, Skill.UPPER_LEVEL],
            [43, Skill.UPPER_LEVEL],
            [35, Skill.UPPER_LEVEL],
        ],
    }
    laser_pos['screen_7'] = laser_pos['screen_0']

    def valid_context(self, state):
        return (state['has_lasers'] and self.player_static(state))

    def laser_to_left(self, state):
        coords = self.laser_pos['screen_{}'.format(state['screen'])]
        return any([(state['player_x'] - self.outer_margin) <= x
                   and (state['player_x'] - self.inner_margin) >= x
                   and state['player_y'] == y for x,y in coords]) # yapf: disable

    def laser_to_right(self, state):
        coords = self.laser_pos['screen_{}'.format(state['screen'])]
        return any([(state['player_x'] + self.outer_margin) >= x
                   and (state['player_x'] + self.inner_margin) <= x
                   and state['player_y'] == y for x,y in coords]) # yapf: disable

    def can_run(self, state):
        if not self.valid_context(state):
            return False
        can_run = self.laser_to_right(state) or self.laser_to_left(state)
        can_run = can_run and state[self.waiting_for] > 0
        return can_run

    def policy(self, state):
        return actions.NOOP

    def is_done(self, state):
        return state[self.waiting_for] == 0

class WaitForLaserToDisappear(_LaserSkill):
    waiting_for = 'time_to_disappear'

class WaitForLaserToAppear(_LaserSkill):
    waiting_for = 'time_to_appear'

class _BridgeSkill(Skill):
    bridge_margin = 3
    bridge_left = 0x22
    bridge_center = 0x4d
    bridge_right = 0x78

    def at_bridge_level(self, state):
        return (state['player_y'] == self.UPPER_LEVEL)

    def above_bridge(self, state):
        return state['player_y'] > self.UPPER_LEVEL

    def left_of_bridge(self, state):
        return state['player_x'] < self.bridge_left

    def right_of_bridge(self, state):
        return state['player_x'] > self.bridge_right

    def on_bridge(self, state):
        return (self.player_static(state) and state['player_y'] == self.UPPER_LEVEL
                and self.bridge_left <= state['player_x'] <= self.bridge_right)

    def near_bridge(self, state):
        return (abs(state['player_y'] - self.UPPER_LEVEL) <= self.bridge_margin
                and (self.bridge_left - self.bridge_margin <= state['player_x']
                     and state['player_x'] <= self.bridge_right + self.bridge_margin))

    def valid_context(self, state):
        return (state['has_bridge'] and self.player_static(state) and not self.on_bridge(state))

    def can_run(self, state):
        if not self.valid_context(state):
            return False
        if state[self.waiting_for] > 0:
            return True
        return False

    def policy(self, state):
        return actions.NOOP

    def is_done(self, state):
        if state['respawning']:
            return True
        if state[self.waiting_for] == 0:
            return True
        return False

class WaitForBridgeToAppear(_BridgeSkill):
    waiting_for = 'time_to_appear'

class WaitForBridgeToDisappear(_BridgeSkill):
    waiting_for = 'time_to_disappear'

class ClimbPlatforms(Skill):
    """
    This skill is for climbing the disappearing platforms (there are two) on screen 8.
    The skill gets us from the ground floor to the top of the screen. It doesn't stop executing at
    halfway, because you can't actually move then - you just have to keep climbing. So we wait for
    the stairs to appear, then climb to the middle. Then wait again and climb to the top
    """
    def __init__(self):
        self.HALFWAY = 0xb9

        self.LEFT_PLATFORM_X_RANGE = (0x05, 0x09)
        self.CENTER_PLATFORM_X_RANGE = (0x48, 0x51)
        self.RIGHT_PLATFORM_X_RANGE = (0x91, 0x95)
        self.ROPE_X = 0x4d
        self.TOP_RIGHT_X_RANGE = (0x69, 0x79)

    def can_run(self, state):
        # make sure in right room and on ground
        if not (state['has_platforms'] and self.player_static(state)):
            return False

        # check locations
        if state['player_y'] != self.LOWER_LEVEL_8:
            # we aren't on the ground level
            return False
        # valid x-pos: [05 - 09] and [91-95]
        if not (
            (self.LEFT_PLATFORM_X_RANGE[0] <= state['player_x'] <= self.LEFT_PLATFORM_X_RANGE[1])
            or (self.RIGHT_PLATFORM_X_RANGE[0] <= state['player_x'] <= self.RIGHT_PLATFORM_X_RANGE[1])
        ):# yapf:disable
            return False

        # if we're in position, we can execute (regardless if stairs are present)
        return True

    def policy(self, state):
        if not self.player_static(state):
            # player is jumping, so do nothing
            return actions.NOOP

        if state['time_to_appear'] > 0:
            # platforms not yet there. Wait.
            return actions.NOOP

        if state['player_y'] not in [self.LOWER_LEVEL_8, self.HALFWAY, self.UPPER_LEVEL]:
            # we're on the steps! So just keep jumping!
            return actions.FIRE  # fire = jump

        if state['time_to_disappear'] < 80:  # TODO may need to tune for levels 2 and 3
            # potentially not enough time, so wait
            return actions.NOOP

        # we're on the ground or mid level and now need to ascend!
        return actions.FIRE  # fire = jump

    def is_done(self, state):
        # make sure on ground
        if not self.player_static(state):
            # player in middle of jumping. So cannot be done just yet
            return False
        # check if we're back at the top
        return state['player_y'] == self.UPPER_LEVEL

class _JumpSkill(Skill):
    jump_action = None

    def can_run(self, state):
        if self.level_changing(state):
            return False
        return self.player_static(state)

    def policy(self, state):
        if self.player_static(state):
            return self.jump_action
        else:
            return actions.NOOP

    def is_done(self, state):
        if state['respawning']:
            return True
        if self.level_changing(state):
            return True
        return self.player_static(state)

class JumpLeftSkill(_JumpSkill):
    jump_action = actions.LEFT_FIRE

class JumpRightSkill(_JumpSkill):
    jump_action = actions.RIGHT_FIRE

class JumpInPlaceSkill(_JumpSkill):
    jump_action = actions.FIRE

    def can_run(self, state):
        return (super().can_run(state)
                and (state['player_status'] not in ['on-rope', 'climbing-rope']
                     or state['player_y'] == self.UPPER_LEVEL))

class _ClimbSkill(Skill):
    climb_margin = 4  # number of pixels left or right of region center where climbing is possible

    # Ladder x_centers
    LEFT_LADDER = 0x14
    CENTER_LADDER = 0x4d
    RIGHT_LADDER = 0x85

    # Rope regions by screen: x_center, (y_min, y_max)
    ROPE_1 = 0x6d, (0xae, 0xd2)  # true y_max is 0xd4, but jumping above 0xd2 kills you
    ROPE_5_LEFT = 0x26, (0xbb, Skill.UPPER_LEVEL)
    ROPE_5_RIGHT = 0x7c, (0xa2, 0xd4)
    ROPE_5_RIGHT_ALT = 0x7c, (0xa2, 0xd8)
    ROPE_8 = 0x4d, (0xa2, Skill.UPPER_LEVEL)
    ROPE_14_LEFT = 0x44, (0xa5, Skill.UPPER_LEVEL)
    ROPE_14_RIGHT = 0x55, (0xb5, Skill.UPPER_LEVEL)

    def get_climb_regions(self, state):
        raise NotImplementedError

    def at_any_climb_region(self, state):
        for x_center, ylim in self.get_climb_regions(state):
            if (ylim[0] <= state['player_y'] <= ylim[1]
                    and abs(state['player_x'] - x_center) <= self.climb_margin):
                return True
        return False

    def above_any_climb_region(self, state):
        for x_center, ylim in self.get_climb_regions(state):
            if (state['player_y'] == ylim[1]
                    and abs(state['player_x'] - x_center) <= self.climb_margin):
                return True
        return False

    def below_any_climb_region(self, state):
        for x_center, ylim in self.get_climb_regions(state):
            if (state['player_y'] == ylim[0]
                    and abs(state['player_x'] - x_center) <= self.climb_margin):
                return True
        return False

    def policy(self, state):
        if state['screen_changing']:
            return actions.NOOP
        return self.climb_action

    def is_done(self, state):
        if state['screen_changing']:
            return False
        if state['respawning']:
            return True
        return self.player_static(state) and self.at_exit(state)

class _ClimbLadderSkill(_ClimbSkill):
    climb_margin_y = 6

    def get_climb_regions(self, state):
        regions = []
        # start-screen ladders
        if state['screen'] == 1:
            regions.append((self.CENTER_LADDER, (self.MIDDLE_LEVEL_1, self.UPPER_LEVEL)))
            regions.append((self.LEFT_LADDER, (self.LOWER_LEVEL_1, self.MIDDLE_LEVEL_1)))
            regions.append((self.RIGHT_LADDER, (self.LOWER_LEVEL_1, self.MIDDLE_LEVEL_1)))
        if state['screen'] == 5:
            regions.append((self.CENTER_LADDER, (self.SCREEN_BOTTOM, self.LOWER_LEVEL_5)))
            regions.append((self.CENTER_LADDER, (self.SCREEN_BOTTOM - 1, self.SCREEN_BOTTOM)))
        if state['screen'] == 14:
            regions.append((self.CENTER_LADDER, (self.SCREEN_BOTTOM, self.LOWER_LEVEL_14)))
            regions.append((self.CENTER_LADDER, (self.SCREEN_BOTTOM - 1, self.SCREEN_BOTTOM)))
        # tall bottom ladders
        if state['screen'] in [0, 2, 3, 4, 7, 11, 13]:
            regions.append((self.CENTER_LADDER, (self.SCREEN_BOTTOM, self.UPPER_LEVEL)))
            regions.append((self.CENTER_LADDER, (self.SCREEN_BOTTOM - 1, self.SCREEN_BOTTOM)))
        # short top ladders
        if state['screen'] in [4, 6, 9, 11, 13, 19, 21]:
            regions.append((self.CENTER_LADDER, (self.UPPER_LEVEL, self.SCREEN_TOP)))
            regions.append((self.CENTER_LADDER, (self.SCREEN_TOP, self.SCREEN_TOP + 1)))
        elif state['screen'] in [10, 22]:
            # add vertical landmark just above the bridge
            regions.append(
                (self.CENTER_LADDER, (self.UPPER_LEVEL, self.UPPER_LEVEL + self.climb_margin_y)))
            regions.append(
                (self.CENTER_LADDER, (self.UPPER_LEVEL + self.climb_margin_y, self.SCREEN_TOP)))
            regions.append((self.CENTER_LADDER, (self.SCREEN_TOP, self.SCREEN_TOP + 1)))
        return regions

    def can_run(self, state):
        if not (self.player_static(state) or state['has_ladder']):
            return False
        return self.at_start(state)

class ClimbUpLadderSkill(_ClimbLadderSkill):
    climb_action = actions.UP
    at_start = _ClimbLadderSkill.below_any_climb_region
    at_exit = _ClimbLadderSkill.above_any_climb_region

class ClimbDownLadderSkill(_ClimbLadderSkill):
    climb_action = actions.DOWN
    at_start = _ClimbLadderSkill.above_any_climb_region
    at_exit = _ClimbLadderSkill.below_any_climb_region

class _ClimbRopeSkill(_ClimbSkill):
    def get_climb_regions(self, state):
        ropes = defaultdict(list, {
            1: [self.ROPE_1],
            5: [self.ROPE_5_LEFT, self.ROPE_5_RIGHT, self.ROPE_5_RIGHT_ALT],
            8: [self.ROPE_8],
            14: [self.ROPE_14_LEFT, self.ROPE_14_RIGHT]
        })# yapf: disable
        regions = ropes[state['screen']]
        return regions

    def can_run(self, state):
        if not (self.player_static(state) or state['has_rope']):
            return False
        return self.at_start(state) and not self.at_exit(state)

class ClimbUpRopeSkill(_ClimbRopeSkill):
    climb_action = actions.UP
    at_start = _ClimbRopeSkill.at_any_climb_region
    at_exit = _ClimbRopeSkill.above_any_climb_region

    def get_climb_regions(self, state):
        regions = super().get_climb_regions(state)
        if state['screen'] == 5:
            regions = regions[:-1]  # Can't climb up on ROPE_5_RIGHT_ALT
        return regions

class ClimbDownRopeSkill(_ClimbRopeSkill):
    climb_action = actions.DOWN
    at_start = _ClimbRopeSkill.at_any_climb_region
    at_exit = _ClimbRopeSkill.below_any_climb_region

class DropFromRopeSkill(_ClimbRopeSkill):
    climb_action = actions.DOWN

    def can_run(self, state):
        if not (self.player_static(state) or state['has_rope']):
            return False
        return self.below_any_climb_region(state)

    def is_done(self, state):
        if state['respawning']:
            return True
        return (self.player_static(state) and state['player_status'] == 'standing')

class _RunSkill(Skill):
    object_margin = 12
    snake_margin = 10
    door_margin = 5
    charge_enemy = False

    def on_any_platform(self, state):
        platforms = self.get_platforms(state)
        if state['player_y'] not in platforms.keys():
            return False
        for left, right in platforms[state['player_y']]:
            if left <= state['player_x'] <= right:
                return True
        return False

    def left_of_any_platform(self, state):
        platforms = self.get_platforms(state)
        if state['player_y'] not in platforms.keys():
            return False
        for left, right in platforms[state['player_y']]:
            if state['player_x'] == left:
                return True
        return False

    def right_of_any_platform(self, state):
        platforms = self.get_platforms(state)
        if state['player_y'] not in platforms.keys():
            return False
        for left, right in platforms[state['player_y']]:
            if state['player_x'] == right:
                return True
        return False

    def _at_platform_edge(self, state):
        return self.at_exit(state) and not self.at_start(state)

    def at_left_wall(self, state):
        screen = state['screen']
        x, y = state['player_x'], state['player_y']
        if screen in [3, 8, 10, 15, 21] and x == 0x05:
            return True
        elif screen == 1 and ((x == 0x09 and y == self.MIDDLE_LEVEL_1) or
                              (x == 0x14 and y == self.LOWER_LEVEL_1)):
            return True
        elif screen == 5 and ((x in [0x24, 0x80] and y == self.UPPER_LEVEL) or
                              (x == 0x05 and y == self.LOWER_LEVEL_5)):
            return True
        elif screen == 14 and y < self.UPPER_LEVEL and x == 0x11:
            return True
        return False

    def at_right_wall(self, state):
        screen = state['screen']
        x, y = state['player_x'], state['player_y']
        if screen in [2, 9, 15, 20] and x == 0x95:
            return True
        elif screen == 1 and ((x == 0x91 and y == self.MIDDLE_LEVEL_1) or
                              (x == 0x85 and y == self.LOWER_LEVEL_1)):
            return True
        elif screen == 5 and ((x in [0x1a, 0x76] and y == self.UPPER_LEVEL) or
                              (x == 0x95 and y == self.LOWER_LEVEL_5)):
            return True
        elif screen == 8 and x == 0x95:
            return (y < self.UPPER_LEVEL)
        elif screen == 14 and y < self.UPPER_LEVEL and x == 0x89:
            return True
        elif screen == 15 and x >= 0x95:
            return True
        return False

    def near_right_of_object(self, state, object_x, margin=None):
        return self.right_of_object(state, object_x, 0) and self.near_object(
            state, object_x, margin)

    def near_left_of_object(self, state, object_x, margin=None):
        return self.left_of_object(state, object_x, 0) and self.near_object(
            state, object_x, margin)

    def would_hit_door(self, state):
        if state['door_left'] == 'unlocked' and state['door_right'] == 'unlocked':
            return False
        if 'key' in state['inventory']:
            return False
        door_y_positions = {1: self.UPPER_LEVEL, 5: self.LOWER_LEVEL_5, 17: self.UPPER_LEVEL}
        screen = state['screen']
        if (screen not in door_y_positions.keys()
                or state['player_y'] != door_y_positions[screen]):
            return False
        door_x_positions = defaultdict(list, {
            1: [(0x0e + 0x18) / 2, (0x82 + 0x8c) / 2],
            5: [(0x32 + 0x3c) / 2, (0x5e + 0x68) / 2],
            17: [(0x0e + 0x18) / 2, (0x82 + 0x8c) / 2],
        })# yapf: disable
        for door, status in zip(door_x_positions[screen],
                                [state['door_left'], state['door_right']]):
            if status == 'locked' and self.almost_hitting(state, door, self.door_margin):
                return True
        return False

    def would_hit_enemy(self, state):
        '''Returns whether self.run_action would cause player to hit an enemy or rolling skull'''
        #NOTE: The game treats rolling skulls differently from other enemies/objects, but this
        # function considers rolling skulls as well as other enemies the player might hit.
        if not state['has_enemy'] and not state['has_skull']:
            return False
        # Check for traditional enemy objects
        if state['has_enemy'] and state['player_y'] >= self.UPPER_LEVEL:
            margin = self.object_margin
            if state['has_snake']:
                margin = self.snake_margin
            elif state['level'] == 2:
                margin += 4
            if self.for_any_objects(state, fn=lambda x: self.almost_hitting(state, x, margin)):
                return True
        # Check for rolling skulls
        roll_skull_y = defaultdict(lambda: None, {
            1: self.LOWER_LEVEL_1,
            5: self.MIDDLE_LEVEL_5,
            18: self.UPPER_LEVEL,
        })[state['screen']]
        if state['has_skull'] and state['player_y'] == roll_skull_y:
            margin = self.object_margin
            if state['level'] == 1:
                margin += 4
            if state['level'] == 2:
                margin += 6
            if self.almost_hitting(state, state['skull_x'], margin):
                return True
        return False

    def can_run(self, state):
        if self.level_changing(state):
            return False
        if not self.player_static(state):
            return False
        if not self.on_any_platform(state):
            return False
        if self.would_hit_wall(state) or self.would_hit_door(state):
            return False
        if self.would_hit_enemy(state):
            return self.charge_enemy
        return not self.charge_enemy

    def policy(self, state):
        if self.player_static(state):
            return self.run_action
        else:
            return actions.NOOP

    def is_done(self, state):
        if self.level_changing(state):
            return True
        if state['screen_changing']:
            return False
        if state['respawning']:
            return True
        if self.at_exit(state) or self._at_platform_edge(state):
            return True
        if self.would_hit_wall(state) or self.would_hit_door(state):
            return True
        if (not state['has_enemy']) and state['has_object']:
            if (state['player_y'] == self.get_object_y_position(state)
                    and self.for_any_objects(state, fn=lambda x: self.at_object(state, x))):
                return True
        if self.charge_enemy:
            return not self.would_hit_enemy(state)
        return self.would_hit_enemy(state)

class RunLeftSkill(_RunSkill):
    run_action = actions.LEFT
    at_exit = _RunSkill.left_of_any_platform
    at_start = _RunSkill.right_of_any_platform
    would_hit_wall = _RunSkill.at_left_wall
    almost_hitting = _RunSkill.near_right_of_object

class RunRightSkill(_RunSkill):
    run_action = actions.RIGHT
    at_exit = _RunSkill.right_of_any_platform
    at_start = _RunSkill.left_of_any_platform
    would_hit_wall = _RunSkill.at_right_wall
    almost_hitting = _RunSkill.near_left_of_object

class _ChargeEnemySkill(_RunSkill):
    charge_enemy = True

class ChargeEnemyLeft(_ChargeEnemySkill, RunLeftSkill):
    pass

class ChargeEnemyRight(_ChargeEnemySkill, RunRightSkill):
    pass

class WaitForRespawn(Skill):
    def can_run(self, state):
        return bool(state['respawning'] or state['player_falling'])

    def policy(self, state):
        return actions.NOOP

    def is_done(self, state):
        return (not state['respawning']) and (not state['player_falling'])

class WaitForLevelChange(Skill):
    skull_start = 0x5b

    def can_run(self, state):
        return self.level_changing(state)

    def policy(self, state):
        return actions.NOOP

    def is_done(self, state):
        # We check the skull position because there's sometimes a bug during
        # the first frame that prevents the player from jumping. Waiting for
        # the skull to move from its starting position avoids this problem.
        return not self.level_changing(state) and state['skull_x'] != self.skull_start
