import copy
import os
import random

import numpy as np
import torch
import torch.nn.functional as F
from torch.optim.lr_scheduler import MultiStepLR

import ramps
from hmix_args import process_args
from hmix_image_dataset import load_image_dataset
from nnPU_loss import PNloss, PNCEloss, PUCEloss, ABSPUloss, DistPUloss, FOPULoss, PULBloss, BalancedPUCEloss, BalancedPULBloss\
    , PULBloss2, BalancedPUCEloss2, BalancedPULBloss2
from statistic import test
from hmix_model import MultiLayerPerceptron
from model.alexnet import alexnet
from model.cnn7 import cnn_cifar, cnn_stl
from model.lenet5 import lenet_fmnist, LeNet, LeNet_FMNIST
from model.loss import loss_entropy, loss_ft
from model.ResNet_Zoo import ResNet18, ResNet50
from utils.misc import multi_class_accuracy

args = process_args()
os.environ['CUDA_VISIBLE_DEVICES'] = str(args.gpu)
device = torch.device('cuda' if torch.cuda.is_available() else 'cpu')


def seed_torch(seed=args.seed):
    random.seed(seed)
    os.environ["PYTHONHASHSEED"] = str(seed)
    np.random.seed(seed)
    torch.random.manual_seed(seed)
    torch.manual_seed(seed)
    torch.cuda.manual_seed(seed)
    torch.cuda.manual_seed_all(seed)

    # torch.backends.cudnn.enabled = True
    # torch.backends.cudnn.benchmark = True
    torch.backends.cudnn.deterministic = True


seed_torch()


def select_loss(loss_name, gamma=1, tau=0.5, eta=1.0, pi=0.5, a0=2.0, b0=1.0, b1=0.8, k1=8.0, k2=8.0, t=0.5):
    losses = {
        # ===== 常见分类损失 =====
        "zero-one": lambda x: -0.5 * torch.sign(x) + 0.5,
        "hinge": lambda x: torch.clamp(1 - x, min=0),
        "exponential": lambda x: torch.exp(-x),
        "perceptron": lambda x: torch.clamp(-x, min=0),  # Logistic 的前身
        "logistic": lambda x: F.softplus(-x),
        "sigmoid": lambda x: torch.sigmoid(-x),

        # ===== 变体 =====
        "squared-hinge": lambda x: torch.clamp(1 - x, min=0) ** 2,  # Squared Hinge（Hinge 的平方版）
        "smooth-hinge": lambda x: torch.where(x < 0, 0.5 - x,
                                              torch.where(x < 1, 0.5 * (1 - x) ** 2, torch.zeros_like(x))), # Hinge 的平滑版
        "modified-huber": lambda x: torch.where(x >= -1, 0.25 * torch.clamp(-x, min=0) ** 2, -x),   # Hinge 与 Squared 结合的分段形式
        "squared": lambda x: (1 - x) ** 2,

        # ===== 鲁棒损失函数 =====
        "unhinged": lambda x: 1 - x,
        "tangent": lambda x: (2 * torch.atan(x) - 1) ** 2,
        "savage": lambda x: 1 / (1 + torch.exp(x)) ** 2,
        "focal": lambda x, gamma_focal=2.0: (1 - torch.sigmoid(x)) ** gamma_focal * F.softplus(-x), # Logistic 的变体，提升小概率样本的关注
        "gsigmoid": lambda x: torch.sigmoid(-gamma * x),  # Sigmoid 的 gamma 控制扩展
        "ramp": lambda x: torch.clamp(torch.min(torch.tensor(1.0, device=x.device), (1 - x) / 2), min=0),

        # ===== 特殊设计 =====
        "pinball": lambda x: torch.max(1 - x, -tau * (1 - x)),
        "rescaled-hinge": lambda x: eta * (1 - torch.exp(-eta * torch.clamp(1 - x, min=0))) / (1 - torch.exp(-eta)),
        "double-hinge": lambda x: torch.max(torch.stack([
            torch.zeros_like(x), (1 - x) / 2, -x
        ]), dim=0).values,

        # ===== mine =====
        "mine": lambda x: 1 - torch.exp(
            -calc_alpha(pi, a0, k1) *
            torch.clamp(1 - x, min=0) ** calc_beta(pi, b0, b1, k2, t)
        ),
    }

    return losses[loss_name]


def calc_alpha(pi, a0=2.0, k1=8.0):
    """计算 alpha(π) 参数"""
    # 将浮点数转换为标量张量
    pi_tensor = torch.tensor(pi, dtype=torch.float)
    return a0 / (1 + torch.exp(k1 * (pi_tensor - 0.5)))

def calc_beta(pi, b0=1.0, b1=0.8, k2=8.0, t=0.5):
    """计算 beta(π) 参数"""
    # 将浮点数转换为标量张量
    pi_tensor = torch.tensor(pi, dtype=torch.float)
    return b0 + b1 * torch.sigmoid(k2 * (pi_tensor - t))

def select_model(model_name):
    models = {"cnn_cifar": cnn_cifar, "cnn_stl": cnn_stl,
              "lenet5": LeNet, "resnet50": ResNet50,
              'mlp': MultiLayerPerceptron}
    return models[model_name]

def make_optimizer(model, optim, stepsize):
    if optim == "adam":
        optimizer = torch.optim.Adam(model.parameters(),
                                     lr=stepsize,
                                     betas=(0.5, 0.99),
                                     weight_decay=args.weight_decay)
    elif optim == "adagrad":
        optimizer = torch.optim.Adagrad(model.parameters(),
                                        lr=stepsize,
                                        weight_decay=args.weight_decay)
    elif optim == "adamw":
        optimizer = torch.optim.AdamW(model.parameters(),
                                      lr=stepsize,
                                      weight_decay=args.weight_decay)
    elif optim == "sgd":
        optimizer = torch.optim.SGD(model.parameters(),
                                    lr=stepsize,
                                    momentum=0.9,
                                    weight_decay=args.weight_decay)
    elif optim == "adadelta":
        optimizer = torch.optim.Adadelta(model.parameters(),
                                         lr=stepsize,
                                         rho=0.95)
    return optimizer


def make_scheduler(optim):
    # scheduler = MultiStepLR(optim, milestones=[50, 100], gamma=0.1)
    scheduler = MultiStepLR(optim,
                            milestones=args.milestones,
                            gamma=args.scheduler_gamma)

    return scheduler


class trainer_pn():
    def __init__(self, models, loss_funcs, optimizers, schedulers,
                 XYtrainLoader, XYvalidLoader, XYtestLoader,
                 prior, loss_func_pn):
        self.models = models
        self.loss_funcs = loss_funcs.values()
        self.optimizers = optimizers.values()
        self.schedulers = schedulers.values()

        self.XYtrainLoader = XYtrainLoader
        self.XYvalidLoader = XYvalidLoader
        self.XYtestLoader = XYtestLoader
        self.prior = prior
        self.loss_func_pn = loss_func_pn

        # only for PULBLoss and BalancedPULBloss
        self.bound = 0

    def train(self, epoch):
        results_overall_val = {}
        results_class_val = {}
        results_overall_test = {}
        results_class_test = {}

        for model_name in self.models.keys():
            results_overall_val[model_name] = []
            results_class_val[model_name] = []
            results_overall_test[model_name] = []
            results_class_test[model_name] = []

        for net, opt, scheduler, loss_func, model_name in zip(
                self.models.values(), self.optimizers, self.schedulers,
                self.loss_funcs, self.models.keys()):
            XYtrainLoader_iter = iter(self.XYtrainLoader)
            net.train()
            total_steps = args.val_iterations

            if net.num_classifier == 1:
                outputs_all = np.array([]).reshape(0, 1)
            else:
                outputs_all = np.array([]).reshape(0, 2)
            targets_all = np.array([])

            targets_all = np.array([])
            predicts_all = np.array([])
            probs_all = np.array([])

            for i in range(args.val_iterations):
                # load positive data
                try:
                    data, target = next(XYtrainLoader_iter)
                except:
                    XYtrainLoader_iter = iter(self.XYtrainLoader)
                    data, target = next(XYtrainLoader_iter)

                data = data.to(device, non_blocking=True)
                target = target.to(device, non_blocking=True)
                output = net(data)  # get output for every net
                loss = loss_func(output, target)

                size = len(target)
                p = np.reshape(torch.sigmoid(output).detach().cpu().numpy(), size)
                probs_all = np.hstack((probs_all, p))
                # o = np.reshape(torch.sign(outputs).detach().cpu().numpy(), size)
                o = np.where(p > 0.5, 1, -1)
                predicts_all = np.hstack((predicts_all, o))

                # ############### add

                outputs_all = np.vstack((outputs_all, output.detach().cpu().numpy()))
                targets_all = np.hstack((targets_all, target.detach().cpu().numpy()))

                # ############### end add

                opt.zero_grad()  # clear gradients for next train
                loss.backward()  # backpropagation, compute gradients
                opt.step()  # apply gradients

            outputs_all = torch.from_numpy(outputs_all).to(device)
            targets_all = torch.from_numpy(targets_all).to(device)

            val_overall_metrics, val_class_metrics = multi_class_accuracy(probs_all, predicts_all, targets_all)

            test_overall_metrics, test_class_metrics, test_loss, test_loss_p, test_loss_n = test(args, self.XYtestLoader, net, self.loss_func_pn, device)
            results_overall_test[model_name].extend(test_overall_metrics)
            results_class_test[model_name].extend(test_class_metrics)

            results_overall_val[model_name].extend(val_overall_metrics)
            results_class_val[model_name].extend(val_class_metrics)

        return results_overall_val, results_class_val, results_overall_test, results_class_test

    def run(self, Epochs):
        results_val = {}
        results_test = {}
        # [precision,recall,error_rate, num_predicted_positive]
        metrics_overall = ['ACC', 'AUC', 'Macro_F1', 'Micro_F1', 'Precision', 'Recall', 'ErrorRate']
        metrics_class = ['Class_F1', 'Class_Precision', 'Class_Recall', 'Class_NPP']

        for model_name in self.models.keys():
            results_val[model_name] = [[] for _ in range(len(metrics_overall) + len(metrics_class) * args.num_classifier)]
            results_test[model_name] = [[] for _ in range(len(metrics_overall) + len(metrics_class) * args.num_classifier)]

        print("Epoch\t" + "".join([
            "".join([
                # 首先输出整体指标
                "{}/{}\t".format(model_name, metric)
                for metric in metrics_overall
            ]) +
            "".join([
                # 然后对每个分类指标，依次输出所有类别
                "".join([
                    "{}/{}_{}\t".format(model_name, metric, j + 1)
                    for j in range(args.num_classifier)
                ])
                for metric in metrics_class
            ])
            for model_name in sorted(results_test.keys())
        ]) + "\n")


        for epoch in range(Epochs):
            _results_overall_val, _results_class_val, _results_overall_test, _results_class_test = self.train(epoch)

            print("{}\t".format(epoch) + "".join([
                # 打印总体指标
                "".join(["{:-8}\t".format(round(_results_overall_test[model_name][i], 4))
                         for i in range(len(metrics_overall))]) +
                # 打印类别指标，每个类别指标都是2个值
                "".join([
                    "".join(["{:-8}\t".format(round(value, 4))
                             for value in _results_class_test[model_name][j]])
                    for j in range(len(_results_class_test[model_name]))])
                for model_name in sorted(results_test.keys())
            ]) + "\n")

            for model_name in self.models.keys():
                for i in range(len(metrics_overall)):
                    results_test[model_name][i].append(_results_overall_test[model_name][i])
                    results_val[model_name][i].append(_results_overall_val[model_name][i])
                for i in range(len(metrics_class)):
                    for j in range(args.num_classifier):
                        results_test[model_name][args.num_classifier * i + len(metrics_overall) + j].\
                            append(_results_class_test[model_name][i][j])
                        results_val[model_name][args.num_classifier * i + len(metrics_overall) + j].\
                            append(_results_class_val[model_name][i][j])

        return results_val, results_test


def load_pretrained_vectors(vocab, fname):
    """Load pretrained vectors and create embedding layers.

    Args:
        word2idx (Dict): Vocabulary built from the corpus
        fname (str): Path to pretrained vector file

    Returns:
        embeddings (np.array): Embedding matrix with shape (N, d) where N is
            the size of word2idx and d is embedding dimension
    """

    print("Loading pretrained vectors...")
    fin = open(fname, 'r', encoding='utf-8', newline='\n', errors='ignore')
    n, d = map(int, fin.readline().split())

    # Initilize random embeddings
    embeddings = np.random.uniform(-0.25, 0.25, (len(vocab), d))
    embeddings[vocab['<pad>']] = np.zeros((d, ))

    # Load pretrained vectors
    count = 0
    for line in fin:
        tokens = line.rstrip().split(' ')
        word = tokens[0]
        if word in vocab:
            count += 1
            embeddings[vocab[word]] = np.array(tokens[1:], dtype=np.float32)

    print(f"There are {count} / {len(vocab)} pretrained vectors found.")

    return np.asarray(embeddings, dtype=np.float32)


def main():
    # args = process_args()
    print("using:", device)

    image_datasets = ['mnist', 'fashionmnist', 'cifar10', 'stl10', 'alzheimer']
    text_datasets = [
        'imdb', 'yelp_full', '20ng', 'yelp', 'amazon', 'amazon_full'
    ]

    text_flag = False
    dim = None
    pretrained_embedding = None
    vocab_size = None

    # dataset setup
    if args.dataset in image_datasets:
        (XYPtrainLoader, XYUtrainLoader, XYtrainLoader, XYvalidLoader, XYtestLoader,
         XYPtrainSet, dim,
         prior) = load_image_dataset(args.dataset, args.labeled,
                                     args.batchsize, args.positive_label_list)
        prior = args.prior

    # model setup
    loss_type = select_loss(args.loss, args.loss_gamma, args.loss_tau, args.loss_eta,
                            pi=args.prior, a0=args.a0, b0=args.b0, b1=args.b1, k1=args.k1, k2=args.k2, t=0.5)
    selected_model = select_model(args.model)
    model = selected_model(int(dim))
    models_pn = {
        "PN": copy.deepcopy(model).to(device),
    }

    loss_funcs_pn = {
        "PN": PNloss(args.prior, loss=loss_type),
    }
    loss_func_pn = PNCEloss(args.prior, loss=loss_type)

    # trainer setup
    optimizers_pn = {
        k: make_optimizer(v, args.optim, args.stepsize)
        for k, v in models_pn.items()
    }
    schedulers_pn = {k: make_scheduler(v) for k, v in optimizers_pn.items()}

    print("input dim: {}".format(dim))
    print("prior: {}".format(args.prior))
    print("loss: {}".format(args.loss))
    print("batchsize: {}".format(args.batchsize))
    print("model: {}".format(args.model))
    print("beta: {}".format(args.beta))
    print("gamma: {}".format(args.gamma))
    print("")

    # run training
    PNtrainer = trainer_pn(models_pn, loss_funcs_pn, optimizers_pn,
                           schedulers_pn, XYtrainLoader, XYvalidLoader,
                           XYtestLoader, prior, loss_func_pn)
    results_pn_val, results_pn_test = PNtrainer.run(args.epoch)

    save_path = "./results_pn/"
    save_result(save_path, results_pn_val, saved_mode="val")
    save_result(save_path, results_pn_test, saved_mode="test")


def save_result(save_path, results_pu, saved_mode="test"):
    if args.method == "FOPU":
        filename_result = save_path + "".join(["{}".format(model_name) for model_name in sorted(results_pu.keys())]) \
                          + "_{}_{}_{}_{}_{}_{}".format(args.preset, args.model, args.stepsize, args.weight_decay,
                                                        args.labeled, args.lam_f)
    elif args.method == "PULB" or args.method == "PULB2":
        filename_result = save_path + "".join(["{}".format(model_name) for model_name in sorted(results_pu.keys())]) \
                          + "_{}_{}_{}_{}_{}_{}".format(args.preset, args.model, args.stepsize, args.weight_decay,
                                                        args.labeled, args.momentum)
    elif args.method == "BalancePU" or args.method == "BalancePU2":
        filename_result = save_path + "".join(["{}".format(model_name) for model_name in sorted(results_pu.keys())]) \
                          + "_{}_{}_{}_{}_{}_{}".format(args.preset, args.model, args.stepsize, args.weight_decay,
                                                        args.labeled, args.balance_weight)
    elif args.method == "PULB+BalancePU" or args.method == "PULB2+BalancePU2":
        filename_result = save_path + "".join(["{}".format(model_name) for model_name in sorted(results_pu.keys())]) \
                          + "_{}_{}_{}_{}_{}_{}_{}".format(args.preset, args.model, args.stepsize, args.weight_decay,
                                                           args.labeled, args.momentum, args.balance_weight)
    else:
        filename_result = save_path + "".join(["{}".format(model_name) for model_name in sorted(results_pu.keys())]) \
                          + "_{}_{}_{}_{}_{}".format(args.preset, args.model, args.stepsize, args.weight_decay, args.labeled)

    if args.loss == 'gsigmoid':
        filename_result += "_{}_{}{}_{}.txt".format(args.seed, args.loss, args.loss_gamma, saved_mode)
    elif args.loss == 'pinball':
        filename_result += "_{}_{}{}_{}.txt".format(args.seed, args.loss, args.loss_tau, saved_mode)
    elif args.loss == 'rescaled_hinge':
        filename_result += "_{}_{}{}_{}.txt".format(args.seed, args.loss, args.loss_eta, saved_mode)
    else:
        filename_result += "_{}_{}_{}.txt".format(args.seed, args.loss, saved_mode)

    metrics_overall = ['ACC', 'AUC', 'Macro_F1', 'Micro_F1', 'Precision', 'Recall', 'ErrorRate']
    metrics_class = ['Class_F1', 'Class_Precision', 'Class_Recall', 'Class_NPP']

    with open(filename_result, 'w') as file_result:
        file_result.write("Epoch\t" + "".join([
            "".join([
                "{}/{}\t".format(model_name, metric) for metric in metrics_overall
            ]) +
            "".join([
                "".join([
                    "{}/{}_{}\t".format(model_name, metric, j + 1) for j in range(args.num_classifier)
                ])
                for metric in metrics_class
            ])
            for model_name in sorted(results_pu.keys())
        ]) + "\n")

        for epoch in range(args.epoch):
            file_result.write("{}\t".format(epoch) + "".join([
                    "".join(["{:-8}\t".format(round(results_pu[model_name][i][epoch], 4))
                             for i in range(len(results_pu[model_name]))])  # 循环处理每个指标的所有类别
            for model_name in sorted(results_pu.keys())
            ]) + "\n")


def save_loss(save_path, pu_loss):
    if args.method == "FOPU":
        filename_result = save_path + "".join(["{}".format(model_name) for model_name in sorted(pu_loss.keys())]) \
                          + "_{}_{}_{}_{}_{}_{}".format(args.preset, args.model, args.stepsize, args.weight_decay,
                                                        args.labeled, args.lam_f)
    elif args.method == "PULB" or args.method == "PULB2":
        filename_result = save_path + "".join(["{}".format(model_name) for model_name in sorted(pu_loss.keys())]) \
                          + "_{}_{}_{}_{}_{}_{}".format(args.preset, args.model, args.stepsize, args.weight_decay,
                                                        args.labeled, args.momentum)
    elif args.method == "BalancePU" or args.method == "BalancePU2":
        filename_result = save_path + "".join(["{}".format(model_name) for model_name in sorted(pu_loss.keys())]) \
                          + "_{}_{}_{}_{}_{}_{}".format(args.preset, args.model, args.stepsize, args.weight_decay,
                                                        args.labeled, args.balance_weight)
    elif args.method == "PULB+BalancePU" or args.method == "PULB2+BalancePU2":
        filename_result = save_path + "".join(["{}".format(model_name) for model_name in sorted(pu_loss.keys())]) \
                          + "_{}_{}_{}_{}_{}_{}_{}".format(args.preset, args.model, args.stepsize, args.weight_decay,
                                                        args.labeled, args.momentum, args.balance_weight)
    else:
        filename_result = save_path + "".join(["{}".format(model_name) for model_name in sorted(pu_loss.keys())]) \
                          + "_{}_{}_{}_{}_{}".format(args.preset, args.model, args.stepsize, args.weight_decay, args.labeled)

    if args.loss == 'gsigmoid':
        filename_result += "_{}_{}{}_loss.txt".format(args.seed, args.loss, args.loss_gamma)
    elif args.loss == 'pinball':
        filename_result += "_{}_{}{}_loss.txt".format(args.seed, args.loss, args.loss_tau)
    elif args.loss == 'rescaled_hinge':
        filename_result += "_{}_{}{}_loss.txt".format(args.seed, args.loss, args.loss_eta)
    else:
        filename_result += "_{}_{}_loss.txt".format(args.seed, args.loss)

    with open(filename_result, 'w') as file_result:
        file_result.write("Epoch\t" + "".join([
            "{}/test_loss\t{}/test_loss_p\t{}/test_loss_n\t{}/train_loss\t{}/train_loss_p\t{}/train_loss_u\t{}/R_p_pos\t{}/R_u_neg\t{}/R_p_neg\t".format(
                model_name, model_name, model_name, model_name, model_name, model_name, model_name, model_name, model_name)
            for model_name in sorted(pu_loss.keys())
        ]) + "\n")

        for epoch in range(args.epoch):
            file_result.write("{}\t".format(epoch) + "".join([
                "{:-8}\t{:-8}\t{:-8}\t{:-8}\t{:-8}\t{:-8}\t{:-8}\t{:-8}\t{:-8}\t".format(
                    round(pu_loss[model_name][0][epoch], 4),
                    round(pu_loss[model_name][1][epoch], 4),
                    round(pu_loss[model_name][2][epoch], 4),
                    round(pu_loss[model_name][3][epoch], 4),
                    round(pu_loss[model_name][4][epoch], 4),
                    round(pu_loss[model_name][5][epoch], 4),
                    round(pu_loss[model_name][6][epoch], 4),
                    round(pu_loss[model_name][7][epoch], 4),
                    round(pu_loss[model_name][8][epoch], 4),
                ) for model_name in sorted(pu_loss.keys())
            ]) + "\n")


if __name__ == '__main__':
    import os
    import sys
    os.chdir(sys.path[0])
    print("working dir: {}".format(os.getcwd()))
    main()
