"""resnet in pytorch
[1] Kaiming He, Xiangyu Zhang, Shaoqing Ren, Jian Sun.
    Deep Residual Learning for Image Recognition
    https://arxiv.org/abs/1512.03385v1
"""
from asyncio.unix_events import BaseChildWatcher
from asyncio import _leave_task
from base.base_net import BaseNet
import torch.nn as nn
import torch.nn.functional as F
import torch

class BasicBlock(nn.Module):
    """Basic Block for resnet 18 and resnet 34
    """

    #BasicBlock and BottleNeck block
    #have different output size
    #we use class attribute expansion
    #to distinct
    expansion = 1

    def __init__(self, in_channels, out_channels, stride=1):
        super().__init__()

        #residual function
        self.residual_function = nn.Sequential(
            nn.Conv2d(in_channels, out_channels, kernel_size=3, stride=stride, padding=1, bias=False),
            nn.BatchNorm2d(out_channels),
            nn.LeakyReLU(inplace=True),
            nn.Conv2d(out_channels, out_channels * BasicBlock.expansion, kernel_size=3, padding=1, bias=False),
            nn.BatchNorm2d(out_channels * BasicBlock.expansion)
        )

        #shortcut
        self.shortcut = nn.Sequential()

        #the shortcut output dimension is not the same with residual function
        #use 1*1 convolution to match the dimension
        if stride != 1 or in_channels != BasicBlock.expansion * out_channels:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_channels, out_channels * BasicBlock.expansion, kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(out_channels * BasicBlock.expansion)
            )

    def forward(self, x):
        return nn.LeakyReLU(inplace=True)(self.residual_function(x) + self.shortcut(x))


class BottleNeck(nn.Module):
    """Residual block for resnet over 50 layers
    """
    expansion = 4
    def __init__(self, in_channels, out_channels, stride=1):
        super().__init__()
        self.residual_function = nn.Sequential(
            nn.Conv2d(in_channels, out_channels, kernel_size=1, bias=False),
            nn.BatchNorm2d(out_channels),
            nn.LeakyReLU(inplace=True),
            nn.Conv2d(out_channels, out_channels, stride=stride, kernel_size=3, padding=1, bias=False),
            nn.BatchNorm2d(out_channels),
            nn.LeakyReLU(inplace=True),
            nn.Conv2d(out_channels, out_channels * BottleNeck.expansion, kernel_size=1, bias=False),
            nn.BatchNorm2d(out_channels * BottleNeck.expansion),
        )

        self.shortcut = nn.Sequential()

        if stride != 1 or in_channels != out_channels * BottleNeck.expansion:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_channels, out_channels * BottleNeck.expansion, stride=stride, kernel_size=1, bias=False),
                nn.BatchNorm2d(out_channels * BottleNeck.expansion)
            )

    def forward(self, x):
        return nn.LeakyReLU(inplace=True)(self.residual_function(x) + self.shortcut(x))


class ResNet(nn.Module):

    def __init__(self, block, num_block, num_classes=100):
        super().__init__()

        self.in_channels = 64

        self.conv1 = nn.Sequential(
            nn.Conv2d(3, 64, kernel_size=3, padding=1, bias=False),
            nn.BatchNorm2d(64),
            nn.LeakyReLU(inplace=True))
        #we use a different inputsize than the original paper
        #so conv2_x's stride is 1
        self.conv2_x = self._make_layer(block, 64, num_block[0], 1)
        self.conv3_x = self._make_layer(block, 128, num_block[1], 2)
        self.conv4_x = self._make_layer(block, 256, num_block[2], 2)
        self.conv5_x = self._make_layer(block, 512, num_block[3], 2)
        self.avg_pool = nn.AdaptiveAvgPool2d((1, 1))
        self.fc = nn.Linear(512 * block.expansion, num_classes, bias=False)

    def _make_layer(self, block, out_channels, num_blocks, stride):
        """make resnet layers(by layer i didnt mean this 'layer' was the
        same as a neuron netowork layer, ex. conv layer), one layer may
        contain more than one residual block
        Args:
            block: block type, basic block or bottle neck block
            out_channels: output depth channel number of this layer
            num_blocks: how many blocks per layer
            stride: the stride of the first block of this layer
        Return:
            return a resnet layer
        """

        # we have num_block blocks per layer, the first block
        # could be 1 or 2, other blocks would always be 1
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_channels, out_channels, stride))
            self.in_channels = out_channels * block.expansion

        return nn.Sequential(*layers)

    def forward(self, x):
        output = self.conv1(x)
        output = self.conv2_x(output)
        output = self.conv3_x(output)
        output = self.conv4_x(output)
        output = self.conv5_x(output)
        output = self.avg_pool(output)
        output = output.view(output.size(0), -1)
        output = self.fc(output)

        return output


class ResNet18AutoEncoder(BaseNet):

    def __init__(self, in_channels=1, mid_dim=128, mid_size=4):
        super().__init__()

        self.in_channels = in_channels
        self.mid_size = mid_size
        self.mid_dim = mid_dim
        self.expand_channels = 512
        self.change_inputs_layer = nn.Sequential(
            nn.Conv2d(self.in_channels, 3, kernel_size=3, padding=1, bias=False),
            nn.BatchNorm2d(3),
            nn.LeakyReLU(inplace=True)
            )
        # self.encoder = ResNet(BasicBlock, [2, 2, 2, 2], num_classes=self.mid_dim)
        self.encoder = ResNet(BasicBlock, [4, 4, 4, 4], num_classes=self.mid_dim)

        self.decoder = nn.Sequential(

            BasicBlock(in_channels=int(self.mid_dim / (self.mid_size ** 2)), out_channels=self.expand_channels),
            nn.Upsample(scale_factor=2, mode='nearest'),
            BasicBlock(in_channels=self.expand_channels, out_channels=int(self.expand_channels / 2)),
            nn.Upsample(scale_factor=2, mode='nearest'),
            BasicBlock(in_channels=int(self.expand_channels / 2), out_channels=int(self.expand_channels / 4)),
            nn.Upsample(scale_factor=2, mode='nearest'),
            BasicBlock(in_channels=int(self.expand_channels / 4), out_channels=int(self.expand_channels / 8)),
            nn.Upsample(scale_factor=2, mode='nearest'),
            BasicBlock(in_channels=int(self.expand_channels / 8), out_channels=int(self.expand_channels / 16)),
        )
        self.output = nn.Sequential(
            nn.Conv2d(int(self.expand_channels / 16), int(self.expand_channels / 32), kernel_size=3, padding=1, bias=False),
            nn.BatchNorm2d(int(self.expand_channels / 32)),
            nn.LeakyReLU(inplace=True),
            nn.Conv2d(int(self.expand_channels / 32), 3, kernel_size=3, padding=1, bias=False)
            )
        self.change_outputs_layer = nn.Sequential(
            nn.BatchNorm2d(3),
            nn.LeakyReLU(inplace=True),
            nn.Conv2d(3, in_channels, kernel_size=3, padding=1, bias=False)
        )

    def forward(self, x):
        if self.in_channels == 1:
            x = self.change_inputs_layer(x)
        x = self.encoder(x)
        mid_repre = x
       
        x = x.view(x.size(0), int(self.mid_dim / (self.mid_size ** 2)), self.mid_size, self.mid_size)
        x = self.decoder(x)
        x = self.output(x)
        if self.in_channels == 1:
            x = self.change_outputs_layer(x)
        x = torch.sigmoid(x)
        return x, mid_repre
