import torch
import torch.utils.data as Data
from torch import nn
import numpy as np
import torch.nn.init as init
import math

def get_attn_pad_mask(seq_q, seq_k):
    '''
        seq_q: [batch_size, seq_len]
        seq_k: [batch_size, seq_len]
        seq_len could be src_len or it could be tgt_len
        seq_len in seq_q and seq_len in seq_k maybe not equal
    '''
    batch_size, len_q = seq_q.size()
    batch_size, len_k = seq_k.size()
    # eq(zero) is PAD token
    pad_attn_mask = seq_k.data.eq(0).unsqueeze(1)  # [batch_size, 1, len_k], True is masked
    return pad_attn_mask.expand(batch_size, len_q, len_k)  # [batch_size, len_q, len_k]

def get_attn_subsequence_mask(seq, device):
    """
    seq: [batch_size, tgt_len]
    """
    batch_size, tgt_len = seq.size()
    attn_shape = (batch_size, tgt_len, tgt_len)
    subsequence_mask = torch.triu(torch.ones(attn_shape, dtype=torch.uint8, device=device), diagonal=1)
    return subsequence_mask

def attn_mask(X_input, device):
    '''
        X_input: [batch_size, tgt_len]
    '''
    dec_self_attn_pad_mask = get_attn_pad_mask(X_input, X_input) # [batch_size, tgt_len, d_model] 遮挡padding部分
    dec_self_attn_subsequence_mask = get_attn_subsequence_mask(X_input, device) # [batch_size, tgt_len, d_model] 遮挡未来时刻的词
    # 两个mask之和只要有一个为1的地方，就为1
    dec_self_attn_mask = torch.gt((dec_self_attn_pad_mask + dec_self_attn_subsequence_mask), 0) # [batch_size, tgt_len, d_model] 

    return dec_self_attn_mask


def rotate_positional_encoding(x):
    """
    实现RoPE位置编码
    :param x: 输入的查询或键向量 (batch_size, head, seq_len, dim)
    :return: 位置编码后的向量 (batch_size, head, seq_len, dim)
    """
    # 假设dim是偶数，因为我们要成对使用sin/cos
    _, _, seq_len, dim = x.shape
    assert dim % 2 == 0, "Embedding dimension must be even."
    
    # 生成位置索引
    position = torch.arange(seq_len, dtype=torch.float32, device=x.device).unsqueeze(1)  # (seq_len, 1)
    
    # 生成每个维度的频率指数（使用奇偶数分别为sin/cos频率）
    index = torch.arange(dim // 2, dtype=torch.float32, device=x.device).unsqueeze(0)  # (1, dim//2)
    
    # 计算 sin 和 cos 的频率
    angles = position / torch.pow(10000, (2 * index) / dim)  # (seq_len, dim//2)
    
    # 将 sin 应用于偶数维度，将 cos 应用于奇数维度
    pos_enc = torch.zeros((seq_len, dim), dtype=torch.float32, device=x.device)  # 初始化位置编码矩阵
    pos_enc[:, 0::2] = torch.sin(angles)  # 偶数维度
    pos_enc[:, 1::2] = torch.cos(angles)  # 奇数维度
    
    # 对输入x应用旋转编码
    # pos_enc的形状是 (seq_len, dim)，需要扩展到 (1, 1, seq_len, dim) 以广播到 x 的维度
    pos_enc = pos_enc.unsqueeze(0).unsqueeze(0)  # (1, 1, seq_len, dim)
    
    x_rotated = torch.zeros_like(x)
    x_rotated[:, :, :, 0::2] = x[:, :, :, 0::2] * pos_enc[:, :, :, 1::2] - x[:, :, :, 1::2] * pos_enc[:, :, :, 0::2]
    x_rotated[:, :, :, 1::2] = x[:, :, :, 1::2] * pos_enc[:, :, :, 1::2] + x[:, :, :, 0::2] * pos_enc[:, :, :, 0::2]
    
    return x_rotated

# class ScaledDotProductAttention(nn.Module):
#         def __init__(self):
#             super(ScaledDotProductAttention, self).__init__()

#         def forward(self, Q, K, V, attn_mask, d_k):
#             '''
#                 Q: [batch_size, n_heads, len_q, d_k]
#                 K: [batch_size, n_heads, len_k, d_k]
#                 V: [batch_size, n_heads, len_v(=len_k), d_v]
#                 attn_mask: [batch_size, n_heads, seq_len, seq_len]
#             '''
#             scores = torch.matmul(Q, K.transpose(-1, -2)) / np.sqrt(
#                 d_k)  # scores : [batch_size, n_heads, len_q, len_k]
#             scores.masked_fill_(attn_mask, -1e9)  # Fills elements of self tensor with value where mask is True.

#             attn = nn.Softmax(dim=-1)(scores)
#             context = torch.matmul(attn, V)  # [batch_size, n_heads, len_q, d_v]
#             return context, attn


class MultiHeadAttention(nn.Module):
    def __init__(self, args):
        super(MultiHeadAttention, self).__init__()

        self.n_head = args.n_heads
        self.d_k = args.d_k
        self.d_v = args.d_v
        self.d_model = args.d_model
    
        self.W_Q = nn.Linear(args.d_model, args.d_k * args.n_heads, bias=False)
        init.normal_(self.W_Q.weight, mean=0.0, std=(args.d_model)**(-args.qk_std))

        self.W_K = nn.Linear(args.d_model, args.d_k * args.n_heads, bias=False)
        init.normal_(self.W_K.weight, mean=0.0, std=(args.d_model)**(-args.qk_std))

        self.W_V = nn.Linear(args.d_model, args.d_v * args.n_heads, bias=False)
        init.normal_(self.W_V.weight, mean=0.0, std=(args.d_model)**(-args.vo_std))

        self.fc = nn.Linear(args.n_heads * args.d_v, args.d_model, bias=False)
        init.normal_(self.fc.weight, mean=0.0, std=(args.n_heads * args.d_v)**(-args.vo_std))
        self.layernorm = nn.LayerNorm(args.d_model)

    def forward(self, hidden_state, attn_mask):
        '''
        ?input_Q: [batch_size, len_q, d_model]
        ?input_K: [batch_size, len_k, d_model]
        ?input_V: [batch_size, len_v(=len_k), d_model]
        attn_mask: [batch_size, seq_len, seq_len]
        '''
        residual, batch_size = hidden_state, hidden_state.size(0)
        hidden_state = self.layernorm(hidden_state)  # pre layernoem

        # (B, S, D) -proj-> (B, S, D_new) -split-> (B, S, H, W) -trans-> (B, H, S, W)
        Q = self.W_Q(hidden_state).view(batch_size, -1, self.n_head, self.d_k).transpose(1, 2)  # Q: [batch_size, n_heads, S, d_k]
        K = self.W_K(hidden_state).view(batch_size, -1, self.n_head, self.d_k).transpose(1, 2)  # K: [batch_size, n_heads, S, d_k]
        V = self.W_V(hidden_state).view(batch_size, -1, self.n_head, self.d_v).transpose(1, 2)  # V: [batch_size, n_heads, S, d_v]

        RoPE_Q = rotate_positional_encoding(Q) # apply RoPE to Q,K
        RoPE_K = rotate_positional_encoding(K)
        
        attn_mask = attn_mask.unsqueeze(1).repeat(1, self.n_head, 1, 1)  # attn_mask : [batch_size, n_heads, seq_len, seq_len]

        # scores : [batch_size, n_heads, len_q, len_k]
        attn = torch.matmul(RoPE_Q, RoPE_K.transpose(-1, -2)) / np.sqrt(self.d_k)
        masked_attn = attn.masked_fill(attn_mask, -1e9)  # Fills elements of self tensor with value where mask is True.
        softmax_attn = nn.Softmax(dim=-1)(masked_attn)
        qkv = torch.matmul(softmax_attn, V)  # [batch_size, n_heads, len_q, d_v] 

        qkv_out = qkv.transpose(1, 2).reshape(batch_size, -1, self.n_head * self.d_v)  # context: [batch_size, len_q, n_heads * d_v]
        output = self.fc(qkv_out)  # [batch_size, len_q, d_model]

        return output + residual, softmax_attn
    
class PoswiseFeedForwardNet(nn.Module):
    def __init__(self, args):
        super(PoswiseFeedForwardNet, self).__init__()
        self.fc = nn.Sequential(
            nn.Linear(args.d_model, args.d_feedforward, bias=False),
            nn.ReLU(),
            nn.Linear(args.d_feedforward, args.d_model, bias=False)
        )
        self.layernorm=nn.LayerNorm(args.d_model)
        # Initialize weights of Linear layers
        for i, layer in enumerate(self.fc):
            if isinstance(layer, nn.Linear):
                if i == len(self.fc) - 1:  # For the last Linear layer
                    init.normal_(layer.weight, mean=0.0, std=(args.d_model)**(-args.mlp_std) / math.sqrt(2 * args.n_layers))
                else:
                    init.normal_(layer.weight, mean=0.0, std=(args.d_model)**(-args.mlp_std))
                    
                    
    def forward(self, hidden_state):
        '''
        hidden_state: [batch_size, seq_len, d_model]
        '''
        residual = hidden_state
        hidden_state = self.layernorm(hidden_state)  # pre layernoem
        output = self.fc(hidden_state)
        return output + residual  # [batch_size, seq_len, d_model]


class DecoderLayer(nn.Module):
    def __init__(self, args):
        super(DecoderLayer, self).__init__()
        self.dec_self_attn = MultiHeadAttention(args)
        self.pos_ffn = PoswiseFeedForwardNet(args)

    def forward(self, hidden_state, dec_self_attn_mask):
        '''
            hidden_state: [batch_size, tgt_len, d_model]
            dec_self_attn_mask: [batch_size, tgt_len, tgt_len]
        '''
        # Attention层
        # hidden_state: [batch_size, tgt_len, d_model], dec_self_attn: [batch_size, n_heads, tgt_len, tgt_len]
        hidden_state, dec_self_attn = self.dec_self_attn(hidden_state, dec_self_attn_mask)

        # 非线性层
        hidden_state = self.pos_ffn(hidden_state)  # [batch_size, tgt_len, d_model]
        return hidden_state, dec_self_attn


class Decoder(nn.Module):
    def __init__(self, args, device):
        super(Decoder, self).__init__()
        self.device = device
        self.layers = nn.ModuleList([DecoderLayer(args) for _ in range(args.n_layers)])

    def forward(self, hidden_state, dec_self_attn_mask):
        '''
            hidden_state: [batch_size, tgt_len]
        '''
        dec_self_attns = []
        for layer in self.layers:
            # hidden_state: [batch_size, tgt_len, d_model]
            # dec_self_attn: [batch_size, n_heads, tgt_len, tgt_len]
            hidden_state, dec_self_attn = layer(hidden_state, dec_self_attn_mask)
   
            dec_self_attns.append(dec_self_attn)

        return hidden_state, dec_self_attns

class Embedding(nn.Module):
    def __init__ (self, args, device):
        super(Embedding, self).__init__()
        self.device = device
        self.tgt_emb = nn.Embedding(args.vocab_size, args.d_model)

        init.normal_(self.tgt_emb.weight, mean=0.0, std=(args.d_model)**(-args.embedding_std))
    
    def forward(self, X_input):
        tgt_emb = self.tgt_emb(X_input)
        return tgt_emb

class GPT2_prenorm_RoPE(nn.Module):
    def __init__(self, args, device):
        super(GPT2_prenorm_RoPE, self).__init__()

        self.device = device
        self.embedding = Embedding(args, device)
        self.decoder = Decoder(args, device)
        self.projection = nn.Linear(args.d_model, args.vocab_size)

        init.normal_(self.projection.weight, mean=0.0, std=(args.d_model)**(-args.embedding_std))
        # init.normal_(self.projection.bias, mean=0.0, std=1e-3)



    def forward(self, X_input):
        """
            dec_inputs: [batch_size, tgt_len]
        """
        hidden_state = self.embedding(X_input)

        dec_self_attn_mask = attn_mask(X_input, self.device)

        hidden_state, dec_self_attns = self.decoder(hidden_state, dec_self_attn_mask)
        
        dec_logits = self.projection(hidden_state)
        
        return dec_logits.view(-1, dec_logits.size(-1)), dec_self_attns
    

    def greedy_decoder(self,dec_input):

        projected, _ = self.forward(dec_input)

        projected = projected[-1,:].argmax()
        next_word = projected.item() 

        return next_word


    def test(self,sentence):
        dec_input = torch.tensor(sentence, dtype=torch.long, device=self.device).unsqueeze(0)

        output = self.greedy_decoder(dec_input)

        return output




