function nn_neuron_merge_vnn_eran_eval_all(evaluationPathBase)

% SETTINGS ----------------------------------------------------------------

rng(1)

% reduce NN for MAX_N random images
MAX_N = 10; 

% models
models = {
    % dataset, model, ONNX input format
    % eran
    "mnist", "mnist_relu_6_100.onnx", "BCSS"; ...           % 1
    "mnist", "mnist_relu_6_200.onnx", "BCSS"; ...           % 2
    "mnist", "mnist_relu_6_500.onnx", "BCSS"; ...           % 3
    "mnist", "mnist_sigmoid_6_200.onnx", "BCSS"; ...        % 4
    "mnist", "convMedGRELU__Point.onnx", "BCSS"; ...        % 5
    "mnist", "convMedGSIGMOID__Point.onnx", "BCSS"; ...     % 6
    "mnist", "convMedGTANH__Point.onnx", "BCSS"; ...        % 7
    % mnistfc
    "mnist", "mnist-net_256x2.onnx", "CSS"; ...             % 8
    "mnist", "mnist-net_256x4.onnx", "CSS"; ...             % 9
    "mnist", "mnist-net_256x6.onnx", "CSS"; ...             % 10
    % marabou
    "cifar", "cifar10_small.onnx", "BSSC"; ...              % 11
    "cifar", "cifar10_medium.onnx", "BSSC"; ...             % 12
    "cifar", "cifar10_large.onnx", "BSSC"; ...              % 13
    % Cifar2020
    "cifar", "convBigRELU__PGD.onnx", 'BSSC'; ...           % 14           
};

redRates = [0,0.1,0.2,0.4,0.6,0.8,1];
runs = {
    % model, method, perturbation, bucket tolerance, name;

    % mnistfc
    models{8, :}, 'static', 1e-2, 1e-2, [], '';
    models{9, :}, 'static', 1e-2, 1e-2, [], '';
    models{10, :}, 'static', 1e-2, 1e-2, [], '';

    % eran
    models{1, :}, 'static', 1e-3, 5e-3, [], '';
    models{2, :}, 'static', 1e-3, 5e-3, [], '';
    models{3, :}, 'static', 1e-3, 5e-3, [], '';
    models{4, :}, 'static', 1e-3, 5e-3, [], '';

    % ReLU radius/tols
    models{2, :}, 'static', 1e-3, 1e-4, [], 'tinyTol';
    models{2, :}, 'static', 1e-3, 1e-3, [], 'mediumTol';
    models{2, :}, 'static', 1e-3, 5e-3, [], 'mediumTol5';
    models{2, :}, 'static', 1e-3, 1e-2, [], 'largeTol';
    models{2, :}, 'static', 1e-3, 1e-1, [], 'hugeTol';

    models{2, :}, 'static', 2e-3, 1e-4, [], 'tinyTol';
    models{2, :}, 'static', 2e-3, 1e-3, [], 'mediumTol';
    models{2, :}, 'static', 2e-3, 5e-3, [], 'mediumTol5';
    models{2, :}, 'static', 2e-3, 1e-2, [], 'largeTol';
    models{2, :}, 'static', 2e-3, 1e-1, [], 'hugeTol';

    models{2, :}, 'static', 5e-3, 1e-4, [], 'tinyTol';
    models{2, :}, 'static', 5e-3, 1e-3, [], 'mediumTol';
    models{2, :}, 'static', 5e-3, 5e-3, [], 'mediumTol5';
    models{2, :}, 'static', 5e-3, 1e-2, [], 'largeTol';
    models{2, :}, 'static', 5e-3, 1e-1, [], 'hugeTol';

    % Sigmoid radius/tols
    models{4, :}, 'static', 1e-3, 1e-4, [], 'tinyTol';
    models{4, :}, 'static', 1e-3, 1e-3, [], 'mediumTol';
    models{4, :}, 'static', 1e-3, 5e-3, [], 'mediumTol5';
    models{4, :}, 'static', 1e-3, 1e-2, [], 'largeTol';
    models{4, :}, 'static', 1e-3, 1e-1, [], 'hugeTol';

    models{4, :}, 'static', 2e-3, 1e-4, [], 'tinyTol';
    models{4, :}, 'static', 2e-3, 1e-3, [], 'mediumTol';
    models{4, :}, 'static', 2e-3, 5e-3, [], 'mediumTol5';
    models{4, :}, 'static', 2e-3, 1e-2, [], 'largeTol';
    models{4, :}, 'static', 2e-3, 1e-1, [], 'hugeTol';

    models{4, :}, 'static', 5e-3, 1e-4, [], 'tinyTol';
    models{4, :}, 'static', 5e-3, 1e-3, [], 'mediumTol';
    models{4, :}, 'static', 5e-3, 5e-3, [], 'mediumTol5';
    models{4, :}, 'static', 5e-3, 1e-2, [], 'largeTol';
    models{4, :}, 'static', 5e-3, 1e-1, [], 'hugeTol';

    % eran CNN
    models{5, :}, 'dynamic', 1e-3, 5e-3, [], '';
    models{6, :}, 'dynamic', 1e-3, 5e-3, [], '';
    models{7, :}, 'dynamic', 1e-3, 5e-3, [], '';

    % additional runs
    models{6, :}, 'static', 1e-3, 5e-3, [], '';
    models{6, :}, 'dynamic', 1e-3, 5e-3, [], 'compress';
    models{6, :}, 'dynamic', 1e-3, 5e-3, [], 'normalize';

    % fully-automatic time
    models{4, :}, 'static', 1e-3, 1e-3, 0.0, 'automatic';
    models{4, :}, 'static', 1e-3, 1e-3, 0.1, 'automatic';
    models{4, :}, 'static', 1e-3, 1e-3, 0.2, 'automatic';
    models{4, :}, 'static', 1e-3, 1e-3, 0.3, 'automatic';
    models{4, :}, 'static', 1e-3, 1e-3, 0.4, 'automatic';
    models{4, :}, 'static', 1e-3, 1e-3, 0.5, 'automatic';
    models{4, :}, 'static', 1e-3, 1e-3, 0.6, 'automatic';
    models{4, :}, 'static', 1e-3, 1e-3, 0.7, 'automatic';
    models{4, :}, 'static', 1e-3, 1e-3, 0.8, 'automatic';
    models{4, :}, 'static', 1e-3, 1e-3, 0.9, 'automatic';
    models{4, :}, 'static', 1e-3, 1e-3, 1.0, 'automatic';

    models{6, :}, 'dynamic', 1e-2, 1e-2, 0.0, 'automatic';
    models{6, :}, 'dynamic', 1e-2, 1e-2, 0.1, 'automatic';
    models{6, :}, 'dynamic', 1e-2, 1e-2, 0.2, 'automatic';
    models{6, :}, 'dynamic', 1e-2, 1e-2, 0.3, 'automatic';
    models{6, :}, 'dynamic', 1e-2, 1e-2, 0.4, 'automatic';
    models{6, :}, 'dynamic', 1e-2, 1e-2, 0.5, 'automatic';
    models{6, :}, 'dynamic', 1e-2, 1e-2, 0.6, 'automatic';
    models{6, :}, 'dynamic', 1e-2, 1e-2, 0.7, 'automatic';
    models{6, :}, 'dynamic', 1e-2, 1e-2, 0.8, 'automatic';
    models{6, :}, 'dynamic', 1e-2, 1e-2, 0.9, 'automatic';
    models{6, :}, 'dynamic', 1e-2, 1e-2, 1.0, 'automatic';

    % marabou
    models{11, :}, 'dynamic', 1e-2, 1e-2, redRates, 'automatic';
    models{12, :}, 'dynamic', 1e-2, 1e-2, redRates, 'automatic';
    models{13, :}, 'dynamic', 1e-2, 1e-2, redRates, 'automatic';

    % cifar2020
    models{14, :}, 'dynamic', 1e-2, 1e-2, redRates, 'automatic';
    models{14, :}, 'dynamic', 1e-3, 1e-3, redRates, 'automatic';
};

% DATA --------------------------------------------------------------------

data = struct;

% load mnist
disp("Reading data...")
load('mnist.mat','testX','testY');

% normalize
testX = double(testX) / 255;
testY = categorical(testY);

data.mnist.X = testX;
data.mnist.Y = testY;

cifar10Data = tempdir;
url = 'https://www.cs.toronto.edu/~kriz/cifar-10-matlab.tar.gz';
helperCIFAR10Data.download(url,cifar10Data);
[~,~,testImages,testLabels] = helperCIFAR10Data.load(cifar10Data);
data.cifar.X = testImages;
data.cifar.Y = testLabels;

disp(" ")

% START -------------------------------------------------------------------

if isempty(which('neuralNetwork/computeReducedNetwork'))
    throw(CORAerror('CORA:specialError','Wrong git branch!'))
end

disp(['Storing results at: ' evaluationPathBase])
mkdir(evaluationPathBase);

for r=1:size(runs, 1)

    % read run
    run = runs(r, :);
    dataset = run{1};
    modelfile = run{2};
    ONNXinput = run{3};
    
    method = run{4};
    radius = run{5};
    tol = run{6};
    redRate = run{7};
    name = run{8};

    fprintf("\nRunning %i/%i: %s (%s) | %s, r=%.4f, tol=%.4f, redRate=%.2f \n", ...
        r, size(runs, 1), modelfile, name, method, radius, tol, redRate(1:min(1,length(redRate))));

    % read model
    try
        nn = neuralNetwork.readONNXNetwork(modelfile, false, ONNXinput, 'BC', 'dagnetwork');
    catch ME
        try
            % for convBigRELU__PGD.onnx
            nn = neuralNetwork.readONNXNetwork(modelfile, false, ONNXinput, 'BC', 'dlnetwork');
        catch
            rethrow(ME)
        end
    end

    % get normal form for correct neuron comparison
    nn = nn.getNormalForm();

    % parameters
    evParams = struct();
    % evParams.poly_method = "regression";
    % evParams.bound_approx = true;
    % evParams.num_generators = 10000;
    % evParams.reuse_bounds = true;

    % init evaluation
    evResult = struct;
    evResult.time = datestr(datetime());
    evResult.radius = radius;
    evResult.tol = tol;
    evResult.redRate = redRate;
    evResult.method = method;
    evResult.modelfile = modelfile;
    evTable = table;

    try

        % search for correctly predicted images
        cnt = 0;
        for i = randperm(length(data.(dataset).Y))

            cnt = cnt + 1;
            if cnt == MAX_N * 10
                throw(CORAerror('CORA:specialError','Unable to find enough correctly classified images.'))
            end
            
            if strcmp(dataset,'mnist')
                X = data.(dataset).X(i,:)';

                % todo: reshape?
                if strcmp(ONNXinput,'BSCC') || strcmp(ONNXinput,'BCSS')
                    X = reshape(reshape(X,28,28)',1,28,28);
                end

            else
                X = data.(dataset).X(:, :, :, i);
            end
            X = double(X);
            Y = data.(dataset).Y(i);
    
            % reshape to column vector
            X = reshape(X, [], 1);
            y = nn.evaluate(X);
    
            % check if correctly classified
            [~, pred] = max(y);
            correct = double(Y) == pred;
            if ~correct
                continue
            end
    
            % init evaluation table row
            evRow = struct;
            evRow.i = i;
            evRow.label = Y;
    
            % use image as center
            c = X;
            G = eye(length(c)) * radius;
            pZ = polyZonotope(c, G);
    
            % subtract correct label
            label_ind = double(Y);
            I = eye(10);
            I(:, label_ind) = I(:, label_ind) - 1;
            b0 = zeros(10, 1);
            nn_post = neuralNetwork({nnLinearLayer(I, b0)});
            
            % create copy of org network
            nn_i = nn.getNormalForm();
            evRow.orgNeurons = nn_i.getNumNeurons();
    
            % evaluate original network
            tic;
            % verify
            pred = nn_i.evaluate(pZ, evParams);
            pred = nn_post.evaluate(pred, evParams);
            bounds = interval(pred);
            u = supremum(bounds);
            evRow.orgVerified = all(u <= 0);
            if ~evRow.orgVerified
                continue
            end
            evRow.orgTime = toc;
            nn_i.reset();
            
            % compress
            if contains(name,'compress')
                compress = true;
            else
                compress = false;
            end

            % normalize
            if contains(name,'normalize') || compress
                % normalize to [0,1]
                I = interval(pZ);
                I = min(max(I,0),1);
                pZ = polyZonotope(I);
            end
    
            % compute reduced network
            try
                tic;
                % init
                redVerified = false;
                redRate_ = redRate;
                redRate_i = [];


                while ~redVerified
                    if ~isempty(redRate_)
                        redRate_i = redRate_(1);
                    end

                    % verify
                    [nn_i_red, pred_red] = nn_i.computeReducedNetwork(pZ, 'BucketType', method, 'BucketTol', tol, 'InputCompression', compress, 'ReductionRate', redRate_i);
                    pred_red = nn_post.evaluate(pred_red, evParams);
                    bounds = interval(pred_red);
                    u = supremum(bounds);
                    redVerified = all(u <= 0);

                    % check if verified
                    if redVerified || isempty(redRate_)
                        break
                    else
                        % move on to next reduction rate
                        redRate_(1) = [];
                        if isempty(redRate_)
                            % stop if all processed
                            break;
                        end
                    end
                end
    
                % save results
                evRow.redTime = toc;
                evRow.redVerified = redVerified;
                evRow.redNeurons = nn_i_red.getNumNeurons();
            catch ME
                % 0 remaining neurons in subsequent layers can result in an error
                % disp(ME.message)
                % save for error for later
                evResult.ME = ME;
                continue
            end

    
            % save results
            evTable = [evTable; struct2table(evRow)];
    
            if height(evTable) == MAX_N
                break;
            end
        end
    
        % compute reduction rate
        redRateComputed = rowfun( ...
            @(orgNeurons, redNeurons) sum(redNeurons(2:2:end-1))/sum(orgNeurons(2:2:end-1)), ...
            evTable(:, ["orgNeurons", "redNeurons"]), ...
            'OutputVariableNames', 'redRate');
    
        % store results
        evTable = [evTable, redRateComputed];
        evResult.table = evTable;
    catch ME
        disp(ME.message)
        evResult.ME = ME;
        evResult.table = evTable;
    end

    if ~isempty(name)
        name = ['_' name];
    end

    if isscalar(evResult.redRate)
        name = sprintf('%s_%.2f', name, evResult.redRate);
    end

    filename = sprintf( ...
        '%s/%s/%s/evResult_%s%s.mat', ...
        evaluationPathBase, method, radius, replace(modelfile, ".onnx", ""), name);
    dir = fileparts(filename);
    if ~isfolder(dir)
        mkdir(dir)
    end
    save(filename, 'evResult');

%         figure; hold on;
%         title(sprintf("%s: %s",  ...
%             method, replace(modelfile, '_', '\_')));
%         
%         horg = bar(mean(evTable.orgNeurons)', 'EdgeColor', 'none');
%         alpha(horg, 0.2)
%         set(gca,'ColorOrderIndex',1)
%         bar(mean(evTable.redNeurons)', 'EdgeColor', 'none');
%         drawnow;
end

% shut down pc afterwards
% system('shutdown -s')

end
