# Copyright (c) Meta Platforms, Inc. and affiliates.
# All rights reserved.

# This source code is licensed under the license found in the
# LICENSE file in the root directory of this source tree.
# --------------------------------------------------------
# References:
# DeiT: https://github.com/facebookresearch/deit
# BEiT: https://github.com/microsoft/unilm/tree/master/beit
# --------------------------------------------------------

import math
import sys
from typing import Iterable, Optional

import torch
from torchmetrics.classification import (
    BinaryAccuracy, BinaryF1Score, BinaryAUROC, BinaryStatScores,
    MulticlassAccuracy, MulticlassF1Score, MulticlassAUROC
)
from torchmetrics import MetricCollection
from sklearn.metrics import f1_score, balanced_accuracy_score, roc_auc_score

import utils.utils as misc
import utils.lr_sched as lr_sched


def train_one_epoch(model: torch.nn.Module, criterion: torch.nn.Module,
                    data_loader: Iterable, optimizer: torch.optim.Optimizer,
                    device: torch.device, epoch: int, loss_scaler, 
                    max_norm: float = 0, log_writer=None, args=None):
    model.train(True)
    metric_logger = misc.MetricLogger(delimiter="  ")
    metric_logger.add_meter('lr', misc.SmoothedValue(window_size=1, fmt='{value:.6f}'))
    header = 'Epoch: [{}]'.format(epoch)
    print_freq = 20

    accum_iter = args.accum_iter
    is_binary = args.data.num_labels == 1

    optimizer.zero_grad()

    if log_writer is not None:
        print('log_dir: {}'.format(log_writer.log_dir))

    for data_iter_step, batch in enumerate(metric_logger.log_every(data_loader, print_freq, header)):
        
        batch = [b.to(device, non_blocking=True) for b in batch]
        targets = batch[-1]
        
        # we use a per iteration (instead of per epoch) lr scheduler
        # if data_iter_step % accum_iter == 0:
            # lr_sched.adjust_learning_rate(optimizer, data_iter_step / len(data_loader) + epoch, args)

        with torch.cuda.amp.autocast(enabled=False):
            if args.retrieve:
                assert len(batch) == 4, "Batch should contain 4 elements: lesion, patient, image features and targets."
                features = torch.cat(batch[:3], dim=1)
                outputs = model(features)
            else:
                outputs = model(batch, args)
            loss = criterion(outputs, targets if is_binary else targets.squeeze(-1).long())

        loss_value = loss.item()

        if not math.isfinite(loss_value):
            print("Loss is {}, stopping training".format(loss_value))
            sys.exit(1)

        loss /= accum_iter
        loss_scaler(loss, optimizer, clip_grad=None,
                    parameters=model.parameters(), create_graph=False,
                    update_grad=(data_iter_step + 1) % accum_iter == 0)
        if (data_iter_step + 1) % accum_iter == 0:
            optimizer.zero_grad()

        torch.cuda.synchronize()

        metric_logger.update(loss=loss_value)
        min_lr = 10.
        max_lr = 0.
        for group in optimizer.param_groups:
            min_lr = min(min_lr, group["lr"])
            max_lr = max(max_lr, group["lr"])

        metric_logger.update(lr=max_lr)

        loss_value_reduce = misc.all_reduce_mean(loss_value)
        if log_writer is not None and (data_iter_step + 1) % accum_iter == 0:
            """ We use epoch_1000x as the x-axis in tensorboard.
            This calibrates different curves when batch size changes.
            """
            epoch_1000x = int((data_iter_step / len(data_loader) + epoch) * 1000)
            log_writer.add_scalar('loss', loss_value_reduce, epoch_1000x)
            log_writer.add_scalar('lr', max_lr, epoch_1000x)

    # gather the stats from all processes
    metric_logger.synchronize_between_processes()
    print("Averaged stats:", metric_logger)
    return {k: meter.global_avg for k, meter in metric_logger.meters.items()}


@torch.no_grad()
def evaluate(data_loader, model, criterion, device, args=None):
    metric_logger = misc.MetricLogger(delimiter="  ")
    header = 'Test:'

    # switch to evaluation mode
    model.eval()

    is_binary = args.data.num_labels == 1

    if is_binary:
        metrics = MetricCollection({
            'acc1': BinaryAccuracy(),
            'f1': BinaryF1Score(),
            'auroc': BinaryAUROC(),
            'stats': BinaryStatScores()  # TP, FP, TN, FN
        }).to(device)
    else:
        metrics = MetricCollection({
            'acc1': MulticlassAccuracy(num_classes=args.data.num_labels, average='micro'),
            'f1_macro': MulticlassF1Score(num_classes=args.data.num_labels, average='macro'),
            'f1_weighted': MulticlassF1Score(num_classes=args.data.num_labels, average='weighted'),
            'auroc': MulticlassAUROC(num_classes=args.data.num_labels, average='macro'),
        }).to(device)

    for batch in metric_logger.log_every(data_loader, 10, header):
        
        batch = [b.to(device, non_blocking=True) for b in batch]
        target = batch[-1]

        # compute output
        with torch.cuda.amp.autocast(enabled=False):
            if args.retrieve:
                assert len(batch) == 4, "Batch should contain 4 elements: lesion, patient, image features and targets."
                features = torch.cat(batch[:3], dim=1)
                output = model(features)
            else:
                output = model(batch, args)
            loss = criterion(output, target if is_binary else target.squeeze(-1).long())

        metric_logger.update(loss=loss.item())

        # Update metrics
        if is_binary:
            preds = torch.sigmoid(output).squeeze()
            metrics.update(preds, target.squeeze(-1).int())
        else:
            preds = torch.softmax(output, dim=1)
            metrics.update(preds, target.squeeze(-1).long())
    
    results = metrics.compute()
    if is_binary:
        tp, fp, tn, fn, _ = results['stats']
        sensitivity = tp / (tp + fn + 1e-8)
        specificity = tn / (tn + fp + 1e-8)
        balanced_acc = (sensitivity + specificity) / 2
        results['balanced_acc'] = balanced_acc
        del results['stats']
    
    # gather the stats from all processes
    metric_logger.synchronize_between_processes()
    avg_loss = metric_logger.loss.global_avg

    print('* ' + '  '.join(
        f'{k}: {v.item() * 100:.2f}%' if k in ['acc1', 'balanced_acc'] else f'{k}: {v.item():.4f}'
        for k, v in results.items()
    ) + f' | loss: {avg_loss:.4f}'
    )

    return {
        'loss': avg_loss,
        **{
            k: v.item() * 100 if k in ['acc1', 'balanced_acc'] else v.item()
            for k, v in results.items()
        }
    }

@torch.no_grad()
def concat_all_gather(tensor):
    """
    Performs all_gather operation on the provided tensors.
    """
    tensor = tensor.cuda()
    tensors_gather = [torch.zeros_like(tensor) for _ in range(torch.distributed.get_world_size())]
    torch.distributed.all_gather(tensors_gather, tensor, async_op=False)
    return torch.cat(tensors_gather, dim=0)

@torch.no_grad()
def extract_features(
        data_loader,
        model,
        device,
        args=None,
        return_targets_and_preds: bool = True
):
    metric_logger = misc.MetricLogger(delimiter="  ")
    header = 'Test:'

    # switch to evaluation mode
    model.eval()
    targets = []
    features = []

    for batch in metric_logger.log_every(data_loader, 10, header):
        
        batch = [b.to(device, non_blocking=True) for b in batch]
        target = batch[-1]
        # target = target.to(device, non_blocking=True)
        
        # compute output
        with torch.cuda.amp.autocast(enabled=False):
            output_feat = model(batch, args, return_features=True)

        targets.append(target.cpu())
        features.append(output_feat.cpu())

    # gather the stats from all processes
    metric_logger.synchronize_between_processes()
    stats = {k: meter.global_avg for k, meter in metric_logger.meters.items()}

    if return_targets_and_preds:
        targets_tensor = torch.cat(targets)
        features_tensor = torch.cat(features)
        if torch.distributed.is_initialized():
            targets_tensor = concat_all_gather(targets_tensor)
            features_tensor = concat_all_gather(features_tensor)
        stats["targets"] = targets_tensor
        stats["features"] = features_tensor

    return stats

@torch.no_grad()
def knn_classifier(train_features, train_labels, test_features, test_labels, k, T, use_cuda=True, num_classes=2, num_chunks=1):
    top1, total = 0.0, 0
    all_preds, all_targets, all_probs = [], [], []

    train_features = train_features.t()
    num_test_images = test_labels.shape[0]
    imgs_per_chunk = num_test_images // num_chunks
    retrieval_one_hot = torch.zeros(k, num_classes)
    if use_cuda:
        retrieval_one_hot = retrieval_one_hot.cuda()
    
    for idx in range(0, num_test_images, imgs_per_chunk):
        # get the features for test images
        features = test_features[
            idx : min((idx + imgs_per_chunk), num_test_images), :
        ]
        targets = test_labels[idx : min((idx + imgs_per_chunk), num_test_images)]
        batch_size = targets.shape[0]

        # calculate the dot product and compute top-k neighbors
        similarity = torch.mm(features, train_features)
        distances, indices = similarity.topk(k, largest=True, sorted=True)
        candidates = train_labels.view(1, -1).expand(batch_size, -1)
        retrieved_neighbors = torch.gather(candidates, 1, indices).long()

        retrieval_one_hot.resize_(batch_size * k, num_classes).zero_()
        retrieval_one_hot.scatter_(1, retrieved_neighbors.view(-1, 1), 1)
        distances_transform = distances.clone().div_(T).exp_()
        probs = torch.sum(
            torch.mul(
                retrieval_one_hot.view(batch_size, -1, num_classes),
                distances_transform.view(batch_size, -1, 1),
            ),
            1,
        )
        _, predictions = probs.sort(1, True)
        pred_labels = predictions[:, 0]

        # collect for metrics
        all_preds.append(pred_labels.cpu())
        all_targets.append(targets.cpu())
        if num_classes == 2:
            all_probs.append(probs[:, 1].cpu())

        # find the predictions that match the target
        correct = predictions.eq(targets.data.view(-1, 1))
        top1 = top1 + correct.narrow(1, 0, 1).sum().item()
        # correct = pred_labels.eq(targets)
        # top1 += correct.sum().item()
        total += targets.size(0)
    
    # concat all chunks
    all_preds = torch.cat(all_preds).numpy()
    all_targets = torch.cat(all_targets).numpy()
    if num_classes == 2:
        all_probs = torch.cat(all_probs).numpy()
        f1 = f1_score(all_targets, all_preds)
        bal_acc = balanced_accuracy_score(all_targets, all_preds)
        auc = roc_auc_score(all_targets, all_probs)
        top1 = top1 * 100.0 / total
        return top1, f1, bal_acc, auc
    else:
        top1 = top1 * 100.0 / total
        return top1