# Copyright (c) Meta Platforms, Inc. and affiliates.
# All rights reserved.

# This source code is licensed under the license found in the
# LICENSE file in the root directory of this source tree.
# --------------------------------------------------------
# References:
# DeiT: https://github.com/facebookresearch/deit
# BEiT: https://github.com/microsoft/unilm/tree/master/beit
# --------------------------------------------------------
import math
import sys
from typing import Iterable

import torch

import utils.utils as misc
# import util.lr_sched as lr_sched


def train_one_epoch(model: torch.nn.Module, data_loader: Iterable,
                    optimizer: torch.optim.Optimizer, device: torch.device, 
                    epoch: int, loss_scaler=None, log_writer=None, args=None):
    # model.train(True)
    metric_logger = misc.MetricLogger(delimiter="  ")
    metric_logger.add_meter('lr', misc.SmoothedValue(window_size=1, fmt='{value:.6f}'))
    header = 'Epoch: [{}]'.format(epoch)
    print_freq = 20

    accum_iter = args.accum_iter

    optimizer.zero_grad()

    if log_writer is not None:
        print('log_dir: {}'.format(log_writer.log_dir))

    for data_iter_step, batch in enumerate(metric_logger.log_every(data_loader, print_freq, header)):

        # we use a per iteration (instead of per epoch) lr scheduler
        # if data_iter_step % accum_iter == 0:
            # lr_sched.adjust_learning_rate(optimizer, data_iter_step / len(data_loader) + epoch, args)

        # If the patient data are available then follow the proposed nested approach
        if args.trace_pp.metadata_dir not in (None, 'None'):
            patient_data, lesion_data = batch
            patient_data = [p.to(device, non_blocking=True) for p in patient_data]
            lesion_data[0] = [l.to(device, non_blocking=True) for l in lesion_data[0]]
            lesion_data[1] = [l.to(device, non_blocking=True) for l in lesion_data[1]]

            with torch.cuda.amp.autocast(enabled=False):
                total_loss, inner_loss, outer_loss = model(patient_data, lesion_data, mode="nested_loss")

        # If no patient data are available follow a standard flatten training protocol with inner loss only
        # Case: PH2 continual pre-training
        else:
            lesion_tab, images, lesion_targets = batch
            lesion_data = (lesion_tab.to(device, non_blocking=True),
                           images.to(device, non_blocking=True),
                           lesion_targets.to(device, non_blocking=True))

            with torch.cuda.amp.autocast(enabled=False):
                total_loss, inner_loss = model(lesion_data, mode="flattened_loss")
                outer_loss = None

        loss_value = total_loss.item()
        inner_loss_value = inner_loss.item()
        outer_loss_value = outer_loss.item() if outer_loss is not None else 0.0

        # if not math.isfinite(loss_value):
        #     print("Loss is {}, stopping training".format(loss_value))
        #     sys.exit(1)

        total_loss /= accum_iter
        inner_loss /= accum_iter
        if outer_loss is not None:
            outer_loss /= accum_iter
        loss_scaler(total_loss, optimizer, parameters=model.parameters(), update_grad=(data_iter_step + 1) % accum_iter == 0)
        if (data_iter_step + 1) % accum_iter == 0:
            optimizer.zero_grad()

        torch.cuda.synchronize()

        metric_logger.update(loss=loss_value)
        metric_logger.update(inner_loss=inner_loss_value)
        if outer_loss is not None:
            metric_logger.update(outer_loss=outer_loss_value)

        lr = optimizer.param_groups[0]["lr"]
        metric_logger.update(lr=lr)

        loss_value_reduce = misc.all_reduce_mean(loss_value)
        inner_loss_value_reduce = misc.all_reduce_mean(inner_loss_value)
        outer_loss_value_reduce = misc.all_reduce_mean(outer_loss_value) if outer_loss is not None else 0.0
        if log_writer is not None and (data_iter_step + 1) % accum_iter == 0:
            """ We use epoch_1000x as the x-axis in tensorboard.
            This calibrates different curves when batch size changes.
            """
            epoch_1000x = int((data_iter_step / len(data_loader) + epoch) * 1000)
            log_writer.add_scalar('train_total_loss', loss_value_reduce, epoch_1000x)
            log_writer.add_scalar('train_inner_loss', inner_loss_value_reduce, epoch_1000x)
            if outer_loss is not None:
                log_writer.add_scalar('train_outer_loss', outer_loss_value_reduce, epoch_1000x)
            log_writer.add_scalar('lr', lr, epoch_1000x)

    # gather the stats from all processes
    metric_logger.synchronize_between_processes()
    print("Averaged stats:", metric_logger)
    return {k: meter.global_avg for k, meter in metric_logger.meters.items()}