from typing import List, Dict
import math
from typing import List, Dict, Optional
from datetime import datetime
from typing import Optional
import aiosqlite
from typing import List, Dict, Optional, Tuple
from datetime import timedelta, datetime
from functools import lru_cache
import sqlite3
import pandas as pd
import os
import networkx as nx

FORUM_DB_PATH = 'data/ForumDB/syn_100.db'
USER_DB_PATH = 'data/UserDB/sys_100.db'


def init_db_forum(db_path=FORUM_DB_PATH):
    """
    Initialize forum database by deleting the existing file (if any) and creating required tables.

    Args:
        db_path (str): Path to the SQLite database file.
    """
    # If the database file already exists, delete it
    if os.path.exists(db_path):
        os.remove(db_path)
        print(f"Deleted existing database file: {db_path}")

    # Ensure the directory exists
    os.makedirs(os.path.dirname(db_path), exist_ok=True)

    with sqlite3.connect(db_path) as conn:
        # Create posts table
        conn.execute('''
        CREATE TABLE IF NOT EXISTS posts (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id TEXT NOT NULL,
            content TEXT NOT NULL,
            score INTEGER DEFAULT 0,
            belief TEXT,
            type TEXT,    -- New column for post type
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )''')

        # Create reactions table
        conn.execute('''
        CREATE TABLE IF NOT EXISTS reactions (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id TEXT NOT NULL,
            post_id INTEGER NOT NULL,
            type TEXT CHECK(type IN ('repost', 'like', 'unlike')) NOT NULL,  -- New column for reaction type
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (post_id) REFERENCES posts(id)
        )''')

        # Create post_references table
        conn.execute('''
        CREATE TABLE IF NOT EXISTS post_references (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            reference_id INTEGER NOT NULL,  -- ID of the referenced post
            repost_id INTEGER NOT NULL,     -- ID of the repost
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (reference_id) REFERENCES posts(id),
            FOREIGN KEY (repost_id) REFERENCES posts(id),
            UNIQUE(reference_id, repost_id)  -- Ensure each reference_id and repost_id combination is unique
        )''')

        # Create indexes
        conn.execute('CREATE INDEX IF NOT EXISTS idx_posts_user_id ON posts(user_id)')
        conn.execute('CREATE INDEX IF NOT EXISTS idx_reactions_post_id ON reactions(post_id)')
        conn.execute('CREATE INDEX IF NOT EXISTS idx_post_references_ref_id ON post_references(reference_id)')

        conn.commit()
        print(f"Initialized new database file: {db_path}")


def update_posts_score_by_date(
    target_date: str,
    db_path: str = FORUM_DB_PATH
) -> bool:
    """
    Update the score of posts based on reactions for a specific date.

    Args:
        target_date: The date for which to calculate and update scores (format: 'YYYY-MM-DD').
        db_path: Path to the SQLite database.

    Returns:
        bool: True if the update was successful, False otherwise.
    """
    try:
        with sqlite3.connect(db_path) as conn:
            conn.execute("BEGIN")

            # Calculate the total score for each post based on reactions for the target date
            # Assign scores as follows:
            # - 'like' = +1
            # - 'unlike' = -1
            # - 'repost' = +0 (no impact on score)
            cursor = conn.execute('''
                SELECT post_id,
                       SUM(CASE WHEN type = 'like' THEN 1
                               WHEN type = 'unlike' THEN -1
                               ELSE 0 END) as total_score
                FROM reactions
                WHERE DATE(created_at) = ?
                GROUP BY post_id
            ''', (target_date,))

            # Fetch all results
            post_scores = cursor.fetchall()

            # Update the posts table with the calculated scores
            for post_id, total_score in post_scores:
                conn.execute('''
                    UPDATE posts
                    SET score = ?
                    WHERE id = ?
                ''', (total_score, post_id))

            conn.commit()
            return True

    except sqlite3.Error as e:
        print(f"Error updating posts score: {e}")
        conn.rollback()  # Rollback in case of error
        return False


def update_posts_score_by_date_range(
    start_date: str = '2023-01-01',
    end_date: str = '2023-06-16',
    db_path: str = "FORUM_DB_PATH"
) -> tuple[bool, pd.DataFrame]:
    """
    Update the score of posts based on reactions for a specific date range.

    Args:
        start_date: The start date of the range (format: 'YYYY-MM-DD').
        end_date: The end date of the range (format: 'YYYY-MM-DD').
        db_path: Path to the SQLite database.

    Returns:
        tuple: (success: bool, scores_df: pd.DataFrame)
            - success: True if update was successful
            - scores_df: DataFrame containing post IDs and their updated scores
    """
    try:
        with sqlite3.connect(db_path) as conn:
            conn.execute("BEGIN")

            # Reset scores to 0 for posts within the date range
            conn.execute('''
                UPDATE posts 
                SET score = 0 
                WHERE DATE(created_at) BETWEEN ? AND ?
            ''', (start_date, end_date))

            # Calculate the total score for each post based on reactions within the date range
            cursor = conn.execute('''
                SELECT post_id,
                       SUM(CASE WHEN type = 'like' THEN 1
                               WHEN type = 'unlike' THEN -1
                               ELSE 0 END) as total_score
                FROM reactions
                WHERE DATE(created_at) BETWEEN ? AND ?
                GROUP BY post_id
            ''', (start_date, end_date))

            # Fetch all results
            post_scores = cursor.fetchall()

            # Update the posts table with the calculated scores
            for post_id, total_score in post_scores:
                cursor.execute('''
                    SELECT type FROM posts WHERE id = ?
                ''', (post_id,))
                post_type_row = cursor.fetchone()

                if post_type_row and post_type_row[0] == 'repost':
                    root_post_id = find_root_post(post_id, db_path)
                    if root_post_id:
                        path = root_post_id['path']
                        path.append(root_post_id['post_id'])
                        for id in root_post_id['path']:
                            conn.execute('''
                                UPDATE posts
                                SET score = score + ?
                                WHERE id = ?
                            ''', (total_score, id))
                else:
                    conn.execute('''
                        UPDATE posts
                        SET score = ?
                        WHERE id = ?
                    ''', (total_score, post_id))

            # Get final scores for all posts in date range
            scores_df = pd.read_sql_query('''
                SELECT id as post_id, score
                FROM posts 
            ''', conn)

            conn.commit()
            return True, scores_df

    except sqlite3.Error as e:
        print(f"Error updating posts score: {e}")
        conn.rollback()
        return False, pd.DataFrame()


def fetch_posts_score_by_date_range(
    start_date: str = '2023-01-01',
    end_date: str = '2023-06-16',
    db_path: str = "FORUM_DB_PATH"
) -> pd.DataFrame:
    """
    Fetch cumulative post scores based on reactions for a specific date range.

    Args:
        start_date: The start date of the range (format: 'YYYY-MM-DD').
        end_date: The end date of the range (format: 'YYYY-MM-DD').
        db_path: Path to the SQLite database.

    Returns:
        pd.DataFrame: DataFrame containing post IDs and their cumulative scores.
    """
    try:
        with sqlite3.connect(db_path) as conn:
            # 获取所有帖子
            posts_df = pd.read_sql_query('''
                SELECT id AS post_id, type
                FROM posts
                WHERE DATE(created_at) BETWEEN ? AND ?
            ''', conn, params=(start_date, end_date))

            # 获取点赞和取消点赞数据
            reactions_df = pd.read_sql_query('''
                SELECT post_id,
                       SUM(CASE WHEN type = 'like' THEN 1
                               WHEN type = 'unlike' THEN -1
                               ELSE 0 END) as total_score
                FROM reactions
                WHERE DATE(created_at) BETWEEN ? AND ?
                GROUP BY post_id
            ''', conn, params=(start_date, end_date))

        # 合并帖子信息和 reactions 统计数据
        scores_df = posts_df.merge(reactions_df, on="post_id", how="left").fillna(0)

        # 处理 repost 级联加分
        for index, row in scores_df.iterrows():
            if row['type'] == 'repost':
                root_post_id = find_root_post(row['post_id'], db_path)
                if root_post_id:
                    path = root_post_id['path']
                    path.append(root_post_id['post_id'])
                    for post_id in path:
                        scores_df.loc[scores_df['post_id'] == post_id, 'total_score'] += row['total_score']

        # 仅保留最终需要的列
        scores_df = scores_df[['post_id', 'total_score']]

        return scores_df

    except sqlite3.Error as e:
        print(f"Error fetching posts score: {e}")
        return pd.DataFrame()


def get_user_net_likes_and_post_interactions(
    user_id: str,
    start_date: pd.Timestamp,
    end_date: pd.Timestamp,
    db_path: str = FORUM_DB_PATH
) -> Tuple[int, pd.DataFrame]:
    """
    获取用户在指定时间范围内的净点赞数量及其发布的帖子的互动详情。

    Args:
        user_id (str): 用户的ID。
        start_date (pd.Timestamp): 开始时间。
        end_date (pd.Timestamp): 结束时间。
        db_path (str): 数据库路径。默认为 FORUM_DB_PATH。

    Returns:
        Tuple[int, pd.DataFrame]:
            - int: 用户获得的净点赞数量（总点赞数 - 总取消点赞数）。
            - pd.DataFrame: 包含用户发布的帖子的互动详情，列包括：
                - post_id: 帖子ID
                - created_at: 帖子发布时间
                - like_users: 点赞的用户列表
                - unlike_users: 取消点赞的用户列表
                - repost_users: 转发的用户列表
    """
    try:
        with sqlite3.connect(db_path) as conn:
            conn.row_factory = sqlite3.Row

            # 1. 计算用户获得的净点赞数量
            query_net_likes = """
                SELECT 
                    SUM(CASE WHEN r.type = 'like' THEN 1 ELSE 0 END) as total_likes,
                    SUM(CASE WHEN r.type = 'unlike' THEN 1 ELSE 0 END) as total_unlikes
                FROM reactions r
                JOIN posts p ON r.post_id = p.id
                WHERE p.user_id = ?
                AND r.created_at >= ? AND r.created_at <= ?
            """
            cursor = conn.execute(query_net_likes, (user_id, start_date.strftime('%Y-%m-%d'), end_date.strftime('%Y-%m-%d')))
            result = cursor.fetchone()
            total_likes = result["total_likes"] if result["total_likes"] else 0
            total_unlikes = result["total_unlikes"] if result["total_unlikes"] else 0
            net_likes = total_likes - total_unlikes

            # 2. 获取用户发布的帖子的互动详情
            query_post_interactions = """
                SELECT 
                    p.id as post_id,
                    p.created_at,
                    GROUP_CONCAT(DISTINCT CASE WHEN r.type = 'like' THEN r.user_id END) as like_users,
                    GROUP_CONCAT(DISTINCT CASE WHEN r.type = 'unlike' THEN r.user_id END) as unlike_users,
                    GROUP_CONCAT(DISTINCT CASE WHEN r.type = 'repost' THEN r.user_id END) as repost_users
                FROM posts p
                LEFT JOIN reactions r ON p.id = r.post_id
                WHERE p.user_id = ?
                AND p.created_at >= ? AND p.created_at <= ?
                GROUP BY p.id
            """
            cursor = conn.execute(query_post_interactions, (user_id, start_date.strftime('%Y-%m-%d'), end_date.strftime('%Y-%m-%d')))
            rows = cursor.fetchall()

            # 将结果转换为 DataFrame
            data = []
            for row in rows:
                data.append({
                    "post_id": row["post_id"],
                    "created_at": row["created_at"],
                    "like_users": row["like_users"].split(",") if row["like_users"] else [],
                    "unlike_users": row["unlike_users"].split(",") if row["unlike_users"] else [],
                    "repost_users": row["repost_users"].split(",") if row["repost_users"] else []
                })
            df = pd.DataFrame(data)

            return net_likes, df

    except sqlite3.Error as e:
        print(f"Error getting user net likes and post interactions: {e}")
        return 0, pd.DataFrame()


def create_post_db(
    user_id: str,
    content: str,
    belief: Optional[str] = None,
    type: Optional[str] = None,  # New parameter for post type
    created_at: Optional[pd.Timestamp] = None,
    db_path: str = FORUM_DB_PATH
) -> Optional[int]:
    """
    Create a new post in the forum database.

    Args:
        user_id (str): The ID of the user creating the post.
        content (str): The content of the post.
        belief (Optional[str]): The belief of the user (default: None).
        type (Optional[str]): The type of the post (default: None).
        created_at (Optional[pd.Timestamp]): Pandas Timestamp for the post. Defaults to current time if None.
        db_path (str): Path to database. Defaults to FORUM_DB_PATH.

    Returns:
        Optional[int]: The ID of the created post, or None if creation failed.
    """
    try:
        with sqlite3.connect(db_path) as conn:
            if created_at is None:
                created_at = pd.Timestamp.now()

            cursor = conn.execute('''
                INSERT INTO posts (user_id, content, belief, type, created_at)
                VALUES (?, ?, ?, ?, ?)
            ''', (user_id, content, belief, type, created_at.strftime('%Y-%m-%d %H:%M:%S')))

            post_id = cursor.lastrowid
            conn.commit()
            return post_id

    except sqlite3.Error as e:
        print(f"Error creating post: {e}")
        return None


def repost_db(
    reference_id: str,
    user_id: str,
    content: str,
    belief: Optional[str] = None,
    created_at: Optional[pd.Timestamp] = None,
    db_path: str = FORUM_DB_PATH
) -> Optional[int]:
    """
    Create a repost referencing another post and record a reaction of type 'repost'.

    Args:
        reference_id (str): ID of the post being referenced.
        user_id (str): ID of user making the repost.
        content (str): Content of the repost.
        belief (Optional[str]): The belief of the user (default: None).
        created_at (Optional[pd.Timestamp]): Timestamp for the post. Defaults to current time.
        db_path (str): Path to database.

    Returns:
        Optional[int]: ID of created repost, None if failed.
    """
    try:
        with sqlite3.connect(db_path) as conn:
            # Start transaction
            conn.execute("BEGIN")

            # Verify reference exists
            cursor = conn.execute('SELECT 1 FROM posts WHERE id = ?', (reference_id,))
            if not cursor.fetchone():
                print(f"Reference post {reference_id} not found")
                return None

            if created_at is None:
                created_at = pd.Timestamp.now()

            # Create the repost with type set to 'repost'
            cursor = conn.execute('''
                INSERT INTO posts (user_id, content, belief, type, created_at)
                VALUES (?, ?, ?, ?, ?)
            ''', (user_id, content, belief, 'repost', created_at.strftime('%Y-%m-%d %H:%M:%S')))

            new_post_id = cursor.lastrowid

            # Create reference linking the original post and the new repost
            conn.execute('''
                INSERT INTO post_references (reference_id, repost_id, created_at)
                VALUES (?, ?, ?)
            ''', (reference_id, new_post_id, created_at.strftime('%Y-%m-%d %H:%M:%S')))

            # Record a reaction of type 'repost'
            conn.execute('''
                INSERT INTO reactions (user_id, post_id, type, created_at)
                VALUES (?, ?, ?, ?)
            ''', (user_id, reference_id, 'repost', created_at.strftime('%Y-%m-%d %H:%M:%S')))

            conn.commit()
            return new_post_id

    except sqlite3.Error as e:
        print(f"Error creating repost: {e}")
        conn.rollback()  # Rollback in case of error
        return None


async def like_post_db(
    user_id: str,
    post_id: int,
    created_at: Optional[pd.Timestamp] = None,
    db_path: str = FORUM_DB_PATH
) -> bool:
    """Like a post (type 'like') - Only updates reactions table."""
    try:
        async with aiosqlite.connect(db_path) as conn:
            await conn.execute("BEGIN")

            # Check if post exists
            cursor = await conn.execute('SELECT id FROM posts WHERE id = ?', (post_id,))
            post = await cursor.fetchone()
            if not post:
                print(f"Post {post_id} not found")
                return False

            # Get current reaction if exists
            cursor = await conn.execute(
                'SELECT type FROM reactions WHERE user_id = ? AND post_id = ?',
                (user_id, post_id)
            )
            reaction = await cursor.fetchone()

            if created_at is None:
                created_at = pd.Timestamp.now()

            if reaction is None:
                # New reaction
                await conn.execute('''
                    INSERT INTO reactions (user_id, post_id, type, created_at)
                    VALUES (?, ?, ?, ?)
                ''', (user_id, post_id, 'like', created_at.strftime('%Y-%m-%d %H:%M:%S')))
            else:
                # Update existing reaction
                old_type = reaction[0]
                if old_type != 'like':  # Only update if not already liked
                    await conn.execute('''
                        UPDATE reactions 
                        SET type = ?, created_at = ?
                        WHERE user_id = ? AND post_id = ?
                    ''', ('like', created_at.strftime('%Y-%m-%d %H:%M:%S'), user_id, post_id))

            await conn.commit()
            return True

    except aiosqlite.Error as e:
        print(f"Error liking post: {e}")
        return False


async def unlike_post_db(
    user_id: str,
    post_id: int,
    created_at: Optional[pd.Timestamp] = None,
    db_path: str = FORUM_DB_PATH
) -> bool:
    """Unlike a post (type 'unlike') - Only updates reactions table."""
    try:
        async with aiosqlite.connect(db_path) as conn:
            await conn.execute("BEGIN")

            # Check if post exists
            cursor = await conn.execute('SELECT id FROM posts WHERE id = ?', (post_id,))
            post = await cursor.fetchone()
            if not post:
                print(f"Post {post_id} not found")
                return False

            # Get current reaction if exists
            cursor = await conn.execute(
                'SELECT type FROM reactions WHERE user_id = ? AND post_id = ?',
                (user_id, post_id)
            )
            reaction = await cursor.fetchone()

            if created_at is None:
                created_at = pd.Timestamp.now()

            if reaction is None:
                # New reaction
                await conn.execute('''
                    INSERT INTO reactions (user_id, post_id, type, created_at)
                    VALUES (?, ?, ?, ?)
                ''', (user_id, post_id, 'unlike', created_at.strftime('%Y-%m-%d %H:%M:%S')))
            else:
                # Update existing reaction
                old_type = reaction[0]
                if old_type != 'unlike':  # Only update if not already unliked
                    await conn.execute('''
                        UPDATE reactions 
                        SET type = ?, created_at = ?
                        WHERE user_id = ? AND post_id = ?
                    ''', ('unlike', created_at.strftime('%Y-%m-%d %H:%M:%S'), user_id, post_id))

            await conn.commit()
            return True

    except aiosqlite.Error as e:
        print(f"Error unliking post: {e}")
        return False


async def execute_forum_actions(
    forum_args: List[Dict],
    user_id: str,
    db_path: str,
    belief: Optional[str] = None,
    created_at: Optional[pd.Timestamp] = None,
) -> None:
    """
    Execute forum actions based on the provided action list in forum_args.
    """
    if created_at is None:
        created_at = pd.Timestamp.now()

    for action in forum_args:
        action_type = action.get("action")
        post_id = action.get("post_id")
        content = action.get("reason", "")

        if action_type == "repost":
            # Repost action
            if not content:
                continue  # Skip if content is missing

            repost_db(
                reference_id=int(post_id),
                user_id=user_id,
                content=content,
                belief=belief,
                created_at=created_at,
                db_path=db_path
            )

        elif action_type == "like":
            # Like action
            await like_post_db(
                user_id=user_id,
                post_id=post_id,
                created_at=created_at,
                db_path=db_path
            )

        elif action_type == "unlike":
            # Unlike action
            await unlike_post_db(
                user_id=user_id,
                post_id=post_id,
                created_at=created_at,
                db_path=db_path
            )


def get_all_users_posts_db(
    end_date: Optional[pd.Timestamp] = None,
    db_path: str = FORUM_DB_PATH
) -> Dict[str, List[Dict]]:
    """
    Get all posts by all users within a date range, including all post-related attributes.

    Args:
        end_date (Optional[pd.Timestamp]): End of the date range (inclusive).
        db_path (str): Path to the database. Defaults to FORUM_DB_PATH.

    Returns:
        dict: A dictionary where the key is the user ID and the value is a list of post dictionaries.
              Each post dictionary contains:
                - id: Post ID
                - user_id: User ID of the poster
                - content: Post content
                - score: Current post score
                - belief: Belief of the user
                - type: Type of the post
                - created_at: Timestamp of the post
    """
    try:
        with sqlite3.connect(db_path) as conn:
            conn.row_factory = sqlite3.Row  # Enable column access by name
            cursor = conn.cursor()

            # Base query to fetch posts for all users
            query = """
                SELECT 
                    id,
                    user_id,
                    content,
                    score,
                    belief,
                    type,
                    created_at
                FROM posts
                WHERE type != 'repost'
            """
            params = []

            # Add date range filter if end_date is provided
            if end_date:
                query += " AND created_at <= ? ORDER BY created_at DESC"
                params.append(end_date.strftime('%Y-%m-%d %H:%M:%S'))
            else:
                query += " ORDER BY created_at DESC"

            # Execute the query
            cursor.execute(query, params)
            rows = cursor.fetchall()

            # Organize posts by user_id
            user_posts = {}
            for row in rows:
                post = dict(row)
                user_id = post["user_id"]

                if user_id not in user_posts:
                    user_posts[user_id] = []
                user_posts[user_id].append(post)

            return user_posts

    except sqlite3.Error as e:
        print(f"Error getting all users' posts: {e}")
        return {}


def get_user_posts_db(
    user_id: str,
    end_date: Optional[pd.Timestamp] = None,
    db_path: str = FORUM_DB_PATH
) -> List[Dict]:
    """
    Get all posts by a specific user within a date range, including all post-related attributes.

    Args:
        user_id (str): The ID of the user whose posts are being retrieved.
        end_date (Optional[pd.Timestamp]): End of the date range (inclusive).
        db_path (str): Path to the database. Defaults to FORUM_DB_PATH.

    Returns:
        list[dict]: A list of post dictionaries, each containing:
            - id: Post ID
            - user_id: User ID of the poster
            - content: Post content
            - score: Current post score
            - belief: Belief of the user
            - type: Type of the post
            - created_at: Timestamp of the post
    """
    try:
        with sqlite3.connect(db_path) as conn:
            conn.row_factory = sqlite3.Row  # Enable column access by name
            cursor = conn.cursor()

            # Base query to fetch posts by the user
            query = """
                SELECT 
                    id,
                    user_id,
                    content,
                    score,
                    belief,
                    type,
                    created_at
                FROM posts
                WHERE user_id = ?
            """
            params = [user_id]

            # Add date range filter if end_date is provided
            if end_date:
                query += " AND created_at <= ?"
                params.append(end_date.strftime('%Y-%m-%d %H:%M:%S'))

            # Execute the query
            cursor.execute(query, params)
            rows = cursor.fetchall()

            # Convert rows to a list of dictionaries
            posts = [dict(row) for row in rows]

            return posts

    except sqlite3.Error as e:
        print(f"Error getting user posts: {e}")
        return []


@lru_cache(maxsize=128)
def compute_pagerank(graph: nx.Graph) -> Dict:
    """Cache PageRank calculations as they are computationally expensive."""
    return nx.pagerank(graph, weight="weight")


def get_cached_user_posts(
    user_id: str,
    db_path: str,
    start_date: pd.Timestamp,
    end_date: pd.Timestamp
) -> List[Dict]:
    """Cache user posts with TTL."""
    return get_user_posts_db(
        user_id=user_id,
        db_path=db_path,
        start_date=start_date,
        end_date=end_date
    )


def recommend_posts(
    graph: nx.Graph,
    target_user_id: str,
    db_path: str,
    start_date: pd.Timestamp,
    end_date: pd.Timestamp,
    max_return: int = 3
) -> List[Dict]:
    """
    Recommend posts for a target user using the provided hot score formula.
    """
    try:
        with sqlite3.connect(db_path) as conn:
            conn.row_factory = sqlite3.Row  # Return results as dictionaries

            # Step 1: Retrieve posts from all users except target user
            query = '''
                SELECT p.id, p.user_id, p.content, p.score, p.belief, p.type, p.created_at
                FROM posts p
                WHERE p.user_id != ? AND p.created_at BETWEEN ? AND ?
            '''
            params = (target_user_id, start_date.strftime('%Y-%m-%d'), end_date.strftime('%Y-%m-%d'))
            cursor = conn.execute(query, params)
            all_posts = cursor.fetchall()

            # Step 2: For each post, calculate the number of upvotes and downvotes
            ranked_posts = []
            t0 = 1134028003  # Constant from the formula

            for post in all_posts:
                # Step 2.1: Count upvotes and downvotes for this post
                post_id = post["id"]
                reactions_query = '''
                    SELECT score, COUNT(*) AS count
                    FROM reactions
                    WHERE post_id = ?
                    GROUP BY score
                '''
                reactions_cursor = conn.execute(reactions_query, (post_id,))
                reactions = reactions_cursor.fetchall()

                # Default counts for upvotes and downvotes
                upvotes = 0
                downvotes = 0

                # Fill upvote and downvote counts based on the reactions
                for reaction in reactions:
                    if reaction["score"] == 1:
                        upvotes = reaction["count"]
                    elif reaction["score"] == -1:
                        downvotes = reaction["count"]

                # Step 2.2: Calculate the hot score
                u = upvotes  # Number of upvotes
                d = downvotes  # Number of downvotes
                t = pd.to_datetime(post["created_at"]).timestamp()  # Submission time in seconds

                # Calculate the hot score
                h = math.log10(max(abs(u - d), 1)) + math.copysign(1, u - d) * (t - t0) / 45000
                ranked_posts.append((h, dict(post)))  # Convert sqlite3.Row to dict

            # Step 3: Sort posts by hot score in descending order
            ranked_posts.sort(key=lambda x: x[0], reverse=True)

            # Step 4: Return the top posts
            return [post for _, post in ranked_posts[:max_return]]

    except sqlite3.Error as e:
        print(f"Error recommending posts: {e}")
        return []


def recommend_post_graph(
    graph: nx.Graph,
    target_user_id: str,
    db_path: str,
    start_date: pd.Timestamp,
    end_date: pd.Timestamp,
    max_return: int = 3
) -> List[Dict]:
    """
    Recommend posts from users who are directly connected to the target user in the graph.
    Improved algorithm with time decay and logarithmic scaling of interactions.
    """
    try:
        with sqlite3.connect(db_path) as conn:
            conn.row_factory = sqlite3.Row  # Return results as dictionaries

            # Step 1: Get all users that are neighbors of the target user in the graph
            neighbors = list(graph.neighbors(target_user_id))

            if not neighbors:
                return []

            # Step 2: Retrieve posts from these neighboring users
            query = '''
                SELECT p.id, p.user_id, p.content, p.score, p.belief, p.type, p.created_at
                FROM posts p
                WHERE p.user_id IN ({}) AND p.created_at BETWEEN ? AND ?
            '''.format(','.join(['?'] * len(neighbors)))  # Create a parameterized IN clause
            params = neighbors + [start_date.strftime('%Y-%m-%d'), end_date.strftime('%Y-%m-%d')]
            cursor = conn.execute(query, params)
            all_posts = cursor.fetchall()

            # Step 3: For each post, calculate the number of upvotes and downvotes
            ranked_posts = []
            current_time = pd.Timestamp.now().timestamp()  # Get current timestamp

            for post in all_posts:
                # Step 3.1: Count upvotes and downvotes for this post
                post_id = post["id"]
                reactions_query = '''
                    SELECT type, COUNT(*) AS count
                    FROM reactions
                    WHERE post_id = ?
                    GROUP BY type
                '''
                reactions_cursor = conn.execute(reactions_query, (post_id,))
                reactions = reactions_cursor.fetchall()

                # Default counts for upvotes and downvotes
                upvotes = 0
                downvotes = 0

                # Fill upvote and downvote counts based on the reactions
                for reaction in reactions:
                    if reaction["type"] == 'like':
                        upvotes = reaction["count"]
                    elif reaction["type"] == 'unlike':
                        downvotes = reaction["count"]

                # Step 3.2: Calculate the hot score
                net_votes = upvotes - downvotes
                post_time = pd.to_datetime(post["created_at"]).timestamp()
                days_elapsed = max((current_time - post_time) / 86400, 0.1)  # Prevent division by zero

                if net_votes > 0:
                    score = math.log10(net_votes + 1)  # Logarithmic scaling of net votes
                    time_decay = (days_elapsed + 1) ** 1.8  # Time decay factor
                    hot_score = score / time_decay  # Final hot score
                else:
                    hot_score = -days_elapsed  # Negative content sinks quickly

                ranked_posts.append((hot_score, dict(post)))  # Convert sqlite3.Row to dict

            # Step 4: Sort posts by hot score in descending order
            ranked_posts.sort(key=lambda x: x[0], reverse=True)

            # Step 5: Return the top posts
            return [post for _, post in ranked_posts[:max_return]]

    except sqlite3.Error as e:
        print(f"Error recommending posts: {e}")
        return []


def get_user_reactions_db(
    user_id: str,
    reaction_type: Optional[str] = None,
    start_date: Optional[pd.Timestamp] = None,
    end_date: Optional[pd.Timestamp] = None,
    db_path: str = FORUM_DB_PATH
) -> List[Dict]:
    """
    Get all reactions by a user within date range

    Args:
        user_id (str): User ID to get reactions for
        reaction_type (Optional[str]): Filter by 'like' or 'unlike'
        start_date (Optional[pd.Timestamp]): Start of date range
        end_date (Optional[pd.Timestamp]): End of date range
        db_path (str): Database path

    Returns:
        list[dict]: List of reaction dictionaries with keys:
            - post_id: ID of post reacted to
            - type: reaction type (like/unlike)
            - created_at: reaction timestamp
            - post_content: content of post reacted to
    """
    try:
        with sqlite3.connect(db_path) as conn:
            conn.row_factory = sqlite3.Row

            query = """
                SELECT r.*, p.content as post_content
                FROM reactions r
                JOIN posts p ON r.post_id = p.id
                WHERE r.user_id = ?
            """
            params = [user_id]

            if reaction_type:
                query += " AND r.score = ?"
                params.append(1 if reaction_type == "like" else -1)
            if start_date:
                query += " AND r.created_at >= ?"
                params.append(start_date.strftime('%Y-%m-%d'))
            if end_date:
                query += " AND r.created_at <= ?"
                params.append(end_date.strftime('%Y-%m-%d'))

            query += " ORDER BY r.created_at DESC"

            cursor = conn.execute(query, params)
            rows = cursor.fetchall()

            return [dict(row) for row in rows]

    except sqlite3.Error as e:
        print(f"Error getting user reactions: {e}")
        return []


def get_post_by_id_db(
    post_id: int,
    db_path: str = FORUM_DB_PATH
) -> Optional[Dict]:
    """
    Get detailed information about a specific post

    Args:
        post_id (int): ID of the post to retrieve
        db_path (str): Database path

    Returns:
        Optional[dict]: Post information including:
            - id: post ID
            - user_id: poster's user ID
            - content: post content
            - created_at: post timestamp
            - score: current post score
            - likes_count: number of positive reactions
            - unlikes_count: number of negative reactions
            - reference_id: ID of referenced post (if repost)
            Or None if post not found
    """
    try:
        with sqlite3.connect(db_path) as conn:
            conn.row_factory = sqlite3.Row

            query = """
                SELECT 
                    p.*,
                    COUNT(CASE WHEN r.score = 1 THEN 1 END) as likes_count,
                    COUNT(CASE WHEN r.score = -1 THEN 1 END) as unlikes_count,
                    pr.reference_id
                FROM posts p
                LEFT JOIN reactions r ON p.id = r.post_id
                LEFT JOIN post_references pr ON p.id = pr.id
                WHERE p.id = ?
                GROUP BY p.id
            """

            cursor = conn.execute(query, (post_id,))
            row = cursor.fetchone()

            if row is None:
                return None

            return dict(row)

    except sqlite3.Error as e:
        print(f"Error getting post {post_id}: {e}")
        return None


def get_post_count_by_date_range_db(
    start_date: str,
    end_date: str,
    db_path: str = FORUM_DB_PATH
) -> Tuple[int, int]:
    """
    Get total post count and unique user count within a date range.

    Args:
        start_date (str): Start date in 'YYYY-MM-DD' format.
        end_date (str): End date in 'YYYY-MM-DD' format.
        db_path (str): Path to the database. Defaults to FORUM_DB_PATH.

    Returns:
        Tuple[int, int]: 
            First int: Total post count in date range
            Second int: Total unique user count in date range
    """
    try:
        with sqlite3.connect(db_path) as conn:
            query = """
                SELECT 
                    COUNT(*) as total_posts,
                    COUNT(DISTINCT user_id) as unique_users
                FROM posts
                WHERE created_at >= ? AND created_at <= ?
            """

            cursor = conn.execute(query, (start_date, end_date))
            result = cursor.fetchone()

            total_posts = result[0] if result else 0
            unique_users = result[1] if result else 0

            return total_posts, unique_users

    except sqlite3.Error as e:
        print(f"Error getting stats: {e}")
        return 0, 0


def find_root_post(post_id, db_path):
    """
    根据 post_id 和数据库路径，找到该帖子的根帖子。
    首先从 post_references 表中查找引用路径，然后从 posts 表中查找根帖子。

    Args:
        post_id (int): 当前帖子的 ID。
        db_path (str): 数据库路径。

    Returns:
        dict: 根帖子的详细信息，包括 post_id, user_id, created_at, score, content。
              如果找不到根帖子，则返回 None。
    """
    with sqlite3.connect(db_path) as conn:
        cursor = conn.cursor()

        # 初始化路径追踪
        current_post_id = post_id
        path = []

        while True:
            # 查找当前帖子的父帖子 ID
            cursor.execute('''
                SELECT reference_id
                FROM post_references
                WHERE repost_id = ?
            ''', (current_post_id,))
            row = cursor.fetchone()

            if not row:
                break  # 如果没有找到父帖子，则停止查找

            path.append(current_post_id)  # 记录路径
            current_post_id = row[0]  # 更新为父帖子 ID

        # 最后一个有效的帖子 ID 应该是根帖子的候选
        root_candidate_id = current_post_id

        # 在 posts 表中查询根帖子的详细信息
        cursor.execute('''
            SELECT
                p.id,
                p.user_id,
                p.created_at,
                p.type,
                p.score,
                p.content
            FROM posts p
            WHERE p.id = ?
        ''', (root_candidate_id,))
        root_row = cursor.fetchone()

        if not root_row or root_row[3] == 'repost':
            return None  # 如果根帖子不存在或类型为 repost，则返回 None

        return {
            "post_id": root_row[0],
            "user_id": root_row[1],
            "created_at": root_row[2],
            "score": root_row[4],
            "content": root_row[5],
            "path": path  # 返回完整路径
        }

# init_db_forum('./TwinMarket/data/ForumDB/sys_1000.db')
