import inspect
import random
import warnings
from collections import defaultdict
from contextlib import nullcontext
from functools import wraps
from typing import Any, Callable, Dict, List, Literal, Optional, Tuple, Union
from copy import deepcopy

import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
from accelerate import PartialState
from accelerate.utils import is_deepspeed_available
from datasets import Dataset
from torch.utils.data import DataLoader
from transformers import AutoModelForCausalLM, DataCollator, PreTrainedModel, PreTrainedTokenizerBase, Trainer
from transformers import AutoModel
from transformers.trainer_callback import TrainerCallback
from transformers.trainer_utils import EvalLoopOutput
from transformers.utils import is_torch_fx_proxy

from trl.import_utils import is_peft_available, is_wandb_available
from config import OurConfig

from dataclasses import dataclass
from typing import Dict, Literal, Optional

from transformers import TrainingArguments

from trl.trainer.utils import (
    DPODataCollatorWithPadding,
    disable_dropout_in_model,
    pad_to_length,
    peft_module_casting_to_bf16,
    trl_sanitze_kwargs_for_tagging,
)
from trl.models import PreTrainedModelWrapper

from utils import RMAlignmentDataCollatorWithPadding

if is_peft_available():
    from peft import PeftModel, get_peft_model, prepare_model_for_kbit_training

if is_wandb_available():
    import wandb

if is_deepspeed_available():
    import deepspeed


class OurTrainer(Trainer):
    r"""
    Initialize SimPOTrainer.

    Args:
        model (`transformers.PreTrainedModel`):
            The model to train, preferably an `AutoModelForSequenceClassification`.
        args (`OurConfig`):
            The SimPO config arguments to use for training.
        data_collator (`transformers.DataCollator`):
            The data collator to use for training. If None is specified, the default data collator (`DPODataCollatorWithPadding`) will be used
            which will pad the sequences to the maximum length of the sequences in the batch, given a dataset of paired sequences.
        train_dataset (`datasets.Dataset`):
            The dataset to use for training.
        eval_dataset (`datasets.Dataset`):
            The dataset to use for evaluation.
        tokenizer (`transformers.PreTrainedTokenizerBase`):
            The tokenizer to use for training. This argument is required if you want to use the default data collator.
        model_init (`Callable[[], transformers.PreTrainedModel]`):
            The model initializer to use for training. If None is specified, the default model initializer will be used.
        callbacks (`List[transformers.TrainerCallback]`):
            The callbacks to use for training.
        optimizers (`Tuple[torch.optim.Optimizer, torch.optim.lr_scheduler.LambdaLR]`):
            The optimizer and scheduler to use for training.
        preprocess_logits_for_metrics (`Callable[[torch.Tensor, torch.Tensor], torch.Tensor]`):
            The function to use to preprocess the logits before computing the metrics.
        peft_config (`Dict`, defaults to `None`):
            The PEFT configuration to use for training. If you pass a PEFT configuration, the model will be wrapped in a PEFT model.
        compute_metrics (`Callable[[EvalPrediction], Dict]`, *optional*):
            The function to use to compute the metrics. Must take a `EvalPrediction` and return
            a dictionary string to metric values.
    """

    _tag_names = ["trl", "simpo"]

    def __init__(
        self,
        model: Optional[Union[PreTrainedModel, nn.Module, str]] = None,
        ref_model: Optional[Union[PreTrainedModel, nn.Module, str]] = None,
        args: Optional[OurConfig] = None,
        data_collator: Optional[DataCollator] = None,
        train_dataset: Optional[Dataset] = None,
        eval_dataset: Optional[Union[Dataset, Dict[str, Dataset]]] = None,
        tokenizer: Optional[PreTrainedTokenizerBase] = None,
        model_init: Optional[Callable[[], PreTrainedModel]] = None,
        callbacks: Optional[List[TrainerCallback]] = None,
        optimizers: Tuple[torch.optim.Optimizer, torch.optim.lr_scheduler.LambdaLR] = (None, None),
        preprocess_logits_for_metrics: Optional[Callable[[torch.Tensor, torch.Tensor], torch.Tensor]] = None,
        peft_config: Optional[Dict] = None,
        compute_metrics: Optional[Callable[[EvalLoopOutput], Dict]] = None,
    ):
        
        self.model_type = args.model_type
        self.use_ref_model = args.use_ref_model
        
        if self.model_type == "diffusion":
            self.mask_id = 126336
        
        if args.model_init_kwargs is None:
            model_init_kwargs = {}
        elif not isinstance(model, str):
            raise ValueError("You passed model_init_kwargs to OurTrainer. But your model is already instantiated.")
        else:
            model_init_kwargs = args.model_init_kwargs
            model_init_kwargs["torch_dtype"] = (
                model_init_kwargs["torch_dtype"]
                if model_init_kwargs["torch_dtype"] in ["auto", None]
                else getattr(torch, model_init_kwargs["torch_dtype"])
            )

        if isinstance(model, str):
            if self.model_type == "ar":
                model = AutoModelForCausalLM.from_pretrained(model, **model_init_kwargs)
            else:
                model = AutoModel.from_pretrained(model, **model_init_kwargs)

        # Initialize this variable to False. This helps tracking the case when `peft_module_casting_to_bf16`
        # has been called in order to properly call autocast if needed.
        self._peft_has_been_casted_to_bf16 = False

        if not is_peft_available() and peft_config is not None:
            raise ValueError(
                "PEFT is not installed and you passed a `peft_config` in the trainer's kwargs, please install it to use the PEFT models"
            )
        elif is_peft_available() and peft_config is not None:
            # if model is a peft model and we have a peft_config, we merge and unload it first
            if isinstance(model, PeftModel):
                model = model.merge_and_unload()

            if getattr(model, "is_loaded_in_8bit", False) or getattr(model, "is_loaded_in_4bit", False):
                _support_gc_kwargs = hasattr(
                    args, "gradient_checkpointing_kwargs"
                ) and "gradient_checkpointing_kwargs" in list(
                    inspect.signature(prepare_model_for_kbit_training).parameters
                )

                prepare_model_kwargs = {"use_gradient_checkpointing": args.gradient_checkpointing}

                if _support_gc_kwargs:
                    prepare_model_kwargs["gradient_checkpointing_kwargs"] = args.gradient_checkpointing_kwargs

                model = prepare_model_for_kbit_training(model, **prepare_model_kwargs)
            elif getattr(args, "gradient_checkpointing", False):
                # For backward compatibility with older versions of transformers
                if hasattr(model, "enable_input_require_grads"):
                    model.enable_input_require_grads()
                else:

                    def make_inputs_require_grad(module, input, output):
                        output.requires_grad_(True)

                    model.get_input_embeddings().register_forward_hook(make_inputs_require_grad)

            # get peft model with the given config
            model = get_peft_model(model, peft_config)
            if args.bf16 and getattr(model, "is_loaded_in_4bit", False):
                peft_module_casting_to_bf16(model)
                # If args.bf16 we need to explicitly call `generate` with torch amp autocast context manager
                self._peft_has_been_casted_to_bf16 = True

        # For models that use gradient_checkpointing, we need to attach a hook that enables input
        # to explicitly have `requires_grad=True`, otherwise training will either silently
        # fail or completely fail.
        elif getattr(args, "gradient_checkpointing", False):
            # For backward compatibility with older versions of transformers
            if hasattr(model, "enable_input_require_grads"):
                model.enable_input_require_grads()
            else:

                def make_inputs_require_grad(module, input, output):
                    output.requires_grad_(True)

                model.get_input_embeddings().register_forward_hook(make_inputs_require_grad)
        elif getattr(args, "activation_checkpointing", False):
            model.model.set_activation_checkpointing(**args.activation_checkpointing_kwargs)
            print(
                f"Using activation checkpointing with {args.activation_checkpointing_kwargs}")

        if args.generate_during_eval and not is_wandb_available():
            raise ValueError(
                "`generate_during_eval=True` requires Weights and Biases to be installed."
                " Please install `wandb` to resolve."
            )

        if model is not None:
            self.is_encoder_decoder = model.config.is_encoder_decoder
        elif args.is_encoder_decoder is None:
            raise ValueError("When no model is provided, you need to pass the parameter is_encoder_decoder.")
        else:
            self.is_encoder_decoder = args.is_encoder_decoder

        if self.is_encoder_decoder:
            self.decoder_start_token_id = model.config.decoder_start_token_id
            self.pad_token_id = model.config.pad_token_id

        if tokenizer is None:
            raise ValueError("tokenizer must be specified to tokenize a SimPO dataset.")
        if args.max_length is None:
            warnings.warn(
                "`max_length` is not set in the OurConfig's init"
                " it will default to `512` by default, but you should do it yourself in the future.",
                UserWarning,
            )
            max_length = 512
        else:
            max_length = args.max_length
        if args.max_prompt_length is None:
            warnings.warn(
                "`max_prompt_length` is not set in the OurConfig's init"
                " it will default to `128` by default, but you should do it yourself in the future.",
                UserWarning,
            )
            max_prompt_length = 128
        else:
            max_prompt_length = args.max_prompt_length

        if args.max_target_length is None and self.is_encoder_decoder:
            warnings.warn(
                "When using an encoder decoder architecture, you should set `max_target_length` in the OurConfig's init"
                " it will default to `128` by default, but you should do it yourself in the future.",
                UserWarning,
            )
            max_target_length = 128
        else:
            max_target_length = args.max_target_length

        if data_collator is None:
            # data_collator = DPODataCollatorWithPadding(
            #     pad_token_id=tokenizer.pad_token_id,
            #     label_pad_token_id=args.label_pad_token_id,
            #     is_encoder_decoder=self.is_encoder_decoder,
            # )
            data_collator = RMAlignmentDataCollatorWithPadding(
                pad_token_id=tokenizer.pad_token_id,
                label_pad_token_id=args.label_pad_token_id,
                is_encoder_decoder=self.is_encoder_decoder,
            )

            if args.remove_unused_columns:
                args.remove_unused_columns = False
                # warn users
                warnings.warn(
                    "When using DPODataCollatorWithPadding, you should set `remove_unused_columns=False` in your TrainingArguments"
                    " we have set it for you, but you should do it yourself in the future.",
                    UserWarning,
                )

            self.use_dpo_data_collator = True
        else:
            self.use_dpo_data_collator = False

        if args.disable_dropout:
            disable_dropout_in_model(model)

        self.max_length = max_length
        self.generate_during_eval = args.generate_during_eval
        self.label_pad_token_id = args.label_pad_token_id
        self.padding_value = args.padding_value if args.padding_value is not None else tokenizer.pad_token_id
        self.max_prompt_length = max_prompt_length
        self.truncation_mode = args.truncation_mode
        self.max_target_length = max_target_length
        self.tokenizer = tokenizer

        self.beta = args.beta
        self.gamma_beta_ratio = args.gamma_beta_ratio
        self.sft_weight = args.sft_weight
        self.label_smoothing = args.label_smoothing
        self.loss_type = args.loss_type
        self.generator = args.generator
        self.num_responses_to_keep = args.num_responses_to_keep

        self._stored_metrics = defaultdict(lambda: defaultdict(list))

        # Compute that only on the main process for faster data processing.
        # see: https://github.com/huggingface/trl/pull/1255
        args.dataset_num_proc = 8
        with PartialState().local_main_process_first():
            # tokenize the dataset
            train_dataset = train_dataset.map(
                self.tokenize_row,
                num_proc=args.dataset_num_proc,
                desc="Tokenizing train dataset",
                load_from_cache_file=False,
            )
            if eval_dataset is not None:
                eval_dataset = eval_dataset.map(
                    self.tokenize_row,
                    num_proc=args.dataset_num_proc,
                    desc="Tokenizing eval dataset",
                    load_from_cache_file=False,
                )

        super().__init__(
            model=model,
            args=args,
            data_collator=data_collator,
            train_dataset=train_dataset,
            eval_dataset=eval_dataset,
            tokenizer=tokenizer,
            model_init=model_init,
            compute_metrics=compute_metrics,
            callbacks=callbacks,
            optimizers=optimizers,
            preprocess_logits_for_metrics=preprocess_logits_for_metrics,
        )

        # Add tags for models that have been loaded with the correct transformers version
        if hasattr(self.model, "add_model_tags"):
            self.model.add_model_tags(self._tag_names)

        if not hasattr(self, "accelerator"):
            raise AttributeError(
                "Your `Trainer` does not have an `accelerator` object. Consider upgrading `transformers`."
            )
            
        if args.ref_model_init_kwargs is None:
            ref_model_init_kwargs = model_init_kwargs

        if isinstance(ref_model, str) and self.use_ref_model:
            if self.model_type == "ar":
                ref_model = AutoModelForCausalLM.from_pretrained(ref_model, **ref_model_init_kwargs)
            else:
                ref_model = AutoModel.from_pretrained(ref_model, **ref_model_init_kwargs)
            
            self.ref_model = ref_model
            
            if args.disable_dropout:
                if self.ref_model is not None:
                    disable_dropout_in_model(self.ref_model)
                
            if self.is_deepspeed_enabled:
                self.ref_model = self._prepare_deepspeed(self.ref_model)
            else:
                self.ref_model = self.accelerator.prepare_model(self.ref_model, evaluation_mode=True)

    def build_tokenized_answer(self, prompt, answer):
        """
        Llama tokenizer does satisfy `enc(a + b) = enc(a) + enc(b)`.
        It does ensure `enc(a + b) = enc(a) + enc(a + b)[len(enc(a)):]`.
        Reference:
            https://github.com/EleutherAI/lm-evaluation-harness/pull/531#issuecomment-1595586257
        """

        full_tokenized = self.tokenizer(prompt + answer, add_special_tokens=False)
        prompt_input_ids = self.tokenizer(prompt, add_special_tokens=False)["input_ids"]

        answer_input_ids = full_tokenized["input_ids"][len(prompt_input_ids) :]
        answer_attention_mask = full_tokenized["attention_mask"][len(prompt_input_ids) :]

        # Concat tokens to form `enc(a) + enc(a + b)[len(enc(a)):]`
        full_concat_input_ids = np.concatenate([prompt_input_ids, answer_input_ids])

        # Prepare input tokens for token by token comparison
        full_input_ids = np.array(full_tokenized["input_ids"])

        if len(full_input_ids) != len(full_concat_input_ids):
            raise ValueError("Prompt input ids and answer input ids should have the same length.")

        # On some tokenizers, like Llama-2 tokenizer, there are occasions where tokens
        # can be merged together when tokenizing prompt+answer. This could result
        # on the last token from the prompt being different when tokenized on its own
        # vs when done as prompt+answer.
        response_token_ids_start_idx = len(prompt_input_ids)

        # If tokenized prompt is different than both prompt+answer, then it means the
        # last token has changed due to merging.
        if prompt_input_ids != full_tokenized["input_ids"][:response_token_ids_start_idx]:
            response_token_ids_start_idx -= 1

        prompt_input_ids = full_tokenized["input_ids"][:response_token_ids_start_idx]
        prompt_attention_mask = full_tokenized["attention_mask"][:response_token_ids_start_idx]

        if len(prompt_input_ids) != len(prompt_attention_mask):
            raise ValueError("Prompt input ids and attention mask should have the same length.")

        answer_input_ids = full_tokenized["input_ids"][response_token_ids_start_idx:]
        answer_attention_mask = full_tokenized["attention_mask"][response_token_ids_start_idx:]

        return dict(
            prompt_input_ids=prompt_input_ids,
            prompt_attention_mask=prompt_attention_mask,
            input_ids=answer_input_ids,
            attention_mask=answer_attention_mask,
        )

    def tokenize_row(self, feature, model: Optional[Union[PreTrainedModel, nn.Module]] = None) -> Dict:
        """Tokenize a single row from a SimPO specific dataset.

        At this stage, we don't convert to PyTorch tensors yet; we just handle the truncation
        in case the prompt + chosen or prompt + rejected responses is/are too long. First
            we truncate the prompt; if we're still too long, we truncate the chosen/rejected.

        We also create the labels for the chosen/rejected responses, which are of length equal to
            the sum of the length of the prompt and the chosen/rejected response, with
            label_pad_token_id  for the prompt tokens.
        """
        batch = {}
        prompt = feature["prompt"]
        responses = feature["responses"]
        rm_scores = feature["rm_scores"]
        assert not self.is_encoder_decoder
        
        # here to reduce number of response
        num_responses_to_keep = self.num_responses_to_keep

        # if num_responses is one, select the response with highest rm_scores
        if num_responses_to_keep == 1:
            chosen_idx = np.argmax(rm_scores)
            responses = [responses[chosen_idx]]
            rm_scores = [rm_scores[chosen_idx]]
        else:
            # if num_responses is more than one
            # first select the best one and the worst one
            chosen_idx = np.argmax(rm_scores)
            rejected_idx = np.argmin(rm_scores)
            chosen_response = responses[chosen_idx]
            rejected_response = responses[rejected_idx]
            chosen_rm_score = rm_scores[chosen_idx]
            rejected_rm_score = rm_scores[rejected_idx]
            cur_responses = [
                chosen_response,
                rejected_response,
            ]
            cur_rm_scores = [
                chosen_rm_score,
                rejected_rm_score,
            ]
            rest_idx = [
                i for i in range(len(responses)) if i != chosen_idx and i != rejected_idx
            ]
            # select the rest of the responses
            num_rest_to_select = num_responses_to_keep - 2
            if num_rest_to_select > 0:
                rest_idx = random.sample(rest_idx, num_rest_to_select)
                for idx in rest_idx:
                    cur_responses.append(responses[idx])
                    cur_rm_scores.append(rm_scores[idx])
            responses = cur_responses
            rm_scores = cur_rm_scores
        assert len(responses) == num_responses_to_keep
        assert len(rm_scores) == num_responses_to_keep

        # responses = responses[:num_responses_to_keep]
        # rm_scores = rm_scores[:num_responses_to_keep]

        # Check issues below for more details
        #  1. https://github.com/huggingface/trl/issues/907
        #  2. https://github.com/EleutherAI/lm-evaluation-harness/pull/531#issuecomment-1595586257
        #  3. https://github.com/LianjiaTech/BELLE/issues/337

        if not isinstance(prompt, str):
            raise ValueError(f"prompt should be an str but got {type(prompt)}")
        prompt_tokens = self.tokenizer(prompt, add_special_tokens=False)
        prompt_tokens = {f"prompt_{k}": v for k, v in prompt_tokens.items()}
        
        response_tokens = []
        for response in responses:
            if not isinstance(response, str):
                raise ValueError(f"responses should be an str but got {type(response)}")
            cur_response_tokens = self.build_tokenized_answer(prompt, response)
            response_tokens.append(cur_response_tokens)

        # Last prompt token might get merged by tokenizer and
        # it should not be included for generation if that happens
        prompt_len_input_ids = len(prompt_tokens["prompt_input_ids"])
        
        prompt_len_input_ids = min(
            [len(cur_response_tokens["prompt_input_ids"]) for cur_response_tokens in response_tokens]
        )

        for k, v in prompt_tokens.items():
            prompt_tokens[k] = v[:prompt_len_input_ids]

        # add BOS token to head of prompt. Avoid adding if it's already there
        bos_token_id = self.tokenizer.bos_token_id
        if prompt_len_input_ids == 0 or bos_token_id != prompt_tokens["prompt_input_ids"][0]:
            prompt_tokens["prompt_input_ids"] = [bos_token_id] + prompt_tokens["prompt_input_ids"]
            prompt_tokens["prompt_attention_mask"] = [1] + prompt_tokens["prompt_attention_mask"]
        for idx, cur_response_tokens in enumerate(response_tokens):
            if len(cur_response_tokens["prompt_input_ids"]) == 0 or bos_token_id != cur_response_tokens["prompt_input_ids"][0]:
                cur_response_tokens["prompt_input_ids"] = [bos_token_id] + cur_response_tokens["prompt_input_ids"]
                cur_response_tokens["prompt_attention_mask"] = [1] + cur_response_tokens["prompt_attention_mask"]
                # response_tokens[idx] = cur_response_tokens
                
        # add EOS token to end of answer. Avoid adding if it's already there
        eos_token_id = self.tokenizer.eos_token_id
        for idx, cur_response_tokens in enumerate(response_tokens):
            if len(cur_response_tokens["input_ids"]) == 0 or eos_token_id != cur_response_tokens["input_ids"][-1]:
                cur_response_tokens["input_ids"].append(eos_token_id)
                cur_response_tokens["attention_mask"].append(1)
                # response_tokens[idx] = cur_response_tokens
        
        longer_response_length = max(
            [len(cur_response_tokens["input_ids"]) for cur_response_tokens in response_tokens]
        )
        
        # if combined sequence is too long, truncate the prompt
        for answer_tokens in [prompt_tokens] + response_tokens:
            if len(answer_tokens["prompt_input_ids"]) + longer_response_length > self.max_length:
                if self.truncation_mode == "keep_start":
                    for k in ["prompt_input_ids", "prompt_attention_mask"]:
                        answer_tokens[k] = answer_tokens[k][: self.max_prompt_length]
                elif self.truncation_mode == "keep_end":
                    for k in ["prompt_input_ids", "prompt_attention_mask"]:
                        answer_tokens[k] = answer_tokens[k][-self.max_prompt_length :]
                else:
                    raise ValueError(f"Unknown truncation mode: {self.truncation_mode}")

        # if that's still too long, truncate the response
        for answer_tokens in response_tokens:
            if len(answer_tokens["prompt_input_ids"]) + longer_response_length > self.max_length:
                for k in ["input_ids", "attention_mask"]:
                    answer_tokens[k] = answer_tokens[k][: self.max_length - self.max_prompt_length]

        # Create labels
        sequence_tokens = []
        for cur_response_tokens in response_tokens:
            cur_sequence_tokens = {
                k: cur_response_tokens[f"prompt_{k}"] + cur_response_tokens[k] for k in ["input_ids", "attention_mask"]
            }
            cur_sequence_tokens["labels"] = cur_sequence_tokens["input_ids"][:]
            cur_sequence_tokens["labels"][: len(cur_response_tokens["prompt_input_ids"])] = [
                self.label_pad_token_id
            ] * len(cur_response_tokens["prompt_input_ids"])
            sequence_tokens.append(cur_sequence_tokens)
            
        for idx, sequence_token in enumerate(sequence_tokens):
            k = f"{idx}_"
            toks = sequence_token
            for type_key, tokens in toks.items():
                if type_key == "token_type_ids":
                    continue
                batch[f"{k}{type_key}"] = tokens
            batch[f"{k}rm_scores"] = torch.tensor(rm_scores[idx]).float()
        k = ""
        toks = prompt_tokens
        for type_key, tokens in toks.items():
            if type_key == "token_type_ids":
                continue
            batch[f"{k}{type_key}"] = tokens
        
        batch['num_responses'] = len(sequence_tokens)

        return batch

    @staticmethod
    def concatenated_inputs(
        batch: Dict[str, Union[List, torch.LongTensor]],
        is_encoder_decoder: bool = False,
        label_pad_token_id: int = -100,
        padding_value: int = 0,
        device: Optional[torch.device] = None,
    ) -> Dict[str, torch.LongTensor]:
        """Concatenate the chosen and rejected inputs into a single tensor.

        Args:
            batch: A batch of data. Must contain the keys 'chosen_input_ids' and 'rejected_input_ids', which are tensors of shape (batch_size, sequence_length).
            is_encoder_decoder: Whether the model is an encoder-decoder model.
            label_pad_token_id: The label pad token id.
            padding_value: The padding value to use for the concatenated inputs_ids.
            device: The device for the concatenated inputs.

        Returns:
            A dictionary containing the concatenated inputs under the key 'concatenated_input_ids'.
        """
        assert not is_encoder_decoder
        
        concatenated_batch = {}
        
        num_responses = batch["num_responses"][0]
        max_length = max(
            [batch[f"{i}_input_ids"].shape[1] for i in range(num_responses)]
        )
        concatenated_rm_scores = []
        concatenated_batch_index = []
        # max_length = max(batch["chosen_input_ids"].shape[1], batch["rejected_input_ids"].shape[1])
        for idx in range(num_responses):
            for k in batch:
                if k.startswith(f"{idx}_") and isinstance(batch[k], torch.Tensor):
                    if "labels" in k:
                        pad_value = label_pad_token_id
                    elif k.endswith("_input_ids"):
                        pad_value = padding_value
                    elif k.endswith("_attention_mask"):
                        pad_value = 0
                    concatenated_key = k.replace(f"{idx}_", "concatenated_")
                    if idx == 0:
                        concatenated_batch[concatenated_key] = pad_to_length(
                            batch[k], max_length, pad_value=pad_value
                        )
                    else:
                        concatenated_batch[concatenated_key] = torch.cat(
                            (
                                concatenated_batch[concatenated_key],
                                pad_to_length(batch[k], max_length, pad_value=pad_value),
                            ),
                            dim=0,
                        )
                    if idx == num_responses - 1:
                        concatenated_batch[concatenated_key] = concatenated_batch[concatenated_key].to(
                            device=device
                        )  # [num_responses * batch_size, max_length]
                # rm_scores
                if k.startswith(f"{idx}_") and k.endswith("rm_scores"):
                    concatenated_rm_scores.append(torch.tensor(batch[k]).float())
                    concatenated_batch_index.append(torch.arange(len(batch[k])))
        # concatenate rm_scores
        concatenated_rm_scores = torch.cat(concatenated_rm_scores, dim=0)  # [num_responses * batch_size]
        concatenated_batch["concatenated_rm_scores"] = concatenated_rm_scores.to(device=device)
        # batch index
        concatenated_batch_index = torch.cat(concatenated_batch_index, dim=0)  # [num_responses * batch_size]
        concatenated_batch["concatenated_batch_index"] = concatenated_batch_index.to(device=device)

        return concatenated_batch
    
    def our_loss(
        self,
        logps: torch.FloatTensor,  # [num_responses, batch_size]
        logps_ref: torch.FloatTensor,  # [num_responses, batch_size]
        rm_scores: torch.FloatTensor,  # [num_responses, batch_size]
    ):
        g_theta = - self.beta * (logps - logps_ref)
        if self.generator == 'rkl':
            lsm = torch.log_softmax(rm_scores, dim=0) - torch.log_softmax(g_theta, dim=0)
            losses = (torch.nn.functional.softmax(rm_scores, dim=0) * lsm).sum(dim=0)
        elif self.generator == 'kl':
            lsm = torch.log_softmax(g_theta, dim=0) - torch.log_softmax(rm_scores, dim=0)
            losses = (torch.nn.functional.softmax(g_theta, dim=0) * lsm).sum(dim=0)
        return losses  # [batch_size]

    def simpo_loss(
        self,
        policy_chosen_logps: torch.FloatTensor,
        policy_rejected_logps: torch.FloatTensor,
    ) -> Tuple[torch.FloatTensor, torch.FloatTensor, torch.FloatTensor]:
        """Compute the SimPO loss for a batch of policy model log probabilities.

        Args:
            policy_chosen_logps: Log probabilities of the policy model for the chosen responses. Shape: (batch_size,)
            policy_rejected_logps: Log probabilities of the policy model for the rejected responses. Shape: (batch_size,)

        Returns:
            A tuple of three tensors: (losses, chosen_rewards, rejected_rewards).
            The losses tensor contains the SimPO loss for each example in the batch.
            The chosen_rewards and rejected_rewards tensors contain the rewards for the chosen and rejected responses, respectively.
        """
        
        pi_logratios = policy_chosen_logps - policy_rejected_logps
        # pi_logratios = pi_logratios.to(self.accelerator.device)
        logits = pi_logratios - self.gamma_beta_ratio

        losses = (
            -F.logsigmoid(self.beta * logits) * (1 - self.label_smoothing)
            - F.logsigmoid(-self.beta * logits) * self.label_smoothing
        )
        
        return losses
    
    def our_loss_pairwise(
        self,
        policy_chosen_logps: torch.FloatTensor,
        policy_rejected_logps: torch.FloatTensor,
        policy_chosen_logps_ref: torch.FloatTensor,
        policy_rejected_logps_ref: torch.FloatTensor,
    ) -> Tuple[torch.FloatTensor, torch.FloatTensor, torch.FloatTensor]:
        """Compute the SimPO loss for a batch of policy model log probabilities.

        Args:
            policy_chosen_logps: Log probabilities of the policy model for the chosen responses. Shape: (batch_size,)
            policy_rejected_logps: Log probabilities of the policy model for the rejected responses. Shape: (batch_size,)

        Returns:
            A tuple of three tensors: (losses, chosen_rewards, rejected_rewards).
            The losses tensor contains the SimPO loss for each example in the batch.
            The chosen_rewards and rejected_rewards tensors contain the rewards for the chosen and rejected responses, respectively.
        """
        
        pi_logratios = (policy_chosen_logps - policy_chosen_logps_ref) - (policy_rejected_logps - policy_rejected_logps_ref)
        # pi_logratios = pi_logratios.to(self.accelerator.device)
        logits = pi_logratios

        losses = -F.logsigmoid(self.beta * logits)
        
        return losses

    def concatenated_forward(
        self,
        model: nn.Module,
        batch: Dict[str, Union[List, torch.LongTensor]],
    ) -> Tuple[torch.FloatTensor, torch.FloatTensor, torch.FloatTensor, torch.FloatTensor]:
        """Run the given model on the given batch of inputs, concatenating the chosen and rejected inputs together.

        We do this to avoid doing two forward passes, because it's faster for FSDP.
        """
        assert not self.is_encoder_decoder
        concatenated_batch = self.concatenated_inputs(
            batch,
            is_encoder_decoder=self.is_encoder_decoder,
            label_pad_token_id=self.label_pad_token_id,
            padding_value=self.padding_value,
            device=self.accelerator.device,
        )
        # len_chosen = batch["chosen_labels"].shape[0]
        num_batch = concatenated_batch["concatenated_batch_index"].max().item() + 1
        
        # add mask noise if is diffusion
        if self.model_type == "diffusion":
            # for each set of responses in the batch, sample a mask rate
            mask_rate = torch.rand(num_batch)
            for batch_idx in range(num_batch):
                # get the indices of the responses in the batch
                response_indices = torch.where(concatenated_batch["concatenated_batch_index"] == batch_idx)[0]
                # sample a mask rate for this batch
                mask_rate_for_batch = mask_rate[batch_idx].item()
                # loop over the responses and apply the mask
                for response_idx in response_indices:
                    # apply mask to the positions where its label is not self.label_pad_token_id
                    mask_positions = concatenated_batch["concatenated_labels"][response_idx] != self.label_pad_token_id
                    # apply the mask
                    mask_positions = mask_positions.nonzero(as_tuple=True)[0]
                    num_mask_positions = int(mask_rate_for_batch * len(mask_positions))
                    num_mask_positions = max(num_mask_positions, 1)
                    
                    mask_positions = mask_positions[torch.randperm(len(mask_positions))[:num_mask_positions]]
                    concatenated_batch["concatenated_input_ids"][response_idx][mask_positions] = self.mask_id
                    # concatenated_batch["concatenated_attention_mask"][response_idx][mask_positions] = 0
                    # concatenated_batch["concatenated_labels"][response_idx][mask_positions] = self.label_pad_token_id
            mask_positions = concatenated_batch["concatenated_input_ids"] == self.mask_id
        else:
            mask_positions = None

        model_kwargs = {}

        all_logits = model(
            concatenated_batch["concatenated_input_ids"],
            attention_mask=concatenated_batch["concatenated_attention_mask"],
            use_cache=False,
            **model_kwargs,
        ).logits  # [num_responses * batch_size, max_length, vocab_size]
        
        all_logps = self.get_batch_logps(
            all_logits,
            concatenated_batch["concatenated_labels"],
            average_log_prob=True,
            is_encoder_decoder=self.is_encoder_decoder,
            label_pad_token_id=self.label_pad_token_id,
            model_type=self.model_type,
            mask_positions=mask_positions,
        )  # [num_responses * batch_size]
        
        if self.use_ref_model:
            with torch.no_grad():
                all_logits_ref = self.ref_model(
                    concatenated_batch["concatenated_input_ids"],
                    attention_mask=concatenated_batch["concatenated_attention_mask"],
                    use_cache=False,
                    **model_kwargs,
                ).logits  # [num_responses * batch_size, max_length, vocab_size]
                all_logps_ref = self.get_batch_logps(
                    all_logits_ref,
                    concatenated_batch["concatenated_labels"],
                    average_log_prob=True,
                    is_encoder_decoder=self.is_encoder_decoder,
                    label_pad_token_id=self.label_pad_token_id,
                    model_type=self.model_type,
                    mask_positions=mask_positions,
                )  # [num_responses * batch_size]
            all_logps_ref = all_logps_ref.detach()
        else:
            all_logps_ref = None
        
        return all_logps, all_logps_ref, concatenated_batch
        

    @staticmethod
    def get_batch_logps(
        logits: torch.FloatTensor,
        labels: torch.LongTensor,
        average_log_prob: bool = True,
        label_pad_token_id: int = -100,
        is_encoder_decoder: bool = False,
        model_type: str = "ar",
        mask_positions: torch.Tensor = None,
    ) -> torch.FloatTensor:
        """Compute the log probabilities of the given labels under the given logits.

        Args:
            logits: Logits of the model (unnormalized). Shape: (batch_size, sequence_length, vocab_size)
            labels: Labels for which to compute the log probabilities. Label tokens with a value of label_pad_token_id are ignored. Shape: (batch_size, sequence_length)
            average_log_prob: If True, return the average log probability per (non-masked) token. Otherwise, return the sum of the log probabilities of the (non-masked) tokens.
            label_pad_token_id: The label pad token id.
            is_encoder_decoder: Whether the model is an encoder-decoder model.

        Returns:
            A tensor of shape (batch_size,) containing the average/sum log probabilities of the given labels under the given logits.
        """
        if logits.shape[:-1] != labels.shape:
            raise ValueError("Logits (batch and sequence length dim) and labels must have the same shape.")

        if model_type == 'ar':
            # next token prediction
            labels = labels[:, 1:].clone()
            logits = logits[:, :-1, :]
            loss_mask = labels != label_pad_token_id
        else:
            # for diffusion
            labels = labels[:].clone()
            loss_mask = mask_positions

        # dummy token; we'll ignore the losses on these tokens later
        labels[labels == label_pad_token_id] = 0

        per_token_logps = torch.gather(logits.log_softmax(-1), dim=2, index=labels.unsqueeze(2)).squeeze(2)

        if average_log_prob:
            return (per_token_logps * loss_mask).sum(-1) / loss_mask.sum(-1)
        else:
            return (per_token_logps * loss_mask).sum(-1)

    def get_batch_loss_metrics(
        self,
        model,
        batch: Dict[str, Union[List, torch.LongTensor]],
        train_eval: Literal["train", "eval"] = "train",
    ):
        """Compute the SimPO loss and other metrics for the given batch of inputs for train or test."""
        metrics = {}
        prefix = "eval_" if train_eval == "eval" else ""
        
        logps, logps_ref, concatenated_batch = self.concatenated_forward(model, batch)  # [num_responses * batch_size]
        
        rm_scores = concatenated_batch["concatenated_rm_scores"]  # [num_responses * batch_size]
        batch_index = concatenated_batch["concatenated_batch_index"]  # [num_responses * batch_size]
        num_batch = batch_index.max().item() + 1
        
        logps = logps.view(-1, num_batch)  # [num_responses, batch_size]
        if logps_ref is not None:
            logps_ref = logps_ref.view(-1, num_batch)
        rm_scores = rm_scores.view(-1, num_batch)  # [num_responses, batch_size]
        
        # get the chosen and rejected logps where chosen_idx is the argmax of rm_scores along num_responses
        chosen_idx = rm_scores.argmax(dim=0)
        rejected_idx = rm_scores.argmin(dim=0)
        policy_chosen_logps = logps[chosen_idx, torch.arange(num_batch).to(logps.device)]
        policy_rejected_logps = logps[rejected_idx, torch.arange(num_batch).to(logps.device)]
        if logps_ref is not None:
            policy_chosen_logps_ref = logps_ref[chosen_idx, torch.arange(num_batch).to(logps_ref.device)]
            policy_rejected_logps_ref = logps_ref[rejected_idx, torch.arange(num_batch).to(logps_ref.device)]
        
        if self.loss_type == 'simpo':
            losses = self.simpo_loss(
                policy_chosen_logps,
                policy_rejected_logps,
            )
        elif self.loss_type == 'ours_pairwise':
            losses = self.our_loss_pairwise(
                policy_chosen_logps,
                policy_rejected_logps,
                policy_chosen_logps_ref,
                policy_rejected_logps_ref,
            )
        elif self.loss_type == 'ours':
            losses = self.our_loss(
                logps,
                logps_ref,
                rm_scores,
            )
        else:
            raise NotImplementedError(
                f"loss_type {self.loss_type} not implemented yet."
            )
        
        loss = losses.mean()
        
        chosen_rewards = self.beta * policy_chosen_logps.detach()
        rejected_rewards = self.beta * policy_rejected_logps.detach()

        if self.sft_weight > 0.0:
            raise NotImplementedError(
                "SFT loss is not implemented yet."
                )
        
        reward_accuracies = (chosen_rewards > rejected_rewards).float()

        metrics[f"{prefix}rewards/chosen"] = chosen_rewards.mean().cpu()
        metrics[f"{prefix}rewards/rejected"] = rejected_rewards.mean().cpu()
        metrics[f"{prefix}rewards/accuracies"] = reward_accuracies.mean().cpu()
        metrics[f"{prefix}rewards/margins"] = (chosen_rewards - rejected_rewards).mean().cpu()
        metrics[f"{prefix}logps/rejected"] = policy_rejected_logps.detach().mean().cpu()
        metrics[f"{prefix}logps/chosen"] = policy_chosen_logps.detach().mean().cpu()

        return loss, metrics

    def compute_loss(
        self,
        model: Union[PreTrainedModel, nn.Module],
        inputs: Dict[str, Union[torch.Tensor, Any]],
        return_outputs=False,
    ) -> Union[torch.Tensor, Tuple[torch.Tensor, Dict[str, torch.Tensor]]]:
        if not self.use_dpo_data_collator:
            warnings.warn(
                "compute_loss is only implemented for DPODataCollatorWithPadding, and you passed a datacollator that is different than "
                "DPODataCollatorWithPadding - you might see unexpected behavior. Alternatively, you can implement your own prediction_step method if you are using a custom data collator"
            )

        compute_loss_context_manager = torch.cuda.amp.autocast if self._peft_has_been_casted_to_bf16 else nullcontext

        with compute_loss_context_manager():
            loss, metrics = self.get_batch_loss_metrics(model, inputs, train_eval="train")

        # force log the metrics
        self.store_metrics(metrics, train_eval="train")

        if return_outputs:
            return (loss, metrics)
        return loss

    def get_batch_samples(self, model, batch: Dict[str, torch.LongTensor]) -> Tuple[str, str]:
        """Generate samples from the model and reference model for the given batch of inputs."""

        # If one uses `generate_during_eval` with peft + bf16, we need to explicitly call generate with
        # the torch cuda amp context manager as some hidden states are silently casted to full precision.
        generate_context_manager = nullcontext if not self._peft_has_been_casted_to_bf16 else torch.cuda.amp.autocast

        with generate_context_manager():
            policy_output = model.generate(
                input_ids=batch["prompt_input_ids"],
                attention_mask=batch["prompt_attention_mask"],
                max_length=self.max_length,
                do_sample=True,
                pad_token_id=self.tokenizer.pad_token_id,
            )

        policy_output = pad_to_length(policy_output, self.max_length, self.tokenizer.pad_token_id)
        policy_output_decoded = self.tokenizer.batch_decode(policy_output, skip_special_tokens=True)

        return policy_output_decoded

    def prediction_step(
        self,
        model: Union[PreTrainedModel, nn.Module],
        inputs: Dict[str, Union[torch.Tensor, Any]],
        prediction_loss_only: bool,
        ignore_keys: Optional[List[str]] = None,
    ):
        if not self.use_dpo_data_collator:
            warnings.warn(
                "prediction_step is only implemented for DPODataCollatorWithPadding, and you passed a datacollator that is different than "
                "DPODataCollatorWithPadding - you might see unexpected behavior. Alternatively, you can implement your own prediction_step method if you are using a custom data collator"
            )
        if ignore_keys is None:
            if hasattr(model, "config"):
                ignore_keys = getattr(model.config, "keys_to_ignore_at_inference", [])
            else:
                ignore_keys = []

        prediction_context_manager = torch.cuda.amp.autocast if self._peft_has_been_casted_to_bf16 else nullcontext

        with torch.no_grad(), prediction_context_manager():
            loss, metrics = self.get_batch_loss_metrics(model, inputs, train_eval="eval")

        # force log the metrics
        self.store_metrics(metrics, train_eval="eval")

        if prediction_loss_only:
            return (loss.detach(), None, None)
        
        raise NotImplementedError('logits is not implemented yet.')

    def store_metrics(self, metrics: Dict[str, float], train_eval: Literal["train", "eval"] = "train") -> None:
        for key, value in metrics.items():
            self._stored_metrics[train_eval][key].append(value)

    def evaluation_loop(
        self,
        dataloader: DataLoader,
        description: str,
        prediction_loss_only: Optional[bool] = None,
        ignore_keys: Optional[List[str]] = None,
        metric_key_prefix: str = "eval",
    ) -> EvalLoopOutput:
        """
        Overriding built-in evaluation loop to store metrics for each batch.
        Prediction/evaluation loop, shared by `Trainer.evaluate()` and `Trainer.predict()`.

        Works both with or without labels.
        """

        # Sample and save to game log if requested (for one batch to save time)
        if self.generate_during_eval:
            # Generate random indices within the range of the total number of samples
            num_samples = len(dataloader.dataset)
            random_indices = random.sample(range(num_samples), k=self.args.eval_batch_size)

            # Use dataloader.dataset.select to get the random batch without iterating over the DataLoader
            random_batch_dataset = dataloader.dataset.select(random_indices)
            random_batch = self.data_collator(random_batch_dataset)
            random_batch = self._prepare_inputs(random_batch)

            policy_output_decoded = self.get_batch_samples(self.model, random_batch)

            self.log(
                {
                    "game_log": wandb.Table(
                        columns=["Prompt", "Policy"],
                        rows=[
                            [prompt, pol[len(prompt) :]]
                            for prompt, pol in zip(random_batch["prompt"], policy_output_decoded)
                        ],
                    )
                }
            )
            self.state.log_history.pop()

        # Base evaluation
        initial_output = super().evaluation_loop(
            dataloader, description, prediction_loss_only, ignore_keys, metric_key_prefix
        )

        return initial_output

    def log(self, logs: Dict[str, float]) -> None:
        """
        Log `logs` on the various objects watching training, including stored metrics.

        Args:
            logs (`Dict[str, float]`):
                The values to log.
        """
        # logs either has 'loss' or 'eval_loss'
        train_eval = "train" if "loss" in logs else "eval"
        # Add averaged stored metrics to logs
        for key, metrics in self._stored_metrics[train_eval].items():
            logs[key] = torch.tensor(metrics).mean().item()
        del self._stored_metrics[train_eval]
        return super().log(logs)

    @wraps(Trainer.push_to_hub)
    def push_to_hub(self, commit_message: Optional[str] = "End of training", blocking: bool = True, **kwargs) -> str:
        """
        Overwrite the `push_to_hub` method in order to force-add the tag "simpo" when pushing the
        model on the Hub. Please refer to `~transformers.Trainer.push_to_hub` for more details.
        """
        kwargs = trl_sanitze_kwargs_for_tagging(model=self.model, tag_names=self._tag_names, kwargs=kwargs)

        return super().push_to_hub(commit_message=commit_message, blocking=blocking, **kwargs)

    def _prepare_deepspeed(self, model: PreTrainedModelWrapper):
        # Adapted from accelerate: https://github.com/huggingface/accelerate/blob/739b135f8367becb67ffaada12fe76e3aa60fefd/src/accelerate/accelerator.py#L1473
        deepspeed_plugin = self.accelerator.state.deepspeed_plugin
        config_kwargs = deepcopy(deepspeed_plugin.deepspeed_config)

        if model is not None:
            if hasattr(model, "config"):
                hidden_size = (
                    max(model.config.hidden_sizes)
                    if getattr(model.config, "hidden_sizes", None)
                    else getattr(model.config, "hidden_size", None)
                )
                if hidden_size is not None and config_kwargs["zero_optimization"]["stage"] == 3:
                    # Note that `stage3_prefetch_bucket_size` can produce DeepSpeed messages like: `Invalidate trace cache @ step 0: expected module 1, but got module 0`
                    # This is expected and is not an error, see: https://github.com/microsoft/DeepSpeed/discussions/4081
                    config_kwargs.update(
                        {
                            "zero_optimization.reduce_bucket_size": hidden_size * hidden_size,
                            "zero_optimization.stage3_param_persistence_threshold": 10 * hidden_size,
                            "zero_optimization.stage3_prefetch_bucket_size": 0.9 * hidden_size * hidden_size,
                        }
                    )

        # If ZeRO-3 is used, we shard both the active and reference model.
        # Otherwise, we assume the reference model fits in memory and is initialized on each device with ZeRO disabled (stage 0)
        if config_kwargs["zero_optimization"]["stage"] != 3:
            config_kwargs["zero_optimization"]["stage"] = 0
        model, *_ = deepspeed.initialize(model=model, config=config_kwargs)
        model.eval()
        return model