from __future__ import print_function

import torch
import torch.nn as nn
import torch.nn.functional as F


class SupConLoss(nn.Module):
    """It also supports the unsupervised contrastive loss in SimCLR"""
    def __init__(self, temperature=0.07, contrast_mode='all',
                 base_temperature=0.07, device='cpu'):
        super(SupConLoss, self).__init__()
        self.temperature = temperature
        self.contrast_mode = contrast_mode
        self.base_temperature = base_temperature
        self.device = device

    def forward(self, features, labels=None, mask=None):
        """Args:
            features: hidden vector of shape [bsz, n_views, ...].
            labels: ground truth of shape [bsz].
            mask: contrastive mask of shape [bsz, bsz], mask_{i,j}=1 if sample j
                has the same class as sample i. Can be asymmetric.
        Returns:
            A loss scalar.
        """

        if len(features.shape) < 3:
            raise ValueError('`features` needs to be [bsz, n_views, ...],'
                             'at least 3 dimensions are required')
        if len(features.shape) > 3:
            features = features.view(features.shape[0], features.shape[1], -1)

        batch_size = features.shape[0]
        if labels is not None and mask is not None:
            raise ValueError('Cannot define both `labels` and `mask`')
        elif labels is None and mask is None:
            mask = torch.eye(batch_size, dtype=torch.float32).to(self.device) # e.g. (128, 128) eye matrix
        elif labels is not None:
            labels = labels.contiguous().view(-1, 1)
            if labels.shape[0] != batch_size:
                raise ValueError('Num of labels does not match num of features')
            mask = torch.eq(labels, labels.T).float().to(self.device)
        else:
            mask = mask.float().to(self.device) # e.g.

        contrast_count = features.shape[1] # e.g 2
        contrast_feature = torch.cat(torch.unbind(features, dim=1), dim=0) # e.g. (128,2,20) --> (256, 20): 0-127: full 128-256: masked
        if self.contrast_mode == 'one':
            anchor_feature = features[:, 0]
            anchor_count = 1
        elif self.contrast_mode == 'all':
            anchor_feature = contrast_feature  # e.g. (256, 20)
            anchor_count = contrast_count # e.g. 2
        else:
            raise ValueError('Unknown mode: {}'.format(self.contrast_mode))

        # compute logits
        anchor_dot_contrast = torch.div(
            torch.matmul(anchor_feature, contrast_feature.T), # e.g. (256, 20) x (20, 256) = (256, 256)
            self.temperature)
        # for numerical stability
        logits_max, _ = torch.max(anchor_dot_contrast, dim=1, keepdim=True)
        logits = anchor_dot_contrast - logits_max.detach()

        # tile mask
        mask = mask.repeat(anchor_count, contrast_count) # e.g. (256, 256)
        # mask-out self-contrast cases
        logits_mask = torch.scatter(
            torch.ones_like(mask), # (256, 256)
            1,
            torch.arange(batch_size * anchor_count).view(-1, 1).to(self.device),
            0
        )
        mask = mask * logits_mask

        # compute log_prob
        exp_logits = torch.exp(logits) * logits_mask
        log_prob = logits - torch.log(exp_logits.sum(1, keepdim=True))

        # compute mean of log-likelihood over positive
        mean_log_prob_pos = (mask * log_prob).sum(1) / mask.sum(1)

        # loss
        loss = - (self.temperature / self.base_temperature) * mean_log_prob_pos
        loss = loss.view(anchor_count, batch_size).mean()

        return loss
    

def IRD_distill_loss(features_curr, feature_past, temp_curr=0.2, temp_past=0.01, distill_power=1.0):
    """
    Computes the IRD distillation loss between two models.
    """
    features1_sim = torch.div(torch.matmul(features_curr, features_curr.T), temp_curr)
    logits_mask = torch.scatter(
        torch.ones_like(features1_sim),
        1,
        torch.arange(features1_sim.size(0)).view(-1, 1).cuda(non_blocking=True),
        0
    )
    logits_max1, _ = torch.max(features1_sim * logits_mask, dim=1, keepdim=True)
    features1_sim = features1_sim - logits_max1.detach()
    row_size = features1_sim.size(0)
    logits1 = torch.exp(features1_sim[logits_mask.bool()].view(row_size, -1)) / torch.exp(features1_sim[logits_mask.bool()].view(row_size, -1)).sum(dim=1, keepdim=True)

    with torch.no_grad():
        features2_sim = torch.div(torch.matmul(feature_past, feature_past.T), temp_past)
        logits_max2, _ = torch.max(features2_sim*logits_mask, dim=1, keepdim=True)
        features2_sim = features2_sim - logits_max2.detach()
        logits2 = torch.exp(features2_sim[logits_mask.bool()].view(row_size, -1)) /  torch.exp(features2_sim[logits_mask.bool()].view(row_size, -1)).sum(dim=1, keepdim=True)

    loss_distill = (-logits2 * torch.log(logits1)).sum(1).mean()
    loss = distill_power * loss_distill

    return loss