class Sketcher:
    """
    A Sketcher takes datasets and produces sketches of them, and also answers
    questions about a dataset based on a sketch of the dataset. Subclasses of
    Sketcher implement particular kinds of sketches, e.g. signed one-hot.

    Subclass instances may store several parameters controlling how the sketches
    work, but the differential privacy parameter epsilon is not among them;
    instead, epsilon is passed directly to the sketch_values function.
    """

    def __init__(self, hash_function, rng = None):
        """
        Parameters:
        - hash_function: A function mapping strings to nonnegative integers.
        - rng: An instance of numpy.random.Generator, used to add noise for
          privacy. Only used when sketch_values is called with epsilon that is
          not None.
        """
        self._hash_function = hash_function
        self._rng = rng

    def estimate_membership(self, sketch, identity, value):
        """
        Estimate the number of times (identity, value) appears in the dataset.
        """
        raise NotImplementedError

    def estimate_training_weights(self, sketch, identities, num_categories):
        """
        Returns weights for producing a training set. identities should be a
        sequence of strings, and num_categories should match the parameter of
        the same name that was passed to sketch_values. Returns an matrix of
        shape (len(identities), num_categories). The (i, j) entry is the weight
        that should be assigned to a training example with label j and with the
        features associated with identities[i]. Ideally, the i-th row will be
        0 if the identity isn't present in the dataset the sketch was generated
        from, and otherwise a 1-hot vector encoding the label.
        """
        raise NotImplementedError

    def estimate_training_labels_from_difference_sketch(
            self, difference_sketch, identities):
        """
        Like estimate_training_weights, but takes a difference_sketch (see
        sketch_values) as input, and the output is a vector with the same length
        as identities. Each output element is (-1)**j if the training example
        should have label j (0 or 1), or 0 if the training example should not be
        included.
        """
        raise NotImplementedError

    def sketch_values(self, epsilon, identities, num_categories, values,
                      difference_sketch = False):
        """
        Parameters:
        - epsilon: Differential privacy parameter. A value of None means add no
          noise.
        - identities: An iterable of strings used as the identities to do the
          join on.
        - num_categories: The number of distinct possible values to be
          sketched.
        - values: An iterable parallel to identities, containing the value
          associated with each identity. Elements must be nonnegative integers
          less than num_categories.
        - difference_sketch: If True, and the num_categories parameter to
          __init__ was 2, produces a one-row instead of a 2-row sketch,
          consisting of the difference of the two unnoised rows (first row minus
          second row) plus noise. Must be False if num_categories != 2.
        """
        raise NotImplementedError
