from dataclasses import dataclass
from typing import Optional

from defs import EXPERIMENT_SEEDS, BaseConfigArgs
from lib_llm.models import ModelConfig, get_tokenizer_type
from lib_project.experiment import ExperimentHandle
from utils.memorization import get_memorization_training_config

from .experiment import (
    EXP_ABBREVIATION,
    ExperimentConfig,
    ModificationConfig,
    RandomStringConfig,
    TokenConfig,
    rt_experiment,
)


@dataclass
class ConfigArgs(BaseConfigArgs):
    alphabet_size: int
    string_length: int
    tokens: list[TokenConfig]
    num_epochs: int = 50


FREQUENCY_OCCURRENCE_FACTOR = 5

FREQUENCY = {
    (
        f"{model_id}_t-{string_length}_freq-{num_inserted_tokens}_pre-{prefix_length}_ocf-{FREQUENCY_OCCURRENCE_FACTOR}"
    ): ConfigArgs(
        model_id=model_id,
        alphabet_size=alphabet_size,
        string_length=string_length,
        tokens=[
            TokenConfig(
                prefix_length=prefix_length,
                num_occurrences=FREQUENCY_OCCURRENCE_FACTOR * (i + 1),
                agreement_ratio=1.0,
            )
            for i in range(num_inserted_tokens)
        ],
    )
    for model_id in [
        "pyt-1b",
    ]
    for alphabet_size in [2, 4, 7, 13, 26]
    for string_length in [256, 512, 1024]
    for num_inserted_tokens in [4, 8, 16]
    for prefix_length in [1, 2, 3, 4]
}
PREFIX_LENGTH = {
    (
        f"{model_id}_t-{string_length}_pre-{num_inserted_tokens}_oc-{num_occurrences}"
    ): ConfigArgs(
        model_id=model_id,
        alphabet_size=alphabet_size,
        string_length=string_length,
        tokens=[
            TokenConfig(
                prefix_length=i + 1,
                num_occurrences=num_occurrences,
                agreement_ratio=1.0,
            )
            for i in range(num_inserted_tokens)
        ],
    )
    for model_id in [
        "pyt-1b",
    ]
    for alphabet_size in [2, 4, 7, 13, 26]
    for string_length in [256, 512, 1024]
    for num_inserted_tokens in [4, 8, 16]
    for num_occurrences in [5, 10, 20]
}


CONFIG_ARGS = (
    {
        "test": ConfigArgs(
            model_id="pyt-70m",
            string_length=32,
            alphabet_size=7,
            tokens=[
                TokenConfig(
                    prefix_length=1,
                    num_occurrences=8,
                    agreement_ratio=1.0,
                ),
            ],
        ),
    }
    | FREQUENCY
    | PREFIX_LENGTH
)


def create_config(
    eval_type: str,
    seed_id: Optional[int] = None,
) -> ExperimentConfig:
    args = CONFIG_ARGS[eval_type]
    model_config = ModelConfig(
        model_id=args.model_name,
        base_dir=args.model_dir,
        pretrained=True,
    )
    config = ExperimentConfig(
        name=eval_type,
        seed_id=-1,
        seed=-1,
        data=RandomStringConfig(
            seed_id=-1,
            alphabet="latin",
            alphabet_size=args.alphabet_size,
            num_tokens=args.string_length,
            tokenizer_type=get_tokenizer_type(args.model_name),
        ),
        modification=ModificationConfig(
            seed=-1,
            tokens=args.tokens,
        ),
        model=model_config,
        training=get_memorization_training_config(
            seed_id=-1,
            model_id=model_config.model_id_not_none,
            num_epochs=args.num_epochs,
            wandb_project_name=f"llm_mem_{EXP_ABBREVIATION}",
        ),
    )
    if eval_type == "test":
        config.training.args.num_train_epochs = 1
    if seed_id is not None:
        set_seeds(config, seed_id)
    return config


def set_seeds(
    config: ExperimentConfig,
    seed_id: int,
) -> ExperimentConfig:
    seed = EXPERIMENT_SEEDS[seed_id]
    config.seed_id = seed_id
    config.seed
    config.data.seed_id = seed_id
    config.modification.seed = seed
    config.training.seed = seed
    return config


def get_configs() -> list[ExperimentConfig]:
    configs = []
    for eval_type in CONFIG_ARGS.keys():
        config = create_config(eval_type)
        configs.append(config)
    return configs


RTokHandle = ExperimentHandle(
    id=EXP_ABBREVIATION,
    create_configs=get_configs,
    set_seed=set_seeds,
    experiment=rt_experiment,
)
