from dataclasses import dataclass, replace
from typing import Optional

from defs import EXPERIMENT_SEEDS, BaseConfigArgs
from lib_llm.models import ModelConfig, get_tokenizer_type
from lib_project.experiment import ExperimentHandle
from utils.memorization import get_memorization_training_config

from .experiment import (
    EXP_ABBREVIATION,
    ExperimentConfig,
    RandomFactsConfig,
    rf_experiment,
)


@dataclass
class ConfigArgs(BaseConfigArgs):
    alphabet_size: int
    num_sentences: int
    num_test_samples: int
    num_epochs: int
    pretrained: bool = True


MODEL_IDS = [
    # "pyt-70m",
    "pyt-1b",
    "llama2-13b",
    "phi-2.7b",
]
ARGS = {
    (
        f"{model_id}"
        + ("_u" if not pretrained else "")
        + f"_a-{alphabet_size}_l-{num_sentences}"
        + f"_t-{num_test_samples}"
    ): ConfigArgs(
        model_id=model_id,
        alphabet_size=alphabet_size,
        num_sentences=num_sentences,
        num_test_samples=num_test_samples,
        num_epochs=200,
        pretrained=pretrained,
    )
    for model_id in MODEL_IDS
    for pretrained in [True, False]
    for alphabet_size in [2, 4, 8, 16, 64]
    for num_sentences in [16, 32, 64, 100, 128]
    for num_test_samples in [1, 8]
}


CONFIG_ARGS = {
    "test": ConfigArgs(
        model_id="pyt-70m",
        # model_id="pyt-1b",
        # model_id="llama2-7b",
        alphabet_size=4,
        num_sentences=8,
        num_test_samples=1,
        num_epochs=1,
    ),
} | ARGS


def create_config(
    eval_type: str,
    seed_id: Optional[int] = None,
) -> ExperimentConfig:
    args = CONFIG_ARGS[eval_type]

    model_config = ModelConfig(
        model_id=args.model_name,
        base_dir=args.model_dir,
        pretrained=args.pretrained,
    )
    training_config = get_memorization_training_config(
        seed_id=-1,
        model_id=model_config.model_id_not_none,
        num_epochs=args.num_epochs,
        wandb_project_name=f"llm_mem_{EXP_ABBREVIATION}",
        save_final_checkpoint=False,
        is_pretrained=args.pretrained,
        # use_wandb=False,
    )

    config = ExperimentConfig(
        name=eval_type,
        seed_id=-1,
        seed=-1,
        data=RandomFactsConfig(
            seed=-1,
            alphabet_size=args.alphabet_size,
            num_tokens=args.num_sentences,
            num_test_samples=args.num_test_samples,
            tokenizer_type=get_tokenizer_type(args.model_name),
        ),
        model=model_config,
        training=training_config,
    )
    if eval_type == "test":
        config.training.args.num_train_epochs = 1
    if seed_id is not None:
        set_seeds(config, seed_id)
    return config


def set_seeds(
    config: ExperimentConfig,
    seed_id: int,
) -> ExperimentConfig:
    config.seed_id = seed_id
    seed = EXPERIMENT_SEEDS[seed_id]

    config.data.seed = seed
    config.training.seed = seed
    return config


def get_configs() -> list[ExperimentConfig]:
    configs = []
    for eval_type in CONFIG_ARGS.keys():
        config = create_config(eval_type)
        configs.append(config)
    return configs


RFHandle = ExperimentHandle(
    id=EXP_ABBREVIATION,
    create_configs=get_configs,
    set_seed=set_seeds,
    experiment=rf_experiment,
)
