from dataclasses import dataclass, replace
from typing import Optional

from defs import EXPERIMENT_SEEDS, BaseConfigArgs
from lib_llm.models import ModelConfig, get_tokenizer_type
from lib_project.experiment import ExperimentHandle
from utils.memorization import get_memorization_training_config

from .experiment import (
    EXP_ABBREVIATION,
    ConditionalRandomStringConfig,
    ExperimentConfig,
    cpmd_experiment,
)


@dataclass
class ConfigArgs(BaseConfigArgs):
    alphabet_size: int
    num_tokens: int
    relative_probability: int
    ngram_length: int
    num_epochs: int = 100
    pretrained: bool = True


MODEL_IDS = [
    # "pyt-70m",
    "pyt-1b",
    "llama2-13b",
    "phi-2.7b",
]
ALPHABET_SIZES = [2, 7, 26]
RELATIVE_PROBABILITY_ARGS = {
    (
        f"{model_id}"
        + ("_u" if not pretrained else "")
        + f"_a-{alphabet_size}_t-{num_tokens}_"
        + f"rp-{relative_probability}_n-{ngram_length}"
    ): ConfigArgs(
        model_id=model_id,
        alphabet_size=alphabet_size,
        num_tokens=num_tokens,
        relative_probability=relative_probability,
        ngram_length=ngram_length,
    )
    for model_id in MODEL_IDS
    for pretrained in [True, False]
    for alphabet_size in ALPHABET_SIZES
    for num_tokens in [1024]
    for relative_probability in [2, 4, 8, 16, 32, 64]
    for ngram_length in [1]  # , 2, 3]
}
NGRAM_LENGTH_ARGS = {
    (
        f"{model_id}"
        + ("_u" if not pretrained else "")
        + f"_a-{alphabet_size}_t-{num_tokens}_"
        + f"rp-{relative_probability}_n-{ngram_length}"
    ): ConfigArgs(
        model_id=model_id,
        alphabet_size=alphabet_size,
        num_tokens=num_tokens,
        relative_probability=relative_probability,
        ngram_length=ngram_length,
    )
    for model_id in MODEL_IDS
    for pretrained in [True]  # , False]
    for alphabet_size in ALPHABET_SIZES
    for num_tokens in [1024]
    for relative_probability in [16]
    for ngram_length in [2, 3, 4]
}


CONFIG_ARGS = (
    {
        "test": ConfigArgs(
            model_id="pyt-70m",
            # model_id="pyt-1b",
            # model_id="llama2-7b",
            alphabet_size=26,
            num_tokens=256,
            num_epochs=1,
            relative_probability=2,
            ngram_length=2,
        ),
    }
    | RELATIVE_PROBABILITY_ARGS
    | NGRAM_LENGTH_ARGS
)


def create_config(
    eval_type: str,
    seed_id: Optional[int] = None,
) -> ExperimentConfig:
    args = CONFIG_ARGS[eval_type]

    model_config = ModelConfig(
        model_id=args.model_name,
        base_dir=args.model_dir,
        pretrained=args.pretrained,
    )
    training_config = get_memorization_training_config(
        seed_id=-1,
        model_id=model_config.model_id_not_none,
        num_epochs=args.num_epochs,
        batch_size=1,
        wandb_project_name=f"llm_mem_{EXP_ABBREVIATION}",
        save_final_checkpoint=False,
        is_pretrained=args.pretrained,
    )
    training_config.args = replace(
        training_config.args,
        evaluation_strategy="steps",
        eval_steps=1,
        # Turn off WandB logging, since it's causing connection issues
        # in some runs
        report_to=["none"],
    )
    data_config = ConditionalRandomStringConfig(
        seed=-1,
        alphabet="latin",
        alphabet_size=args.alphabet_size,
        num_tokens=args.num_tokens,
        num_partitions=1,
        ngram_length=args.ngram_length,
        tokenizer_type=get_tokenizer_type(args.model_name),
    )
    data_config.set_relative_probability(args.relative_probability)

    config = ExperimentConfig(
        name=eval_type,
        seed_id=-1,
        seed=-1,
        data=data_config,
        model=model_config,
        training=training_config,
    )
    if eval_type == "test":
        config.training.args.num_train_epochs = 1
    if seed_id is not None:
        set_seeds(config, seed_id)
    return config


def set_seeds(
    config: ExperimentConfig,
    seed_id: int,
) -> ExperimentConfig:
    config.seed_id = seed_id
    seed = EXPERIMENT_SEEDS[seed_id]
    config.data.seed = seed
    config.training.seed = seed
    return config


def get_configs() -> list[ExperimentConfig]:
    configs = []
    for eval_type in CONFIG_ARGS.keys():
        config = create_config(eval_type)
        configs.append(config)
    return configs


CPMDHandle = ExperimentHandle(
    id=EXP_ABBREVIATION,
    create_configs=get_configs,
    set_seed=set_seeds,
    experiment=cpmd_experiment,
)
