from dataclasses import dataclass
from typing import Optional

from defs import BaseConfigArgs
from lib_llm.models import ModelConfig, get_tokenizer_type
from lib_project.experiment import ExperimentHandle

from .experiment import (
    EXP_ABBREVIATION,
    ExperimentConfig,
    RandomStringConfig,
    ice_experiment,
)


SEEDS = [
    5932,
    4152,
    4967,
    2938,
    84163,
    2663,
    27,
    8206,
    1625,
    6232,
]


@dataclass
class ConfigArgs(BaseConfigArgs):
    sequence_length: int
    alphabet_size: int = 26


SEQUENCE_LENGTH_ALPHABET_SIZE_ARGS = {
    f"{model_id}_sl-{seq_length}_al-{alphabet_size}": ConfigArgs(
        model_id=model_id,
        sequence_length=seq_length,
        alphabet_size=alphabet_size,
    )
    for model_id in [
        "pyt-1b",
    ]
    for seq_length in [16, 32, 64, 128, 256, 512, 1024]
    for alphabet_size in [2, 4, 7, 13, 26]
}


CONFIG_ARGS = {
    "test": ConfigArgs(
        model_id="pyt-1b",
        sequence_length=64,
    ),
} | (SEQUENCE_LENGTH_ALPHABET_SIZE_ARGS)


def create_config(
    eval_type: str,
    seed_id: Optional[int] = None,
) -> ExperimentConfig:
    args = CONFIG_ARGS[eval_type]

    config = ExperimentConfig(
        name=eval_type,
        seed_id=-1,
        seed=-1,
        data=RandomStringConfig(
            seed_id=-1,
            num_tokens=args.sequence_length,
            alphabet_size=args.alphabet_size,
            tokenizer_type=get_tokenizer_type(args.model_name),
        ),
        # model=ModelConfig(
        #     model_id=args.model_name,
        #     base_dir=args.model_dir,
        #     pretrained=args.pretrained,
        # ),
    )
    if eval_type == "test":
        pass
    if seed_id is not None:
        set_seeds(config, seed_id)
    return config


def set_seeds(
    config: ExperimentConfig,
    seed_id: int,
) -> ExperimentConfig:
    seed = SEEDS[seed_id]
    config.seed_id = seed_id
    config.seed = seed
    config.data.seed_id = seed_id
    return config


def get_configs() -> list[ExperimentConfig]:
    configs = []
    for eval_type in CONFIG_ARGS.keys():
        config = create_config(eval_type)
        configs.append(config)
    return configs


ICEHandle = ExperimentHandle(
    id=EXP_ABBREVIATION,
    create_configs=get_configs,
    set_seed=set_seeds,
    experiment=ice_experiment,
)
